/*
  Sample program for the HawkNL cross platform network library
  Copyright (C) 2000-2001 Phil Frisbie, Jr. (phil@hawksoft.com)
*/
/*
  This app shows a multithreaded client/server app.

  Choose the network type to use with the command line:
  
  clientserver NL_IP

  The default is NL_IP. Valid network types are: NL_IP, NL_IPX,
  and NL_LOOP_BACK 
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>
#include "../../include/nl.h"

#if defined WIN32 || defined WIN64
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#define sleep(x)    Sleep(1000 * (x))
#endif

pthread_mutex_t printmutex;

void printErrorExit(void)
{
    NLenum err = nlGetError();
    
    if(err == NL_SOCKET_ERROR)
    {
        printf("System error: %s\n", nlGetSystemErrorStr(nlGetSystemError()));
    }
    else
    {
        printf("HawkNL error: %s\n", nlGetErrorStr(err));
    }
    nlShutdown();
    exit(1);
}

#define MAX_CLIENTS 10
NLenum socktype = NL_RELIABLE_PACKETS;
NLushort serverport = 25000;


static void *mainServerLoop(void *s)
{
    NLsocket    sock = *(NLsocket *)s;
    NLsocket    client[MAX_CLIENTS];
    NLint       clientnum = 0;
    NLbyte      string[NL_MAX_STRING_LENGTH];
    NLint       group;
    
    memset(client, 0, sizeof(client));
    
    group = nlGroupCreate();
    while(1)
    {
        NLint       i, count;
        NLbyte      buffer[128];
        NLsocket    s[MAX_CLIENTS];
        
        /* check for a new client */
        NLsocket newsock = nlAcceptConnection(sock);
        
        if(newsock != NL_INVALID)
        {
            NLaddress   addr;
            
            nlGetRemoteAddr(newsock, &addr);
            client[clientnum] = newsock;
            /* add socket to the group */
            nlGroupAddSocket(group, newsock);
			(void)pthread_mutex_lock(&printmutex);
            printf("SERVER:Client %d connected from %s on socket %d\n", clientnum,
                        nlAddrToString(&addr, string), newsock);
			(void)pthread_mutex_unlock(&printmutex);
            clientnum++;
        }
        else
        {
            if(nlGetError() == NL_SOCKET_ERROR)
            {
                printf("\n\nServer shutdown!!!!!!!!!!!!!\n");
                printErrorExit();
            }
        }
        /* check for incoming messages */
        count = nlPollGroup(group, NL_READ_STATUS, s, MAX_CLIENTS, 0);
        if(count == NL_INVALID)
        {
            NLenum err = nlGetError();
            
            if(err == NL_SOCKET_ERROR)
            {
                printf("nlPollGroup system error: %s\n", nlGetSystemErrorStr(nlGetSystemError()));
            }
            else
            {
                printf("nlPollGroup HawkNL error: %s\n", nlGetErrorStr(err));
            }
        }
        if(count > 0)
        {
			(void)pthread_mutex_lock(&printmutex);
            printf("\n\n!!!!!!!!!count = %d\n", count);
			(void)pthread_mutex_unlock(&printmutex);
        }
        /* loop through the clients and read the packets */
        for(i=0;i<count;i++)
        {
            int readlen;

			(void)pthread_mutex_lock(&printmutex);
            while((readlen = nlRead(s[i], buffer, sizeof(buffer))) > 0)
            {
                buffer[127] = 0; /* null terminate the char string */
                printf("SERVER:socket %d sent %s\n", s[i], buffer);
                /* send to the whole group */
                nlWrite(group, buffer, strlen(buffer) + 1);
            }
			(void)pthread_mutex_unlock(&printmutex);
            if(readlen == NL_INVALID)
            {
                NLenum err = nlGetError();

                if( err == NL_MESSAGE_END || err == NL_SOCK_DISCONNECT)
                {
                    nlGroupDeleteSocket(group, s[i]);
                    nlClose(s[i]);
					(void)pthread_mutex_lock(&printmutex);
                    printf("SERVER:socket %d closed\n", s[i]);
					(void)pthread_mutex_unlock(&printmutex);
                    clientnum--;
                }
            }
        }
        sleep(0);
    }
    return NULL;
}

static void mainClientLoop()
{
    NLsocket    sock[MAX_CLIENTS];
    NLaddress   addr;
    NLint       i, count = 10;
	NLbyte		str[256];

    /* create the client sockets */
    for(i=0;i<MAX_CLIENTS;i++)
    {
        sock[i] = nlOpen(0, socktype);
        if(sock[i] == NL_INVALID)
            printErrorExit();
        /* now connect */
        nlGetLocalAddr(sock[i], &addr);
        nlSetAddrPort(&addr, serverport);
        if(!nlConnect(sock[i], &addr))
        {
            printErrorExit();
        }
        printf("CLIENT %d connect to %s\n", i, nlAddrToString(&addr, str));
    }
    while(count-- > 0)
    {
        NLbyte  string[NL_MAX_STRING_LENGTH];

        for(i=0;i<MAX_CLIENTS;i++)
        {
			(void)pthread_mutex_lock(&printmutex);
            sprintf(string, "Client %d says hello, hello", i);
            nlWrite(sock[i], string, strlen(string) + 1);
            sprintf(string, "... client %d out.", i);
            nlWrite(sock[i], string, strlen(string) + 1);
            printf("\n\nCLIENT %d received: ",i);
            while(nlRead(sock[i], string, sizeof(string)) > 0)
            {
                printf("\"%s\",", string);
            }
			(void)pthread_mutex_unlock(&printmutex);
        }
        sleep(1);
    }
    /* create the client sockets */
	(void)pthread_mutex_lock(&printmutex);
    for(i=0;i<MAX_CLIENTS;i++)
    {
        nlClose(sock[i]);
        printf("CLIENT %d CLOSED\n", i);
    }
	(void)pthread_mutex_unlock(&printmutex);
    sleep(4);
}

int main(int argc, char **argv)
{
    NLsocket        serversock;
    pthread_attr_t  attr;
    pthread_t       tid;
    NLenum          type = NL_IP;/* default network type */

    if(!nlInit())
        printErrorExit();
    
    printf("nlGetString(NL_VERSION) = %s\n\n", nlGetString(NL_VERSION));
    printf("nlGetString(NL_NETWORK_TYPES) = %s\n\n", nlGetString(NL_NETWORK_TYPES));

    if (argc == 2)
    {
        if(strcmp(argv[1], "NL_IPX") == 0)
        {
            type = NL_IPX;
        }
        else if(strcmp(argv[1], "NL_LOOP_BACK") == 0)
        {
            type = NL_LOOP_BACK;
        }
    }

    if(!nlSelectNetwork(type))
        printErrorExit();

	(void)pthread_mutex_init(&printmutex, NULL);
    /* create the server socket */
    serversock = nlOpen(serverport, socktype); /* just a random port number ;) */
    
    if(serversock == NL_INVALID)
        printErrorExit();
    
    if(!nlListen(serversock))       /* let's listen on this socket */
    {
        nlClose(serversock);
        printErrorExit();
    }
    /* start the server thread */
    (void)pthread_attr_init(&attr);
    (void)pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
    (void)pthread_create(&tid, &attr, mainServerLoop, (void *)&serversock);
    /*now enter the client loop */
    mainClientLoop();
    nlShutdown();
	(void)pthread_mutex_destroy(&printmutex);
    return 0;
}

