#include "RenderEngine.h"
#include "mapent.h"
#include "transform.h"

#include "Player.h"
#include "PlayerList.h"
#include "BasicGraphicList.h"
#include "ModelTypeList.h"
#include "XPModelList.h"
#include "XPSubPart.h"
#include "GraphicModel.h"
#include "BSPDataModel.h"
#include "ppmPictureList.h"
#include "SoundObjectList.h"

double RenderEngine::getFrameTime() { return g_frametime; }
void RenderEngine::setFrameTime(double time) { g_frametime = time; }
extern XPModelList *projectileList;
extern bool matrixmode;

RenderEngine::RenderEngine()
{
	g_frametime = 0;
	fpstime = 0;
	fpsframes = 0;
	r_eyefov = 90.0;
	enableCollDetect = true;
	lightMode = NORMAL;
	fogMode = false;

	bspDataModel = new BSPDataModel;
	bspDataModel->load(WORLDPATH, WORLDFILE);

	initParticleSystem();
	initEmitter(&emitter);
}

RenderEngine::~RenderEngine()
{
	glDisable(GL_LIGHTING);
	glDisable(GL_FOG);
	glClearColor(0.0, 0.0, 0.0, 1.0);

    free(facelist.faces);
    free(translist.faces);
    free(r_faceinc);
    free(skylist);
    render_backend_finalize();

	delete bspDataModel;
}

void RenderEngine::setBSP(BSPDataModel *bspDM)
{
	bspDataModel = bspDM;
}

void RenderEngine::render_init(void)
{
	glFinish();
    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_TEXTURE_COORD_ARRAY);
    
    facelist.faces = (rendface_t*)malloc(bspDataModel->r_numfaces * sizeof(rendface_t));
    translist.faces = (rendface_t*)malloc(MAX_TRANSPARENT *
					  sizeof(rendface_t));
    r_faceinc = (int*)malloc(bspDataModel->r_numfaces * sizeof(int));
    skylist = (int*)malloc(100 * sizeof(int));
    render_backend_init();

	emitter.pos.x = 2240;
	emitter.pos.y = 200;
	emitter.pos.z = 200;

	//find_start_pos();

    glDisable(GL_DITHER);
    glShadeModel(GL_SMOOTH);
    glClearColor(0.0, 0.0, 0.0, 0.0);
    glColor3f(1.0, 1.0, 1.0);

	GLfloat light_pos[] = {0.0, 1.0, 0.0, 0.0};
/*
		vec3 eyeDir=viewer->getEyeOrientation();
		light_pos[0] = eyeDir[0];
		light_pos[1] = eyeDir[1];
		light_pos[2] = eyeDir[2];
		light_pos[3] = 0.0;
*/
	glLightfv(GL_LIGHT0, GL_POSITION, light_pos);

	glFogi(GL_FOG_MODE, GL_EXP);
	glFogf(GL_FOG_START, 1.0);
	glFogf(GL_FOG_END, 5.0);

#if 0
	glClearStencil(0x0);
	glEnable(GL_STENCIL_TEST);
	glClear(GL_STENCIL_BUFFER_BIT);
	glStencilFunc(GL_ALWAYS, 0x1, 0x1);
	glStencilOp(GL_REPLACE, GL_REPLACE, GL_REPLACE);
	glBegin(GL_QUADS);
		glVertex2f(-10.0, 0.0);
		glVertex2f(0.0, 10.0);
		glVertex2f(10.0, 0.0);
		glVertex2f(0.0, -10.0);
	glEnd();
#endif

    glEnable(GL_DEPTH_TEST);
    glCullFace(GL_FRONT);
    glEnable(GL_TEXTURE_2D);
    glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
	

    glViewport(0, 0, VIEWPORT_W, VIEWPORT_H);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective(r_eyefov, (float) VIEWPORT_W/(float) VIEWPORT_H, 10.0, 3000.0);
    glMatrixMode(GL_MODELVIEW);

}

void RenderEngine::setFog(bool state, GLfloat *fogcolorv, GLfloat density)
{
  return;
	fogMode = state;
	if (!state)
	{
		glDisable(GL_FOG);
		glClearColor(0.0, 0.0, 0.0, 1.0);
		//glClearColor(0.0, 0.0, 0.0, 0.5);
	}
	else
	{
		glEnable(GL_FOG);
		glFogfv(GL_FOG_COLOR, fogcolorv);
		glFogf(GL_FOG_DENSITY, 0.005);
		glClearColor(fogcolorv[0], fogcolorv[1], fogcolorv[2], 1.0);
	}
}

void RenderEngine::setLight(LightMode newMode)
{
return;
	lightMode = newMode;
	glDisable(GL_LIGHTING);

	switch(lightMode)
	{
	case IR_GLASSES:
	{
		GLfloat light_pos[4]; // = {0.0, 1.0, 0.0, 0.0};
		vec3 eyeDir=viewer->getEyeOrientation();
		light_pos[0] = eyeDir[0];
		light_pos[1] = eyeDir[1];
		light_pos[2] = eyeDir[2];
		light_pos[3] = 0.0;
		//GLfloat light_pos[] = {0.0, 1.0, 0.0, 0.0};
		//glLightfv(GL_LIGHT0, GL_POSITION, light_pos);
		//printf("%f %f %f\n", light_pos[0], light_pos[1], light_pos[2]);
		GLfloat red_light[] = {1.0, 0.0, 0.0, 1.0};
		GLfloat black_light[] = {0.0, 0.0, 0.0, 1.0};
		//GLfloat mat_specular[] = { 1.0, 1.0, 1.0, 1.0};
		//GLfloat mat_shininess[] = { 50.0 };
		//glMaterialfv(GL_FRONT, GL_SPECULAR, mat_specular);
		//glMaterialfv(GL_FRONT, GL_SHININESS, mat_shininess);
		glLightfv(GL_LIGHT0, GL_AMBIENT, red_light);
		glLightfv(GL_LIGHT0, GL_DIFFUSE, red_light);
		glLightfv(GL_LIGHT0, GL_SPECULAR, red_light);
		//glLightf(GL_LIGHT0, GL_SPOT_CUTOFF, 20.0);
		//glLightfv(GL_LIGHT0, GL_SPOT_DIRECTION, r_eyedir);
		//glLightf(GL_LIGHT0, GL_SPOT_EXPONENT, 2.0);
		glEnable(GL_LIGHTING);
		glEnable(GL_LIGHT0);
		break;
	}
	case NOLIGHTS:
	{
		GLfloat light_pos[4]; // = {0.0, 1.0, 0.0, 0.0};
		vec3 eyeDir=viewer->getEyeOrientation();
		light_pos[0] = eyeDir[0];
		light_pos[1] = eyeDir[1];
		light_pos[2] = eyeDir[2];
		light_pos[3] = 0.0;
		GLfloat gray_light[] = {0.3, 0.3, 0.3, 1.0};
		glLightfv(GL_LIGHT0, GL_AMBIENT, gray_light);
		glLightfv(GL_LIGHT0, GL_DIFFUSE, gray_light);
		glLightfv(GL_LIGHT0, GL_SPECULAR, gray_light);
		glEnable(GL_LIGHTING);
		glEnable(GL_LIGHT0);
		break;
	}
	case NORMAL:
		break;
	}

}
void RenderEngine::render_scene(void)
{
    int i;

    r_leafcount = 0;
	vec3 pos=viewer->getPosition();

    /* Find eye cluster for PVS checks */
    if (!r_lockpvs){
		
		vec3_t posit;
		posit[0]=pos[0];posit[1]=pos[1];posit[2]=pos[2];
		r_eyecluster = find_cluster(posit);
	}

    /* Need to enable depth mask before clear */
    glDepthMask(GL_TRUE);

    /* FIXME: color buffer clear can be optionally removed */
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();

    /* Set up camera */
	vec3 eyedir=viewer->getEyeOrientation();
	//eyedir[0]=0;eyedir[1]=0;eyedir[2]=1;
	vec3 look=pos+eyedir;

    gluLookAt(pos[0], pos[1], pos[2],
	      look[0], look[1],
	      look[2], 0.0, 0.0, 1.0);


	//xpModelList->updatePosition(0.2);


#if 0
    /* Useful for exploring face culling effectiveness */
    if (r_lockpvs)
    {
	render_backend(&facelist);
	return;
    }
#endif    

    cos_fov = cos(r_eyefov/2.0f * DEG2RAD);

    /* Get clip coordinate transformation matrix */
    gen_clipmat();

    facelist.numfaces = translist.numfaces = 0;
    numsky = 0;
    /* Clear "included" faces list */
    memset(r_faceinc, 0, bspDataModel->r_numfaces * sizeof(int));

    /* "Walk" the BSP tree to generate a list of faces to render */
    /* FIXME: include other models */
    render_walk_model(0);

    /* Sort the face list */
    sort_faces();

    /* FIXME: Reset depth buffer range based on max/min Z in facelist */

    /* Draw sky first */
    if (numsky)
	render_backend_sky(numsky, skylist);

    /* Draw normal faces */
    render_backend(&facelist);

    /* Draw visible mapents (md3 models) */
	
	for(i=0;i<xpModelList->getLength();i++){
		XPModel *model=xpModelList->getModel(i);
		if(model->isActive()){
			renderModel(model);
		}
		

	}
	
	for(i=0;i<projectileList->getLength();i++){
		XPModel *model=projectileList->getModel(i);
		renderModel(model);
		

	}

	for(i=0;i<playerList->getLength();i++){
		Player *player=playerList->getPlayer(i);
		if((player!=NULL)&&(player!=viewer)){
			renderModel(player);
		}
		

	}


	
	updateEmitter(&emitter);
	renderEmitter(&emitter, false);

	glPushAttrib (GL_ALL_ATTRIB_BITS);

    /* Draw transparent faces last */	
    if (translist.numfaces && !fogMode)
		render_transparent(&translist);


#if 0
    /* Enable for speeds reporting (like r_speeds 1) */
    printf("faces: %d, leafs: %d\n", facelist.numfaces + translist.numfaces,
	   r_leafcount);
#endif

	// add HUD	

	glPopAttrib();

	
	glMatrixMode (GL_PROJECTION);
	glLoadIdentity();
	gluOrtho2D (0,(GLfloat)VIEWPORT_W,(GLfloat)VIEWPORT_H,0);	
	glMatrixMode (GL_MODELVIEW);
	glLoadIdentity();
	glEnable (GL_BLEND);

	glBlendFunc (GL_SRC_ALPHA, GL_ONE);

	
	int activeWeapon = viewer->activeWeapon;
	
	//weapons		
	if (viewer->weapons[0].available) pictureList->getPicture(11)->display(0,600);
	if (viewer->weapons[1].available) pictureList->getPicture(10)->display(70,600);
	if (viewer->weapons[2].available) pictureList->getPicture(22)->display(140,600);
	if (viewer->haveGlasses) pictureList->getPicture(21)->display(210, 600);
	if (viewer->haveMatrix)  pictureList->getPicture(23)->display(280, 600);
	if (viewer->haveGlasses) pictureList->getPicture(21)->display(210, 600);
	if (viewer->haveMatrix)  pictureList->getPicture(23)->display(280, 600);

	//cross hairs
	pictureList->getPicture(24)->display(390,310);

	for (i=0;i<2;i++){
		if (activeWeapon == 0) pictureList->getPicture(11)->display(0,600);
		if (activeWeapon == 1) pictureList->getPicture(10)->display(70,600);
		if (activeWeapon == 2) pictureList->getPicture(22)->display(140,600);
	}


	int health = viewer->getHealth();	
	int clipammo = viewer->weapons[activeWeapon].clipCount;
	int ammo = viewer->weapons[activeWeapon].currentAmmo;

	for (i=0;i<3;i++){
		//ammo
		pictureList->getPicture(19)->display(400,600);
		pictureList->getPicture((clipammo/10)%10)->display(440,600);
		pictureList->getPicture(clipammo%10)->display(465,600);
		pictureList->getPicture((ammo/100)%10)->display(520,600);
		pictureList->getPicture((ammo/10)%10)->display(545,600);
		pictureList->getPicture(ammo%10)->display(570,600);

		//health
		pictureList->getPicture(20)->display(650,600);
		if (health == 100) pictureList->getPicture(1)->display(700,600);
		pictureList->getPicture((health/10)%10)->display(725,600);
		pictureList->getPicture((health)%10)->display(750,600);

	}
	

	glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective(r_eyefov, (float) VIEWPORT_W/(float) VIEWPORT_H, 10.0, 3000.0);
    glMatrixMode(GL_MODELVIEW);

    
	glFlush();

}

void RenderEngine::render_walk_model(int n)
{
    if (n == 0)
	render_walk_node(0, 0);
    else
	/* FIXME: models > 0 are just leafs ? */
	Error("Models > 0 not supported");
}

int RenderEngine::classify_point(vec3_t p, int plane_n)
{
    /* Determine which side of plane p is on */
    plane_t *plane = &(bspDataModel->r_planes[plane_n]);

    return (vec_dot(p, plane->vec) < plane->offset ? -1 : 1);
}

void RenderEngine::render_walk_node(int n, int accept)
{
    node_t *node = &(bspDataModel->r_nodes[n]);
	
    if (!accept)
    {
	/* Test the node bounding box for intersection with the view volume */
	int clipstate = cliptest_bbox(node->bbox);
	/* If this node is rejected, reject all sub-nodes */
	if (!clipstate) return;
	/* If this node is trivially accepted, accept all sub-nodes */
	if (clipstate == 2) accept = 1;
    }

    /* Classify eye wrt splitting plane */
 	vec3 pos=viewer->getPosition();
 	vec3_t posit;
 	posit[0]=pos[0];posit[1]=pos[1];posit[2]=pos[2];
 
    if (classify_point(posit, node->plane) > 0)
    {
	/* In front of plane: render front first, then back */
	if (node->children[0] < 0)
	    render_walk_leaf(-(node->children[0] + 1), accept);
	else
	    render_walk_node(node->children[0], accept);
	if (node->children[1] < 0)
	    render_walk_leaf(-(node->children[1] + 1), accept);
	else
	    render_walk_node(node->children[1], accept);
    }
    else
    {
	/* Behind plane: render back first, then front */
	if (node->children[1] < 0)
	    render_walk_leaf(-(node->children[1] + 1), accept);
	else
	    render_walk_node(node->children[1], accept);
	if (node->children[0] < 0)
	    render_walk_leaf(-(node->children[0] + 1), accept);
	else
	    render_walk_node(node->children[0], accept);	
    }
}

void RenderEngine::render_walk_leaf(int n, int accept)
{
    leaf_t *leaf = &(bspDataModel->r_leafs[n]);
    int i;

    /* Test visibility before bounding box */
    if (r_eyecluster >= 0)
    {
	if (! RENDER_BSP_TESTVIS(r_eyecluster, leaf->cluster)) return;
    }
    
    if (!accept)
    {
	if (!cliptest_bbox(leaf->bbox)) return;
    }    

    r_leafcount++;
    
    for (i=0; i < leaf->numfaces; ++i)
    {
	render_walk_face(bspDataModel->r_lfaces[i + leaf->firstface]);
    }
}

void RenderEngine::render_walk_face(int n)
{
    face_t *face = &(bspDataModel->r_faces[n]);

    /* Check if face is already included in the facelist */
    if (r_faceinc[n]) return;
    r_faceinc[n] = 1;

    if (face->facetype == FACETYPE_NORMAL)
    {
	/* Face plane culling */
	/* FIXME: This simple test is clearly not sufficient.
	   Q3A gets a lot more culling at this point. */
		vec3 eyedir=viewer->getEyeOrientation();
		if (vec_dot(face->v_norm, eyedir) > cos_fov)
			return;
    }
    
    else if (face->facetype == FACETYPE_MESH)
    {
	/* Check bounding box for meshes */
	if (!cliptest_bboxf(face->bbox))
	    return;
    }

    /* Check for sky flag */
    if (bspDataModel->r_shaders[face->shader].flags & SHADER_SKY)
    {
	/* Push to sky list */
	skylist[numsky++] = n;
    }

    /* Check for transparent */
    else if (bspDataModel->r_shaders[face->shader].flags & SHADER_TRANSPARENT)
    {
	translist.faces[translist.numfaces].face = n;
	translist.faces[translist.numfaces++].sortkey = SORTKEY(face);
    }

    /* Normal face */
    else
    {
	/* Push face to facelist */
	facelist.faces[facelist.numfaces].face = n;
	facelist.faces[facelist.numfaces++].sortkey = SORTKEY(face);
    }
}

void RenderEngine::gen_clipmat(void)
{
    mat4_t modelview[32];
    mat4_t proj[2];

    /* Get the modelview and projection matricies from GL */
    glGetFloatv(GL_MODELVIEW_MATRIX, (GLfloat*)modelview);
    glGetFloatv(GL_PROJECTION_MATRIX, (GLfloat*)proj);

    /* Multiply to get clip coordinate transformation */
    mat4_mmult(proj[0], modelview[0], clipmat);
}

int RenderEngine::cliptest_point(vec4_t p)
{
    int mask = 0;
    const float cx = p[0];
    const float cy = p[1];
    const float cz = p[2];
    const float cw = p[3];      
    
    if (cx >  cw) mask |= CLIP_RIGHT_BIT;
    if (cx < -cw) mask |= CLIP_LEFT_BIT;
    if (cy >  cw) mask |= CLIP_TOP_BIT;
    if (cy < -cw) mask |= CLIP_BOTTOM_BIT;
    if (cz >  cw) mask |= CLIP_FAR_BIT;
    if (cz < -cw) mask |= CLIP_NEAR_BIT;

    return mask;
}

/* Test bounding box for intersection with view fustrum.
 * Return val:   0 = reject
 *               1 = accept
 *               2 = trivially accept (entirely in fustrum)
 */
int RenderEngine::cliptest_bboxf(bboxf_t bv)
{
    static int corner_index[8][3] =
    {
	{0, 1, 2}, {3, 1, 2}, {3, 4, 2}, {0, 4, 2},
	{0, 1, 5}, {3, 1, 5}, {3, 4, 5}, {0, 4, 5}
    };

    vec4_t corner[8];
    int clipcode, clip_or, clip_and, clip_in;
    int i;

    /* Check if eye point is contained */
 	vec3 pos=viewer->getPosition();
    if (pos[0] >= bv[0] && pos[0] <= bv[3] && pos[1] >= bv[1] && pos[1] <= bv[4] &&	pos[2] >= bv[2] && pos[2] <= bv[5])
		return 1;
    
    clip_in = clip_or = 0; clip_and = 0xff;
    for (i=0; i < 8; ++i)
    {
	corner[i][0] = bv[corner_index[i][0]];
	corner[i][1] = bv[corner_index[i][1]];
	corner[i][2] = bv[corner_index[i][2]];
	corner[i][3] = 1.0;

	mat4_vmult(clipmat, corner[i], corner[i]);
	clipcode = cliptest_point(corner[i]);
	clip_or |= clipcode;
	clip_and &= clipcode;
	if (!clipcode) clip_in = 1;
    }

    /* Check for trival acceptance/rejection */
    if (clip_and) return 0;
    if (!clip_or) return 2;
    if (clip_in) return 1;   /* At least one corner in view fustrum */

#if 0
    /* FIXME: need something better for this. */
    /* Maybe find maximum radius to each corner */
    {
	/* Normalize coordinates */
	vec3_t center, rad;
	float cw;

	cw = 1.0f/corner[0][3];
	vec_scale(corner[0], cw, corner[0]);
	corner[0][3] = 1.0;
	cw = 1.0f/corner[6][3];
	vec_scale(corner[6], cw, corner[6]);
	corner[6][3] = 1.0;

	/* Check for non-trivial acceptance */
	vec_avg(corner[0], corner[6], center);
	vec_sub(corner[0], center, rad);
	if (sqrt(vec_dot(center, center)) -
	    sqrt(vec_dot(rad, rad)) <= 1.41421356)
	    return 1;
    }
	
    return 0;
#endif
    return 1;
}

int RenderEngine::cliptest_bbox(bbox_t bbox)
{
    bboxf_t bv;

    bv[0] = (float)bbox[0];
    bv[1] = (float)bbox[1];
    bv[2] = (float)bbox[2];
    bv[3] = (float)bbox[3];
    bv[4] = (float)bbox[4];
    bv[5] = (float)bbox[5];

    return cliptest_bboxf(bv);
}

int RenderEngine::find_cluster(vec3_t pos)
{
    node_t *node;
    int cluster = -1;
    int leaf = -1;
    
    node = &(bspDataModel->r_nodes[0]);

    /* Find the leaf/cluster containing the given position */
    
    while (1)
    {
	if (classify_point(pos, node->plane) > 0)
	{
	    if (node->children[0] < 0)
	    {
		leaf = -(node->children[0] + 1);
		break;
	    }
	    else
	    {
		node = &(bspDataModel->r_nodes[node->children[0]]);
	    }
	}
	else
	{
	    if (node->children[1] < 0)
	    {
		leaf = -(node->children[1] + 1);
		break;
	    }
	    else
	    {
		node = &(bspDataModel->r_nodes[node->children[1]]);
	    }
	}	    
    }

    if (leaf >= 0)
	cluster = bspDataModel->r_leafs[leaf].cluster;
    return cluster;
}

int face_cmp(const void *a, const void *b)
{
    return ((rendface_t*)a)->sortkey - ((rendface_t*)b)->sortkey;
}

void RenderEngine::sort_faces(void)
{
    /* FIXME: expand qsort code here to avoid function calls */
    qsort((void*)facelist.faces, facelist.numfaces, sizeof(rendface_t),
	  face_cmp);
}


/* start rendering backend */
void RenderEngine::render_backend_init(void)
{
    /* FIXME: need to account for extra verts/elems in meshes */
    
    arrays.verts = (vec4_t*)malloc(bspDataModel->r_numverts * sizeof(vec4_t));
    arrays.tex_st = (texcoord_t*)malloc(bspDataModel->r_numverts * sizeof(texcoord_t));
    arrays.lm_st = (texcoord_t*)malloc(bspDataModel->r_numverts * sizeof(texcoord_t));
    arrays.elems = (int*)malloc(bspDataModel->r_numelems * sizeof(int));
    arrays.colour = (colour_t*)malloc(bspDataModel->r_numverts * sizeof(colour_t));
}

void RenderEngine::render_backend_finalize(void)
{
    free(arrays.verts);
    free(arrays.tex_st);
    free(arrays.lm_st);
    free(arrays.elems);
    free(arrays.colour);
}

void RenderEngine::render_backend(facelist_t *facelist)
{
    int f, shader, lmtex;
    uint_t key;
    face_t *face;

    
    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_TEXTURE_COORD_ARRAY);

    arrays.numverts = arrays.numelems = 0;
    key = (uint_t)-1;
    for (f=0; f < facelist->numfaces; ++f)
    {
	face = &(bspDataModel->r_faces[facelist->faces[f].face]);

	/* Look for faces that share rendering state */
	if (facelist->faces[f].sortkey != key)
	{
	    /* Flush the renderer and reset */
	    if (f) render_flush(shader, lmtex);
	    shader = face->shader;
	    lmtex = face->lm_texnum;
	    key = facelist->faces[f].sortkey;
	}

	/* Push the face to the triangle arrays */
	switch (face->facetype)
	{
	    case FACETYPE_NORMAL:
	    case FACETYPE_TRISURF:
		if (bspDataModel->r_shaders[shader].flags & SHADER_DEFORMVERTS)
		    render_pushface_deformed(shader, face);
		else
		    render_pushface(face);
		break;
	    case FACETYPE_MESH:
		render_pushmesh(&(bspDataModel->r_meshes[facelist->faces[f].face]));
		break;
	    default:
		break;
	}
    }
    /* Final flush to clear queue */
    render_flush(shader, lmtex);    
}


void RenderEngine::render_transparent(facelist_t *facelist)
{
    int f, shader, lmtex;
    uint_t key;
    face_t *face;
    
    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_TEXTURE_COORD_ARRAY);

    arrays.numverts = arrays.numelems = 0;
    key = (uint_t)-1;
    for (f=0; f < facelist->numfaces; ++f)
    {
	face = &(bspDataModel->r_faces[facelist->faces[f].face]);

	/* Look for faces that share rendering state */
	if (facelist->faces[f].sortkey != key)
	{
	    /* Flush the renderer and reset */
	    if (f) render_flush(shader, lmtex);
	    shader = face->shader;
	    lmtex = face->lm_texnum;
	    key = facelist->faces[f].sortkey;
	}

	/* Push the face to the triangle arrays */
	switch (face->facetype)
	{
	    case FACETYPE_NORMAL:
	    case FACETYPE_TRISURF:
		if (bspDataModel->r_shaders[shader].flags & SHADER_DEFORMVERTS)
		    render_pushface_deformed(shader, face);
		else
		    render_pushface(face);
		break;
	    case FACETYPE_MESH:
			render_pushmesh(&(bspDataModel->r_meshes[facelist->faces[f].face]));
			break;
	    default:
		break;
	}
    }
    /* Final flush to clear queue */
    render_flush(shader, lmtex);    
}


void RenderEngine::render_backend_sky(int numsky, int *skylist)
{
    int s, i, shader;
    float skyheight;
    uint_t *elem;

	/*
	vec3 eyedir=viewer->getEyeOrientation();
	vec3 eyepos = viewer->getPosition();
	*/

    shader = bspDataModel->r_faces[skylist[0]].shader;
    skyheight = bspDataModel->r_shaders[shader].skyheight;
    arrays.numverts = arrays.numelems = 0;

    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_TEXTURE_COORD_ARRAY);

    /* Center skybox on camera to give the illusion of a larger space */
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
	vec3 pos=viewer->getPosition();
    glTranslatef(pos[0], pos[1], pos[2]);
    glScalef(skyheight, skyheight, skyheight);

    /* FIXME: Need to cull skybox based on face list */
    for (s=0; s < 5; s++)
    {
		/*
		vec3 skypoint1(bspDataModel->r_skybox->points[s][0][0], bspDataModel->r_skybox->points[s][0][1], bspDataModel->r_skybox->points[s][0][2]);
		vec3 skypoint2(bspDataModel->r_skybox->points[s][1][0], bspDataModel->r_skybox->points[s][1][1], bspDataModel->r_skybox->points[s][1][2]);
		vec3 skypoint3(bspDataModel->r_skybox->points[s][2][0], bspDataModel->r_skybox->points[s][2][1], bspDataModel->r_skybox->points[s][2][2]);
		vec3 skypoint4(bspDataModel->r_skybox->points[s][3][0], bspDataModel->r_skybox->points[s][3][1], bspDataModel->r_skybox->points[s][3][2]);
		if (vec_dot(skypoint1, eyedir) > cos_fov && vec_dot(skypoint2, eyedir) > cos_fov && vec_dot(skypoint3, eyedir) > cos_fov && vec_dot(skypoint4, eyedir) > cos_fov)
		{
			printf("dont render sky %d\n", s);
			continue;
		}
		*/

	elem = bspDataModel->r_skybox->elems;
	for (i=0; i < bspDataModel->r_skybox->numelems; i++)
	{
	    arrays.elems[arrays.numelems++] = arrays.numverts + *elem++;
	}
	for (i=0; i < bspDataModel->r_skybox->numpoints; i++)
	{
	    vec_copy(bspDataModel->r_skybox->points[s][i], arrays.verts[arrays.numverts]);
	    arrays.verts[arrays.numverts][3] = 1.0f;
	    vec2_copy(bspDataModel->r_skybox->tex_st[s][i], arrays.tex_st[arrays.numverts]);
	    arrays.numverts++;
	}
    }

    render_flush(shader, 0);

    /* Restore world space */
    glMatrixMode(GL_MODELVIEW);
    glPopMatrix();
}

#if 0
/******************************* dont compile *************************/
void RenderEngine::render_backend_mapent(int mapent)
{
//#if 0
    int i, j, k;
    mapent_inst_t *inst = getInstance(mapent);
    mapent_class_t *klass = getClass(inst->klass);
    md3model_t *model;
    md3mesh_t *mesh;
	
    uint_t *elem;
    float funcargs[4];
    float bob;

    arrays.numverts = arrays.numelems = 0;

    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_TEXTURE_COORD_ARRAY);

    /* Calculate bob amount */
    funcargs[0] = funcargs[2] = 0.0f;
    funcargs[1] = klass->bobheight;
    funcargs[3] = inst->bobspeed;
    bob = (float)render_func_eval(SHADER_FUNC_SIN, funcargs);

    /* Translate to model origin + bob amount */
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glTranslatef(inst->origin[0], inst->origin[1], inst->origin[2] + bob);

    for (i=0; i < klass->numparts; i++)
    {
	model = getMD3(klass->parts[i].md3index);

	/* Scale and rotate part */
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glScalef(klass->parts[i].scale, klass->parts[i].scale,
		 klass->parts[i].scale); 
	glRotated(klass->parts[i].rotspeed * g_frametime, 0.0, 0.0, 1.0);
	    
	for (j = 0; j < model->nummeshes; j++)
	{
	    mesh = &model->meshes[j];
	    
	    elem = mesh->elems;
	    for (k = 0; k < mesh->numelems; k++)
	    {
		arrays.elems[arrays.numelems++] = arrays.numverts + *elem++;
	    }

	    for (k = 0; k < mesh->numverts; k++)
	    {
		vec_copy(mesh->points[k], arrays.verts[arrays.numverts]);
		arrays.verts[arrays.numverts][3] = 1.0f;
		vec2_copy(mesh->tex_st[k],  arrays.tex_st[arrays.numverts]);
		arrays.numverts++;
	    }

	    /* Draw it */
	    render_flush(mesh->shader, 0);
	}
	/* Restore unrotated state */
	glMatrixMode(GL_MODELVIEW);
	glPopMatrix();
    }

    /* Restore world space */
    glMatrixMode(GL_MODELVIEW);
    glPopMatrix();
//#endif
}
/***************** dont compile ***************************************/
#endif

void RenderEngine::render_pushface(face_t *face)
{
    int i, *elem;
    vertex_t *vert;

    elem = &(bspDataModel->r_elems[face->firstelem]);
    for (i = 0; i < face->numelems; ++i)
    {
	arrays.elems[arrays.numelems++] = arrays.numverts + *elem++;
    }
    
    vert = &(bspDataModel->r_verts[face->firstvert]);
    for (i = 0; i < face->numverts; ++i)
    {
	vec_copy(vert->v_point, arrays.verts[arrays.numverts]);
	arrays.verts[arrays.numverts][3] = 1.0f;
	vec2_copy(vert->tex_st, arrays.tex_st[arrays.numverts]);
	vec2_copy(vert->lm_st, arrays.lm_st[arrays.numverts]);	
	if (bspDataModel->r_shaders[face->shader].flags & SHADER_NEEDCOLOURS)
	    colour_copy(vert->colour, arrays.colour[arrays.numverts]);
	vert++;
	arrays.numverts++;
    }	    
}

void RenderEngine::render_pushface_deformed(int shadernum, face_t *face)
{
    /* Push the face, deforming each vertex as we go. */
    /* FIXME: Better to deform vertexes after pushing, but where
       does the normal info come from ? */
    /* Only wave deformation supported here */
    shader_t *shader = &(bspDataModel->r_shaders[shadernum]);
    float args[4], startoff, off, wavesize, deflect;
    int i, *elem;
    vertex_t *vert;
    vec3_t v;

    /* Setup wave function */
    args[0] = shader->deformv_wavefunc.args[0];
    args[1] = shader->deformv_wavefunc.args[1];
    args[3] = shader->deformv_wavefunc.args[3];
    startoff = shader->deformv_wavefunc.args[2];
    wavesize = shader->deformv_wavesize;

    elem = &(bspDataModel->r_elems[face->firstelem]);
    for (i = 0; i < face->numelems; ++i)
    {
	arrays.elems[arrays.numelems++] = arrays.numverts + *elem++;
    }
        
    vert = &(bspDataModel->r_verts[face->firstvert]);
    for (i = 0; i < face->numverts; ++i)
    {
	/* FIXME: this clearly isn't the way deform waves are applied to
	   world coordinates.  For now, it at least waves the banners :) */
	off = (vert->v_point[0] + vert->v_point[1] + vert->v_point[2]) /
	    wavesize;

	/* Evaluate wave function */
	args[2] = startoff + off;
	deflect = render_func_eval(shader->deformv_wavefunc.func, args);
	/* Deflect vertex along its normal vector by wave amount */
	vec_copy(vert->v_norm, v);
	vec_scale(v, deflect, v);
	vec_add(v, vert->v_point, v);

	/* Push it */
	vec_copy(v, arrays.verts[arrays.numverts]);
	arrays.verts[arrays.numverts][3] = 1.0f;
	vec2_copy(vert->tex_st, arrays.tex_st[arrays.numverts]);
	vec2_copy(vert->lm_st, arrays.lm_st[arrays.numverts]);	
	if (bspDataModel->r_shaders[face->shader].flags & SHADER_NEEDCOLOURS)
	    colour_copy(vert->colour, arrays.colour[arrays.numverts]);
	vert++;
	arrays.numverts++;	
    }	    
}

void RenderEngine::render_pushmesh(mesh_t *mesh)
{
    int u, v, i, *elem;

    elem = (int *)mesh->elems;
    for (i = 0; i < mesh->numelems; ++i)
    {
	arrays.elems[arrays.numelems++] = arrays.numverts + *elem++;
    }
    
    i = 0;    
    for (v = 0; v < mesh->size[1]; ++v)
    {
	for (u = 0; u < mesh->size[0]; ++u)
	{
	    vec_copy(mesh->points[i], arrays.verts[arrays.numverts]);
	    arrays.verts[arrays.numverts][3] = 1.0f;
	    vec2_copy(mesh->tex_st[i],  arrays.tex_st[arrays.numverts]);
	    vec2_copy(mesh->lm_st[i], arrays.lm_st[arrays.numverts]);
	    arrays.numverts++;
	    i++;
	}
    }
}

void RenderEngine::render_stripmine(int numelems, int *elems)
{
    int toggle;
    uint_t a, b, elem;

    /* Vertexes are in tristrip order where possible.  If we can't lock
     * the vertex arrays (glLockArraysEXT), then it's better to send
     * tristrips instead of triangles (less transformations).
     * This function looks for and sends tristrips.
     */

    /* Tristrip order elems look like this:
     *  0 1 2 2 1 3 2 3 4 4 3 5 4 5 7 7 5 6  <-- elems
     *    b a a b b a b a a b b a b a a b b  <-- ab pattern
     *    \ 1 / \ 2 / \ 3 / \ 4 / \ 5 /      <-- baa/bba groups
     */
    
    elem = 0;
    while (elem < numelems)
    {
	toggle = 1;
	glBegin(GL_TRIANGLE_STRIP);
	
	glArrayElement(elems[elem++]);
	b = elems[elem++];
	glArrayElement(b);
	a = elems[elem++];
	glArrayElement(a);
	
	while (elem < numelems)
	{
	    if (a != elems[elem] || b != elems[elem+1])
		break;
	    
	    if (toggle)
	    {
		b = elems[elem+2];
		glArrayElement(b);
	    }
	    else
	    {
		a = elems[elem+2];
		glArrayElement(a);
	    }
	    elem += 3;
	    toggle = !toggle;
	}
	glEnd();
    }
}

void RenderEngine::render_flush(int shadernum, int lmtex)
{
    int p;
    shader_t *shader = &(bspDataModel->r_shaders[shadernum]);
    
    if (arrays.numverts == 0) return;

    /* Face culling */
    if (shader->flags & SHADER_NOCULL)
	glDisable(GL_CULL_FACE);
    else
	glEnable(GL_CULL_FACE);

    /* FIXME: if compiled vertex arrays supported, lock vertex array here */
    glVertexPointer(4, GL_FLOAT, 0, arrays.verts);
    
    if (shader->flags & SHADER_NEEDCOLOURS)
	glColorPointer(4, GL_UNSIGNED_BYTE, 0, arrays.colour);

    /* FIXME: Multitexturing, if supported...
     * Multitexturing can be handled by examining the number of passes
     * for this shader, and spreading them amongst available texture
     * units.  E.g. if there are 3 passes and 2 tex units, we do 2 at
     * once and then one -- glDrawElements() is called twice.
     */
    
    for (p=0; p < shader->numpasses; p++)
    {
	/* Set rendering state for this pass */
	if (!render_setstate(&shader->pass[p], lmtex))
	    continue;
	
#if 0
	glDrawElements(GL_TRIANGLES, arrays.numelems, GL_UNSIGNED_INT,
		       arrays.elems);
#else
	/* We don't have compiled vertex arrays (locking) so find tristrips */
	render_stripmine(arrays.numelems, arrays.elems);
#endif
	
	/* Clear certain rendering state variables */
	render_clearstate(&shader->pass[p]);
    }
    
    /* Clear arrays */
    arrays.numverts = arrays.numelems = 0; 
}

double RenderEngine::render_func_eval(uint_t func, float *args)
{
    double x, y;

    /* Evaluate a number of time based periodic functions */
    /* y = args[0] + args[1] * func( (time + arg[3]) * arg[2] ) */
    
    x = (g_frametime + args[2]) * args[3];
    x -= floor(x);

    switch (func)
    {
	case SHADER_FUNC_SIN:
	    y = sin(x * TWOPI);
	    break;
	    
	case SHADER_FUNC_TRIANGLE:
	    if (x < 0.5)
		y = 2.0 * x - 1.0;
	    else
		y = -2.0 * x + 2.0;
	    break;
	    
	case SHADER_FUNC_SQUARE:
	    if (x < 0.5)
		y = 1.0;
	    else
		y = -1.0;
	    break;
	    
	case SHADER_FUNC_SAWTOOTH:
	    y = x;
	    break;
	    
	case SHADER_FUNC_INVERSESAWTOOTH:
	    y = 1.0 - x;
	    break;
    }

    return y * args[1] + args[0];
}

int RenderEngine::render_setstate(shaderpass_t *pass, uint_t lmtex)
{
	static unsigned int oldtime = 0, imagenum = 4;
	int curtime;

    if (pass->flags & SHADER_LIGHTMAP)
    {
			/* Select lightmap texture */
			glTexCoordPointer(2, GL_FLOAT, 0, arrays.lm_st);
			glBindTexture(GL_TEXTURE_2D, bspDataModel->r_lightmaptex[lmtex]);
    }
    else if (pass->flags & SHADER_ANIMMAP)
    {
		uint_t texobj;
		int frame;

		/* Animation: get frame for current time */
		frame = (int)(g_frametime * pass->anim_fps) % pass->anim_numframes;
		
		glTexCoordPointer(2, GL_FLOAT, 0, arrays.tex_st);
		texobj = (bspDataModel->r_textures[pass->anim_frames[frame]]);
		if (texobj < 0) return 0;
		glBindTexture(GL_TEXTURE_2D, texobj);
    }
    else
    {
	uint_t texobj;

		/* Set normal texture */

		if (matrixmode)
		{/*
			curtime = glutGet(GLUT_ELAPSED_TIME);
			if (curtime - oldtime > MATRIXMODETIMEDIFF)
			{
				oldtime = curtime;
				imagenum = (int) NUM_MATRIX_TEXTURES * (rand()/(float)(RAND_MAX+1));
				//if (imagenum == 1) imagenum = 0;
			}
			*/
			glTexCoordPointer(2, GL_FLOAT, 0, arrays.tex_st);
			glBindTexture(GL_TEXTURE_2D, bspDataModel->matrix_textures[1]);
		}
		else
		{
			glTexCoordPointer(2, GL_FLOAT, 0, arrays.tex_st);
			if (pass->texref < 0) return 0;
			texobj = (bspDataModel->r_textures[pass->texref]);
			if (texobj < 0) return 0;
			glBindTexture(GL_TEXTURE_2D, texobj);
		}
    }

    if (pass->flags & SHADER_BLEND)
    {
	glEnable(GL_BLEND);
	glBlendFunc(pass->blendsrc, pass->blenddst);
    }
    else
	glDisable(GL_BLEND);

    if (pass->flags & SHADER_ALPHAFUNC)
    {
	glEnable(GL_ALPHA_TEST);
	glAlphaFunc(pass->alphafunc, pass->alphafuncref);
    }
    else
	glDisable(GL_ALPHA_TEST);
    
    glDepthFunc(pass->depthfunc);
    if (pass->flags & SHADER_DEPTHWRITE)
	glDepthMask(GL_TRUE);
    else
	glDepthMask(GL_FALSE);

    if (pass->rgbgen == SHADER_GEN_IDENTITY)
	glColor4f(1.0f, 1.0f, 1.0f, 1.0f);
    else if (pass->rgbgen == SHADER_GEN_WAVE)
    {
	float rgb = (float)render_func_eval(pass->rgbgen_func.func,
					    pass->rgbgen_func.args);
	glColor4f(rgb, rgb, rgb, 1.0f);
    }
    else if (pass->rgbgen == SHADER_GEN_VERTEX)
	/* FIXME: I don't think vertex colours are what is meant here */
	glEnableClientState(GL_COLOR_ARRAY);

    if (pass->flags & SHADER_TCMOD)
    {
	/* Save identity texture transform */
	glMatrixMode(GL_TEXTURE);
	glPushMatrix();

	/* Move center of texture to origin */
	glTranslatef(0.5f, 0.5f, 0.0f);
	
	/* FIXME: Is this the right order for these transforms ? */

	if (pass->tcmod & SHADER_TCMOD_ROTATE)
	    glRotated(pass->tcmod_rotate * g_frametime, 0.0, 0.0, 1.0);
	if (pass->tcmod & SHADER_TCMOD_SCALE)
	    glScalef(pass->tcmod_scale[0], pass->tcmod_scale[1], 1.0f);

	if (pass->tcmod & SHADER_TCMOD_TURB)
	{
	    /* Don't know what the exact transform is for turbulance, but
	       this seems to do something close: sin wave scaling in s
	       and t with a 90 degrees phase difference */
	    double x, y1, y2;
	    x = (g_frametime + pass->tcmod_turb[2]) * pass->tcmod_turb[3];
	    x -= floor(x);
	    y1 = sin(x * TWOPI) * pass->tcmod_turb[1] + pass->tcmod_turb[0];
	    y2 = sin((x+0.25) * TWOPI) * pass->tcmod_turb[1] +
		pass->tcmod_turb[0];
	    glScaled(1.0+y1*TURB_SCALE, 1.0+y2*TURB_SCALE, 1.0);
	}

	if (pass->tcmod & SHADER_TCMOD_STRETCH)
	{
	    double y = render_func_eval(pass->tcmod_stretch.func,
					pass->tcmod_stretch.args);
	    glScaled(1.0/y, 1.0/y, 1.0);
	}
	
	if (pass->tcmod & SHADER_TCMOD_SCROLL)
	    glTranslated(pass->tcmod_scroll[0] * g_frametime,
			 pass->tcmod_scroll[1] * g_frametime, 0.0);

	/* Replace center of texture */
	glTranslatef(-0.5f, -0.5f, 0.0f);
    }

    return 1;
}

void RenderEngine::render_clearstate(shaderpass_t *pass)
{
    if (pass->flags & SHADER_TCMOD)
    {
	/* Revert to identity texture transform */
	glMatrixMode(GL_TEXTURE);
	glPopMatrix();
    }
    if (pass->rgbgen == SHADER_GEN_VERTEX)
	glDisableClientState(GL_COLOR_ARRAY);
}


/* find the first spawn position */

void RenderEngine::find_start_pos()
{
    /* Find the first spawn point in the entities list */
    int i;
    const char *cname;
    
    r_eye_el = r_eye_az = 0.0;
	vec3_t posit;
	vec3 pos;
    for (i = 0; i < bspDataModel->g_numentities; i++)
    {
		cname = bspDataModel->entity_value(i, "classname");
		if (!strcmp(cname, "info_player_deathmatch"))
		{
			r_eye_az = bspDataModel->entity_float(i, "angle");
			bspDataModel->entity_vec3(i, "origin", posit);
			pos[0]=posit[0];pos[1]=posit[1];pos[2]=posit[2];
			viewer->setPosition(pos);
			break;
		}
    }
    vec_point(r_eyedir, r_eye_az, r_eye_el);
}


void RenderEngine::move(float  move, float strafe, float jump, double intervaltime)
{
#ifdef SEGFAULTOK
	int steps;
	if (intervaltime > .05 ){
		steps = 1 + (int)(intervaltime/.05f);
		intervaltime /= steps;
		for (int i=0; i<steps; i++){
			RenderEngine::move (move,strafe,jump,intervaltime);
		}
		return;
	}
#endif
	static double totalTime = 0;
	vec3 plV = viewer->getVelocity();
    float vx=plV[0]; float vy=plV[1]; float vz=plV[2];	
	if (jump > 0 && abs(vz) == 0) vz += JUMP_SPEED;
	vz = vz - GRAVITY * intervaltime;
	
	vec3_t delta, newPos, desPos;
	Sphere sp;
	
	vec3 pos=viewer->getPosition();
	vec3_t posit;
	posit[0]=pos[0];posit[1]=pos[1];posit[2]=pos[2]-50;
	vec3_t eyedirect;
	vec3 eyedir=viewer->getEyeOrientation();
	
	// 2-d movement
	eyedirect[0]=eyedir[0];eyedirect[1]=eyedir[1];eyedirect[2]=0.0;
	float size_eyedirect = 1/sqrt(eyedirect[0]*eyedirect[0]+eyedirect[1]*eyedirect[1]);
	vec_scale (eyedirect, size_eyedirect, eyedirect);
	
	// 3-d movement
	// eyedirect[0]=eyedir[0];eyedirect[1]=eyedir[1];eyedirect[2]=eyedir[2];
	
	//printf ("position  %f %f %f\n", pos[0], pos[1], pos[2]);
	
	// add movement
	vec_copy(eyedirect, delta);
	vec_scale(delta, move * MOVE_SPEED * intervaltime, delta);
	vec_add(posit, delta, newPos);
	
	// add strafe
	delta[0] = eyedirect[1]; delta[1] = -eyedirect[0]; delta[2] = 0.0f;
	vec_scale(delta, strafe * MOVE_SPEED * intervaltime, delta);
	
	vec_add(newPos, delta, newPos);
	
	// add dynamics
	delta[0] = vx; delta[1] = vy; delta[2] = vz;
	vec_scale(delta, intervaltime, delta);
	vec_add(newPos, delta, newPos);

	desPos[0] = newPos[0]; 	desPos[1] = newPos[1]; 	desPos[2] = newPos[2];
	
	
	sp.setPosition(newPos[0], newPos[1], newPos[2]);
	sp.setRadius(35.0f);

	if (enableCollDetect){
		int collPart = 0;
		int stairClimb = 0;
		int invalidLocation = 0;
		int rayCollide = 0;
		float colRatio;
		float modelDistance;
		int playerCollide = getClosestPlayer(newPos[0],newPos[1],newPos[2]);
		int modelCollide = getClosestObject(newPos[0],newPos[1],newPos[2],modelDistance);

		if (playerCollide>=0){
			vec3 pPos;
			float playerDist;
			pPos = playerList->getPlayer(playerCollide)->getPosition();
			playerDist = sqrt((newPos[0]-pPos[0])*(newPos[0]-pPos[0])+(newPos[1]-pPos[1])*(newPos[1]-pPos[1])+(newPos[2]-pPos[2])*(newPos[2]-pPos[2]));
			if (playerDist < 50){
				//printf ("collision with player %d\n", playerCollide);
				newPos[0] += (50-playerDist) * ((newPos[0]-pPos[0])/playerDist);
				newPos[1] += (50-playerDist) * ((newPos[1]-pPos[1])/playerDist);
				newPos[2] += (50-playerDist) * ((newPos[2]-pPos[2])/playerDist);
			}
		}

		
		if (modelCollide>=0 && modelDistance < 75){
			viewer->sendPickUpRequest(modelCollide);
		}
		

		collPart = checkCollision(0,&sp);
		
		if  (!collPart){
			vec_copy(newPos,posit);
		}
		else if (collPart>0)
		{
			int collList[50];
			int numColls = checkCollisionMultiple (0, &sp, collList);
			for (int i=0;i<numColls; i++){
				//printf ("correcting with plane %d\n", collList[i]);
				plane_t *plane = &(bspDataModel->r_planes[collList[i]]);
				
				Plane collPlane;
				collPlane.setNormal (plane->vec[0], plane->vec[1], plane->vec[2]);
				collPlane.setDist(plane->offset);
				float distOffset = -collPlane.distance(sp)+sp.getRadius();
				
				newPos[0] += distOffset * plane->vec[0];
				newPos[1] += distOffset * plane->vec[1];
				newPos[2] += distOffset * plane->vec[2];
				sp.setPosition(newPos[0], newPos[1], newPos[2]);
				if (abs(plane->vec[2]) > 0.7) {
					if (distOffset > 2+abs(vz*intervaltime)) { stairClimb = 1; }
					vz = 0;					
				}
			}
			
			invalidLocation = checkInvalidLocation(0,&sp);
			rayCollide = checkCollision (0, posit[0], posit[1], posit[2], 
										(newPos[0]-posit[0]),(newPos[1]-posit[1]),(newPos[2]-posit[2]) , 1 , colRatio);
			if (stairClimb) { newPos[0] = desPos[0]; newPos[1] = desPos[1]; newPos[2] += 10;}
			vec_copy(newPos,posit);
		}
		else {
			printf ("illegal area\n");
		}
	}
	else
		vec_copy(newPos,posit);
	
	posit[2] += 50;
    plV[0] = vx;  plV[1] = vy; plV[2] = vz;	
	viewer->setVelocity(plV);
	viewer->setPosition(vec3(posit[0],posit[1],posit[2]));
}


void RenderEngine::moveEye(int dx, int dy)
{
	r_eye_az += dx * AZ_SCALE;
    r_eye_el += dy * EL_SCALE;

	viewer->lookUp(dy/MOUSE_SPEED_Y);
	viewer->turnLeft(dx/MOUSE_SPEED_X);



}

void RenderEngine::renderModel(XPModel *model){
        // backup normal
        float normals[3];

        float ambient[4]={0.2, 0.2, 0.2, 1.0};
        float diffuse[4]={0.8, 0.8, 0.8, 1.0};
        float specular[4]={0.0, 0.0, 0.0, 1.0};
        float emission[4]={0.0, 0.0, 0.0, 1.0};
        float shiny[1];

        glGetFloatv(GL_CURRENT_NORMAL,normals);

        glGetMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT,ambient);
        glGetMaterialfv(GL_FRONT_AND_BACK,GL_DIFFUSE,diffuse);
        glGetMaterialfv(GL_FRONT_AND_BACK,GL_SPECULAR,specular);
        glGetMaterialfv(GL_FRONT_AND_BACK,GL_EMISSION,emission);
        glGetMaterialfv(GL_FRONT_AND_BACK,GL_SHININESS,shiny);

        glDisable(GL_BLEND);
        //Need this
        glDepthMask(GL_TRUE);
        glDisable(GL_CULL_FACE);

        vec3 pos=model->getPosition();
        vec3_t posit;
        posit[0]=pos[0];
        posit[1]=pos[1];
        posit[2]=pos[2];
        int cluster=find_cluster(posit);

        if(cluster<0){
                return;
        }
        //Culling object not visible in current cluster

        if(!((r_eyecluster < 0) ||
                (RENDER_BSP_TESTVIS(r_eyecluster,cluster)))){
                return;

}

//      printf("Model %d being rendered\n",model->getModelTypeIndex());

    /* Translate to model origin + bob amount */
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();


        vec3 origin=model->getPosition();
        vec3 orientation=model->getOrientation();

        glTranslatef(origin[0], origin[1],origin[2]);

        if(viewer==model){

                        //printf("%f ",orientation[2]);
        }
        glScalef(20, 20,20);
        glRotatef(-orientation[0],-sin(M_PI*orientation[2]/180.0), cos(M_PI*orientation[2]/180.0),0);
		glRotatef(orientation[2],0,0,1);


        //Start rendering the subParts

        int numSubParts=model->getNumberOfSubParts();

        for (int i=0; i < numSubParts; i++)
		{
                glMatrixMode(GL_MODELVIEW);
              glPushMatrix();


                XPSubPart *subPart=model->getSubPart(i);
                pos=subPart->getPosition();
                glTranslatef(pos[0],pos[1],pos[2]);

                /* Scale and rotate part */
                glScalef(subPart->getScale(), subPart->getScale(),subPart->getScale());

                orientation=subPart->getOrientation();
                glRotatef(orientation[0],1,0,0);
                glRotatef(orientation[1],0,1,0);
                glRotatef(orientation[2],0,0,1);

                int index=subPart->getBasicGraphicIndex();
                int numparts = basicGraphicList->getGraphicModel(index)->getNumParts();

                GLint list;
                if (!numparts) {
                        list=basicGraphicList->getGraphicModel(index)->getDisplayList();
                        glCallList(list);
                        glFlush();
                }
                else{
					GraphicModel* gModel = basicGraphicList->getGraphicModel(index);    
					
					if (numparts){
						//Get from model
						float animtime = model->getAnimationTime();
						int action = model->getAction();
						int paction = model->getPreviousAction();
						int animaction;
						
						if (action == 0){
							if (paction == 0) animaction = 0;
							if (paction == 1) {
								animaction = 3;
								if (animtime>gModel->getFinishTime(animaction)){									
									animtime -= gModel->getFinishTime(animaction);
									animaction = 0;
								}
							}
							if (paction == 2) {
								animaction == 6;
								if (animtime>gModel->getFinishTime(animaction)){									
									animtime -= gModel->getFinishTime(animaction);
									animaction = 0;
								}
							}							
						}
						if (action == 1){
							if (paction == 1) animaction = 2;
							if (paction == 0) {
								animaction = 1;
								if (animtime>gModel->getFinishTime(animaction)){									
									animtime -= gModel->getFinishTime(animaction);
									animaction = 2;
								}
							}
							if (paction == 2) {
								animaction = 1;
								if (animtime>gModel->getFinishTime(animaction)){									
									animtime -= gModel->getFinishTime(animaction);
									animaction = 2;
								}
							}							
						}
						if (action == 2){
							if (paction == 2) animaction = 5;
							if (paction == 0) {
								animaction = 4;
								if (animtime>gModel->getFinishTime(animaction)){									
									animtime -= gModel->getFinishTime(animaction);
									animaction = 5;
								}
							}
							if (paction == 1) {
								animaction = 4;
								if (animtime>gModel->getFinishTime(animaction)){									
									animtime -= gModel->getFinishTime(animaction);
									animaction = 5;
								}
							}							
						}
						
						list=basicGraphicList->getGraphicModel(index)->getDisplayList(action,animtime);
						glCallList(list);
						glFlush();
						
						//Store back in model
						//model->setAnimationTime(animtime);
						//model->setAction(action);
					}
                }

                glMatrixMode(GL_MODELVIEW);
                glPopMatrix();
    }

        //restore normal
        glNormal3f (normals[0],normals[1],normals[2]);

        glMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT,ambient);
        glMaterialfv(GL_FRONT_AND_BACK,GL_DIFFUSE,diffuse);
        glMaterialfv(GL_FRONT_AND_BACK,GL_SPECULAR,specular);
        glMaterialfv(GL_FRONT_AND_BACK,GL_EMISSION,emission);
        glMaterialf(GL_FRONT_AND_BACK,GL_SHININESS,shiny[1]);



    /* Restore world space */
    glMatrixMode(GL_MODELVIEW);
    glPopMatrix();

}

int RenderEngine::checkBrush (int n, float x, float y, float z, float dx, float dy, float dz, float ln, float &ratio)
{	
	brush_t *brush = &(bspDataModel->r_brushes[n]);	
	brushside_t *bside;
	shaderref_t *shader =  &(bspDataModel->r_shaderrefs[n]);	
	
	plane_t *plane;
	Plane pl;
	Sphere Sp;
	Sp.setRadius(5.0f);

	float curRatio = -1;
	int returnPlane = 0;
	
	for (int i = brush->firstside; i < (brush->firstside + brush->numsides); i++){
		bside = &(bspDataModel->r_brushsides[i]);
		plane = &(bspDataModel->r_planes[bside->plane]);
		pl.setNormal(plane->vec[0], plane->vec[1], plane->vec[2]);
		pl.setDist(plane->offset);		
		curRatio = pl.collideRatio(x,y,z,dx,dy,dz,ln);
		if (curRatio > 0){
			Sp.setPosition(x+curRatio*dx*ln,y+curRatio*dy*ln,z+curRatio*dz*ln);
			if (checkBrush(n,&Sp)){
				if (ratio > 0 && curRatio < ratio ){
					returnPlane = bside->plane;
					ratio = curRatio;
				}
				else if (ratio < 0) {
					returnPlane = bside->plane;
					ratio = curRatio;
				}
			}
		}		
	}
	return returnPlane;
}

int RenderEngine::checkLeaf (int n, float x, float y, float z, float dx, float dy, float dz, float ln, float &ratio)
{
	leaf_t *leaf = &(bspDataModel->r_leafs[n]);
	int collision;
	int brushnum;
	int returnPlane = 0;

	for (int i=leaf->firstbrush; i < (leaf->firstbrush + leaf->numbrushes); i++){
		brushnum = bspDataModel->r_lbrushes[i];
		collision = checkBrush ( brushnum, x,y,z,dx,dy,dz,ln,ratio);
		if (collision) returnPlane = collision;
	}
	return returnPlane;
}

int RenderEngine::checkCollision (int n, float x, float y, float z, float dx, float dy, float dz, float ln, float &ratio)
{	
	if (n==0) ratio = -1;
	node_t *node = &(bspDataModel->r_nodes[n]);	
	Box bspBox(node->bbox);

	if (!(bspBox.collide(x,y,z,dx,dy,dx,ln))) {
		return 0;
	}

	int returnPlane = 0;
	int collision = 0;

	// leaves		
	if (node->children[0] < 0){
		collision = checkLeaf (~node->children[0], x,y,z,dx,dy,dx,ln,ratio);
		if (collision) returnPlane = collision;
	}

	if (node->children[1] < 0){
		collision = checkLeaf (~node->children[1], x,y,z,dx,dy,dx,ln,ratio);
		if (collision) returnPlane = collision;
	}
	
	if (node->children[0]>=0) {			
		collision = checkCollision (node->children[0], x,y,z,dx,dy,dx,ln,ratio);
		if (collision) returnPlane = collision;
	}
	
	if (node->children[1]>=0) {
		collision = checkCollision (node->children[1], x,y,z,dx,dy,dx,ln,ratio);
		if (collision) returnPlane = collision;
	}

	return returnPlane;
}


int RenderEngine::checkBrush (int n, Sphere *Sp)

{	

	brush_t *brush = &(bspDataModel->r_brushes[n]);	

	brushside_t *bside;

	shaderref_t *shader =  &(bspDataModel->r_shaderrefs[n]);	

	//printf ("brush contents %d\n", (shader->contents & 1));
	//if (!(shader->contents & 1)) return 0;


	plane_t *plane;
	Plane pl;

	float dist;	
	bool allNeg = true;

	float mindist=100000.0f;
	int returnPlane = 0;

	for (int i = brush->firstside; i < (brush->firstside + brush->numsides); i++){
		bside = &(bspDataModel->r_brushsides[i]);
		plane = &(bspDataModel->r_planes[bside->plane]);
		pl.setNormal(plane->vec[0], plane->vec[1], plane->vec[2]);
		pl.setDist(plane->offset);
		dist = pl.distance(*Sp);
		dist -= Sp->getRadius();


		if (abs(dist) < mindist ){
			mindist = abs(dist);
			returnPlane = bside->plane;
		}

		if (dist>0) { 
			allNeg = false;  // can break here
			return 0;
		}		
	}

	if (allNeg) { return returnPlane; }
	return 0;

}





int RenderEngine::checkLeaf (int n, Sphere *Sp)

{
	leaf_t *leaf = &(bspDataModel->r_leafs[n]);

	int collision;
	int brushnum;
	int i;

	for (i=leaf->firstbrush; i < (leaf->firstbrush + leaf->numbrushes); i++){
		brushnum = bspDataModel->r_lbrushes[i];
		collision = checkBrush ( brushnum, Sp );
		if (collision){ return collision; }
	}

#if 0
	//satyam
	for (i=leaf->firstface; i < (leaf->firstface + leaf->numfaces); i++){
		if (bspDataModel->r_faces[i].facetype == FACETYPE_MESH)
			printf("HEY! I GOT A MESHTYPE\n");
	}
#endif

	return 0;

}





int RenderEngine::checkCollision (int n, Sphere *Sp)

{	
	node_t *node = &(bspDataModel->r_nodes[n]);	
	Box bspBox(node->bbox);
	if (!(bspBox.collide(*Sp))) {
		return 0;

	}

	int collision = 0;
	// leaves		

	if (node->children[0] < 0){
		collision = checkLeaf (~node->children[0], Sp);
		if (collision) { return  collision;	}
	}



	if (node->children[1] < 0){
		collision = checkLeaf (~node->children[1], Sp);
		if (collision) { return  collision;	}
	}

	

	if (node->children[0]>=0) {			
		collision = checkCollision (node->children[0], Sp);
		if (collision) { return  collision;	}
	}

	

	if (node->children[1]>=0) {
		collision = checkCollision (node->children[1], Sp);
		if (collision) { return  collision;	}
	}

	return 0;
}


int RenderEngine::checkLeafMultiple (int n, Sphere *Sp, int* collList)
{
	static unsigned char brushflag[ 1025 ];

	leaf_t *leaf = &(bspDataModel->r_leafs[n]);
	int collision;
	int brushnum;
	int numColl = 0;


	for (int i=leaf->firstbrush; i < (leaf->firstbrush + leaf->numbrushes); i++){
		brushnum = bspDataModel->r_lbrushes[i];
		collision = checkBrush ( brushnum, Sp );
		if (collision){			
			collList[numColl] = collision;
			numColl++;			
		}
	}


	return numColl;

}





int RenderEngine::checkCollisionMultiple (int n, Sphere *Sp, int* collList)
{	
	node_t *node = &(bspDataModel->r_nodes[n]);
	int numColl = 0;
	Box bspBox(node->bbox);

	if (!(bspBox.collide(*Sp))) {
		return 0;
	}



	int collision = 0;
	// leaves		
	if (node->children[0] < 0){
		collision = checkLeafMultiple (~node->children[0], Sp, collList+numColl);
		if (collision) {
			numColl+=collision;			
		}

	}

	if (node->children[1] < 0){
		collision = checkLeafMultiple (~node->children[1], Sp, collList+numColl);
		if (collision) {
			numColl+=collision;			
		}
	}
	
	if (node->children[0]>=0) {			
		collision = checkCollisionMultiple (node->children[0], Sp, collList+numColl);
		if (collision) {
			numColl+=collision;			
		}
	}
	
	if (node->children[1]>=0) {
		collision = checkCollisionMultiple (node->children[1], Sp, collList+numColl);
		if (collision) {
			numColl+=collision;			
		}
	}				

	return numColl;	

}

int RenderEngine::checkInvalidLocation (int n, Sphere *Sp)
{
	node_t *node = &(bspDataModel->r_nodes[n]);	
	leaf_t *leaf;
	Box bspBox(node->bbox);

	if (!(bspBox.collide(*Sp))) {
		return 0;
	}

	int invalid = 0;

	// leaves		
	if (node->children[0] < 0){
		leaf = &(bspDataModel->r_leafs[~node->children[0]]);
		//printf ("leaf %d area %d\n",~node->children[0], leaf->area);
		invalid = (leaf->area == -1);
		if (invalid) { return  invalid;}
	}

	if (node->children[1] < 0){
		leaf = &(bspDataModel->r_leafs[~node->children[1]]);
		//printf ("leaf %d area %d\n",~node->children[1], leaf->area);
		invalid = (leaf->area == -1);
		if (invalid) { return  invalid;}
	}
	
	if (node->children[0]>=0) {			
		invalid = checkInvalidLocation (node->children[0], Sp);
		if (invalid) { return  invalid;}
	}
	
	if (node->children[1]>=0) {
		invalid = checkInvalidLocation (node->children[1], Sp);
		if (invalid) { return  invalid;}
	}

	return 0;
}

int RenderEngine::checkPlayerCollision (float x, float y, float z, float dx, float dy, float dz, float ln, float &retdistance)
{
	int numPlayers = playerList->getLength();
	Box partBox;
	float corners[6];
	vec3 playerPos;
	vec3 playerRot;
	vec3 partPos;
	vec3 partRot;
	int numParts;
	
	float distance;
	float minDistance = 1e5f;
	int returnPlayer = -1;	
	float rayLen = ln*sqrt(dx*dx+dy*dy+dz*dz);

	Player* curPlayer;
	GraphicModel* playerModel;

	Transform playerTransform;
	Transform partTransform;
	Matrix partCenter(3,1);

	for (int i=0; i<numPlayers; i++){
		curPlayer = playerList->getPlayer(i);		
		if (curPlayer && curPlayer!=viewer){
			playerPos = curPlayer->getPosition();
			playerRot = curPlayer->getOrientation();

			playerTransform.setIdentitiy();
			playerTransform.addRotationX(playerRot[0]);
			playerTransform.addRotationY(playerRot[1]);
			playerTransform.addRotationZ(playerRot[2]);
			playerTransform.addTranslation(playerPos[0],playerPos[1],playerPos[2]);			

			numParts = curPlayer->getNumberOfSubParts();
			for (int j=0; j<numParts; j++){							
				partPos = curPlayer->getSubPart(j)->getPosition();
				partRot = curPlayer->getSubPart(j)->getOrientation();
				playerModel = basicGraphicList->getGraphicModel(curPlayer->getSubPart(j)->getBasicGraphicIndex());
				playerModel->getBoxCorners(corners);
				partBox = Box(corners);

				partTransform.setIdentitiy();
				partTransform.addRotationX(partRot[0]);
				partTransform.addRotationY(partRot[1]);
				partTransform.addRotationZ(partRot[2]);

				partTransform.addTransform(playerTransform);

				partBox.transform(partTransform);
				//partBox.printSimple();
				//printf ("this model has %d subparts\n",playerModel->getNumParts());
				//for (int k=0; k<playerModel->getNumParts(); k++){
				//}

				if (partBox.collide(x,y,z,dx,dy,dz,ln)){
					//printf("collision\n");
					partCenter = partBox.getCenter();
					distance = sqrt((partCenter(0,0)-x)*(partCenter(0,0)-x) +
						(partCenter(1,0)-y)*(partCenter(1,0)-y) + (partCenter(2,0)-z)*(partCenter(2,0)-z));
					if (distance < minDistance){
						returnPlayer = i;
						minDistance = distance;
						retdistance = minDistance;
					}
				}
			}
		}
	}
	return returnPlayer;
}

int RenderEngine::getClosestPlayer(float x, float y, float z)
{
	int numPlayers = playerList->getLength();
	vec3 playerPos;
	
	float distance;
	float minDistance = 1e15f;
	int returnPlayer = -1;	
	Player* curPlayer;

	for (int i=0; i<numPlayers; i++){
		curPlayer = playerList->getPlayer(i);		
		if (curPlayer && curPlayer!=viewer){
			playerPos = curPlayer->getPosition();
			distance = ((x-playerPos[0])*(x-playerPos[0])+(y-playerPos[1])*(y-playerPos[1])+(z-playerPos[2])*(z-playerPos[2]));
			if (distance<minDistance){
				returnPlayer = i;
				minDistance = distance;
			}
		}
	}
	return returnPlayer;
}

int RenderEngine::getClosestObject(float x, float y, float z, float& retDistance)
{
	int numModels = xpModelList->getLength();
	vec3 modelPos;
	
	float distance;
	float minDistance = 1e15f;
	int returnModel = -1;	
	XPModel* curModel;

	for (int i=0; i<numModels; i++){
		curModel = xpModelList->getModel(i);
		modelPos = curModel->getPosition();
		distance = sqrt((x-modelPos[0])*(x-modelPos[0])+(y-modelPos[1])*(y-modelPos[1])+(z-modelPos[2])*(z-modelPos[2]));
		if (distance<minDistance){
			returnModel = i;
			minDistance = distance;
		}
	}
	retDistance = minDistance;
	return returnModel;
}

void RenderEngine::setCollisionDetection(bool state)
{
	enableCollDetect = state;
}

BasicGraphicList *RenderEngine::getBasicGraphicList(){
	return basicGraphicList;
}

ModelTypeList *RenderEngine::getModelTypeList(){
	return modelTypeList;
}

XPModelList *RenderEngine::getXPModelList(){
	return xpModelList;
}

void RenderEngine::swapViewer(){
	
	//printf("Old position %f %f %f \n",viewer->getPosition()[0],viewer->getPosition()[1],viewer->getPosition()[2]); 
	int id=viewer->getPlayerID();
	do{
		if(id==MAX_CLIENTS-1){
			id=0;
		}
		else{
			id++;
		}
	}while(playerList->getPlayer(id)==NULL);
		
	viewer=playerList->getPlayer(id);
	//printf("New position %f %f %f \n",viewer->getPosition()[0],viewer->getPosition()[1],viewer->getPosition()[2]); 
	
}

Player *RenderEngine::getViewer(){
	return viewer;
}
