#include "GraphicModel.h"

#include <GL/gl.h>
#include <GL/glu.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "Bitmap.h"
#include "transform.h"

#ifdef WIN32
#define PATHSTR "resources\\models\\"
#else
#define PATHSTR "resources/models/"
#endif


struct DIB2D{
	BITMAPINFOHEADER *Info;
	RGBQUAD *palette;
	GLubyte    *bits;
};
struct GLTXTLOAD{
	GLint format;
	GLint perpixel;
	GLint Width;
	GLint Height;
	GLubyte* bits;
};

long GraphicModel::ScanBytes(int pixWidth, int bitsPixel) {
	return (((long)pixWidth*bitsPixel+31) / 32) * 4;
}

bool  GraphicModel::ScaleImage(DIB2D &dib,GLTXTLOAD&p)
{
	GLint glMaxTexDim;     // OpenGL maximum texture dimension
	GLint XDMaxTexDim=512; // user maximum texture dimension
	GLint minsize =2;
	double xPow2, yPow2;
	int ixPow2, iyPow2;
	int xSize2, ySize2;
	GLint m_iWidth=dib.Info->biWidth;
	GLint m_iHeight=dib.Info->biHeight;
	glGetIntegerv(GL_MAX_TEXTURE_SIZE, &glMaxTexDim);
	
	glMaxTexDim = min(XDMaxTexDim, glMaxTexDim);
	
	if (m_iWidth <= glMaxTexDim)
		xPow2 = log((double)m_iWidth) / log(2.0);
	else
		xPow2 = log((double)glMaxTexDim) / log(2.0);
	
	if (m_iHeight <= glMaxTexDim)
		yPow2 = log((double)m_iHeight) / log(2.0);
	else
		yPow2 = log((double)glMaxTexDim) / log(2.0);
	
	ixPow2 = (int)xPow2;
	iyPow2 = (int)yPow2;
	
	if (xPow2 != (double)ixPow2)
		ixPow2++;
	if (yPow2 != (double)iyPow2)
		iyPow2++;
	
	xSize2 = 1 << ixPow2;
	ySize2 = 1 << iyPow2;
	
	if(xSize2<minsize)xSize2=minsize;
	if(ySize2<minsize)ySize2=minsize;
	
	if(((xSize2==m_iWidth) && (ySize2==m_iHeight)))
	{
		if(dib.Info->biBitCount==24){
			p.format=GL_BGR_EXT;
			p.perpixel=3;
			return FALSE;
		}
		if(dib.Info->biBitCount==32)
		{
			p.format=GL_BGRA_EXT;
			p.perpixel=4;
			return FALSE;
		}
	}
	
	GLubyte *bits=(GLubyte *)dib.bits;
	if(dib.Info->biBitCount==8){
		
		// convert to TRUECOLOR
		int _perline=ScanBytes(8,m_iWidth);
		int perline=ScanBytes(24,m_iWidth);
		bits= new GLubyte[perline*m_iHeight * sizeof(GLubyte)];
		for(int y=0;y<m_iHeight;y++){
			GLubyte *_b=((GLubyte *)dib.bits)+y*_perline;
			GLubyte *b=bits+y*perline;
			for(int x=0;x<m_iWidth;x++){
				RGBQUAD _p=dib.palette[*_b];
				_b++;
				*b=_p.rgbBlue;b++;
				*b=_p.rgbGreen;b++;
				*b=_p.rgbRed;b++;
			}
		}
	}
	bool isAlpha=(dib.Info->biBitCount==32);
	int _mem_size=xSize2 * ySize2 *  sizeof(GLubyte);
	if(isAlpha){
		_mem_size*=4;
		p.perpixel=4;
		p.format=GL_BGRA_EXT;
	}else {
		_mem_size*=3;
		p.perpixel=3;
		p.format=GL_BGR_EXT;
	}
	GLubyte *pData = (GLubyte*)new GLubyte[_mem_size];
	if (!pData) return FALSE;
	
	if(isAlpha){
		gluScaleImage(GL_BGRA_EXT, m_iWidth, m_iHeight,
			GL_UNSIGNED_BYTE, bits,
			xSize2, ySize2, GL_UNSIGNED_BYTE, pData);
	}
	else
		gluScaleImage(GL_RGB, m_iWidth, m_iHeight,
		GL_UNSIGNED_BYTE, bits,
		xSize2, ySize2, GL_UNSIGNED_BYTE, pData);
	
	
	if(bits!=dib.bits)delete bits;
	//   m_pBits = pData;
	m_iWidth = xSize2 ;
	m_iHeight = ySize2 ;
	p.Width=m_iWidth;
	p.Height=m_iHeight;
	p.bits=pData;
	
	return TRUE ;
}


void GraphicModel::LoadTexture(char*filename)
{
	DIB2D dib;
	GLTXTLOAD load;
	BITMAPINFO *bitMapInfo;
	
	dib.bits=LoadDIBitmap(filename, &bitMapInfo);
	printf("Loaded DIBitmap\n");
	getchar();
	if(dib.bits){
		dib.Info=&bitMapInfo->bmiHeader;
		dib.palette=bitMapInfo->bmiColors;
		
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		printf("Before scaling image\n");
		getchar();
		if(ScaleImage(dib,load)){
			glTexImage2D(GL_TEXTURE_2D,0,load.perpixel,
                load.Width,load.Height,0,
                load.format,GL_UNSIGNED_BYTE,
                load.bits);
			delete load.bits;
		}
		else{
			
			glTexImage2D(GL_TEXTURE_2D,0,load.perpixel,
                dib.Info->biWidth,dib.Info->biHeight,
                0,load.format,GL_UNSIGNED_BYTE,dib.bits);
		}
		printf("After delete\n");
		getchar();
		delete dib.Info;
	}
	
	//DeleteObject(hbitmap);
}

void GraphicModel::MyMaterial(GLenum mode,GLfloat *f,GLfloat alpha)
{
	GLfloat d[4];
	d[0]=f[0];
	d[1]=f[1];
	d[2]=f[2];
	d[3]=alpha;
	glDisable(GL_BLEND);
	glMaterialfv (GL_FRONT_AND_BACK,mode,d);
	
}
/*
*  SelectMaterial uses OpenGL commands to define facet colors.
*
*  Returns:
*    Nothing
*/

void GraphicModel::SelectMaterial(int i)
{
	//
	// Define the reflective properties of the 3D Object faces.
	//
	glEnd();
	GLfloat alpha=materials[i].alpha;
	MyMaterial (GL_AMBIENT, materials[i].ambient,alpha);
	MyMaterial (GL_DIFFUSE, materials[i].diffuse,alpha);
	MyMaterial (GL_SPECULAR, materials[i].specular,alpha);
	//MyMaterial (GL_EMISSION, materials[i].emission,alpha);
	glMaterialf (GL_FRONT_AND_BACK,GL_SHININESS,materials[i].phExp);
	
	glEnd();
	glDisable(GL_BLEND);
	if(materials[i].texture>-1)
	{
		glEnable(GL_TEXTURE_2D);
		glBindTexture(GL_TEXTURE_2D,texture_maps[materials[i].texture].id);
	}
	else
		glDisable(GL_TEXTURE_2D);
	glBegin(GL_TRIANGLES);
	
	glBegin(GL_TRIANGLES);
	glDisable(GL_BLEND);
	
}


GLint GraphicModel::genTransformList(Matrix **transformList)
{
	int i;
	int j;
	
	for(i=0;i<numMaterials;i++)
	{
		glBindTexture(GL_TEXTURE_2D,texture_maps[i].id);
	}

	
	Transform T;	

	int curPart;
	for(i=0;i<numSubParts;i++){
		T.setIdentitiy();
		T.addTranslation(-partOrigin[i][0],-partOrigin[i][1],-partOrigin[i][2]);
		T.addTransform(*transformList[i]);
		T.addTranslation(partOrigin[i][0],partOrigin[i][1],partOrigin[i][2]);
		*(transformList[i]) = T.getTransform();
	}

	
	for(i=0;i<numSubParts;i++){
		curPart = i;
		while (partDependancy[curPart]>=0){			
			*(transformList[i]) = *(transformList[partDependancy[curPart]]) * *(transformList[i]);
			curPart = partDependancy[curPart];
		}
	}
	
	float nPos[3];
	float* tFast = new float[numSubParts*16];
	for (i=0;i<numSubParts;i++)
		for (int j=0;j<4;j++)
			for (int k=0;k<4;k++)
				tFast[k+j*4+i*16] = (*(transformList[i]))(j,k);
	
	GLint lid=glGenLists(1);
	int mcount=0;
	int mindex=0;

	int tindex;
	glNewList(lid, GL_COMPILE);	
	
    glBegin (GL_TRIANGLES);
	for(i=0;i<numFace_indices;i++)
	{
		if(!mcount)
		{
			SelectMaterial(material_ref[mindex][0]);
			
			mcount=material_ref[mindex][1];
			mindex++;
		}
		mcount--;
		for(j=0;j<3;j++)
        {
			int vi=face_indicies[i][j];
			int ni=face_indicies[i][j+3];//Normal index
			int ti=face_indicies[i][j+6];//Texture index
			glNormal3f (normals[ni][0],normals[ni][1],normals[ni][2]);
			glTexCoord2f(textures[ti][0],textures[ti][1]);		   
			tindex = vertPartList[vi]*16;
			if (tindex != -16){
				nPos[0] = vertices[vi][0] * tFast[tindex] +
						  vertices[vi][1] * tFast[tindex+1] +
						  vertices[vi][2] * tFast[tindex+2] +
						  tFast[tindex+3];
				nPos[1] = vertices[vi][0] * tFast[tindex+4] +
						  vertices[vi][1] * tFast[tindex+5] +
						  vertices[vi][2] * tFast[tindex+6] +
						  tFast[tindex+7];
				nPos[2] = vertices[vi][0] * tFast[tindex+8] +
						  vertices[vi][1] * tFast[tindex+9] +
						  vertices[vi][2] * tFast[tindex+10] +
						  tFast[tindex+11];
				glVertex3f (nPos[0], nPos[1], nPos[2]);
			}
			else		   
				glVertex3f (vertices[vi][0],vertices[vi][1],vertices[vi][2]);
        }
	}
    glEnd ();
	delete tFast;
	
	glEndList();
	return lid;
}


GLint GraphicModel::genTransformList()
{
	int i;
	int j;
	
	for(i=0;i<numMaterials;i++)
	{
		glBindTexture(GL_TEXTURE_2D,texture_maps[i].id);
	}
	
	
	GLint lid=glGenLists(1);
	int mcount=0;
	int mindex=0;
	glNewList(lid, GL_COMPILE);
	Matrix vert(4,1);
	
	
    glBegin (GL_TRIANGLES);
	for(i=0;i<numFace_indices;i++)
	{
		if(!mcount)
		{
			SelectMaterial(material_ref[mindex][0]);
			
			mcount=material_ref[mindex][1];
			mindex++;
		}
		mcount--;
		for(j=0;j<3;j++)
        {
			int vi=face_indicies[i][j];
			int ni=face_indicies[i][j+3];//Normal index
			int ti=face_indicies[i][j+6];//Texture index
			glNormal3f (normals[ni][0],normals[ni][1],normals[ni][2]);
			glTexCoord2f(textures[ti][0],textures[ti][1]);		   
			if (numSubParts){
				if (vertPartList[vi] != -1){
					vert(0,0) = vertices[vi][0]+0.2f;
					vert(1,0) = vertices[vi][1]+0.0f;
					vert(2,0) = vertices[vi][2];
					vert(3,0) = 1;
					glVertex3f (vert(0,0), vert(1,0), vert(2,0));
				}
				else		   
					glVertex3f (vertices[vi][0],vertices[vi][1],vertices[vi][2]);
			}
			else
				glVertex3f (vertices[vi][0],vertices[vi][1],vertices[vi][2]);
        }
	}
    glEnd ();
	
	
	glEndList();
	return lid;
}

GLint GraphicModel::Gen3DObjectList()
{
	int i;
	int j;
	
	char filename[256];
	strcpy(filename, PATHSTR);
	int pathstrlen = strlen(filename);
	
	for(i=0;i<numMaterials;i++)
	{
		GLuint texture_name;
		glGenTextures(1,&texture_name);
		texture_maps[i].id=texture_name;
		glBindTexture(GL_TEXTURE_2D,texture_name);
		strcpy(&filename[pathstrlen], texture_maps[i].name);
		LoadTexture(filename);
	}
	
	GLint lid=glGenLists(1);
	int mcount=0;
	int mindex=0;
	glNewList(lid, GL_COMPILE);
	
	
	
    glBegin (GL_TRIANGLES);
	for(i=0;i<numFace_indices;i++)
	{
		if(!mcount)
		{
			SelectMaterial(material_ref[mindex][0]);
			
			mcount=material_ref[mindex][1];
			mindex++;
		}
		mcount--;
		for(j=0;j<3;j++)
        {
			int vi=face_indicies[i][j];
			int ni=face_indicies[i][j+3];//Normal index
			int ti=face_indicies[i][j+6];//Texture index
			glNormal3f (normals[ni][0],normals[ni][1],normals[ni][2]);
			glTexCoord2f(textures[ti][0],textures[ti][1]);		   
			glVertex3f (vertices[vi][0],vertices[vi][1],vertices[vi][2]);
        }
	}
    glEnd ();
	
	
	glEndList();
	return lid;
};

GraphicModel::GraphicModel(){
	numSubParts = 0;
	numActions = 0;
	minCorner[0] = 1e5;
	minCorner[1] = 1e5;
	minCorner[2] = 1e5;
	maxCorner[0] = -1e5;
	maxCorner[1] = -1e5;
	maxCorner[2] = -1e5;
}


GLint GraphicModel::generateList(){	
	return list;
}

GLint GraphicModel::getDisplayList(){
	return list;	
}

GLint GraphicModel::getDisplayList(int action, float time){
	float itime = time;
	int frame;
	int oversize = (int)(time/actionFinishTime[action]);
	time -= (actionFinishTime[action] * oversize);

	frame = (int)(actionNumFrames[action] * time/actionFinishTime[action]);
	if (frame == actionNumFrames[action]) frame = 0;

	// printf ("action %d itime %f time %f frame %d listnum %d\n",action,itime,time,frame,actionLists[action][frame]);
	
	return actionLists[action][frame];
}

GLint GraphicModel::getDisplayList(int action, int frame){
	//printf ("action %d frame %d listnum %d\n",action,frame,actionLists[action][frame]);

	return actionLists[action][frame];
}

GraphicModel::~GraphicModel(){
	
	delete materials;
	for (int i=0;i<numMaterials; i++){
		glDeleteTextures (1,&(texture_maps[i].id));
	}
	delete texture_maps;
	delete []face_indicies;
	delete []vertices;
	delete []normals;
	delete []textures;
	delete []material_ref;
	glDeleteLists (list, 1);

	if (numSubParts) {
		delete vertPartList;
		delete []partOrigin;
		delete partDependancy;
		for (int i=0;i<numSubParts;i++)
			delete partBox[i];
		delete partBox;
	}
	if (numActions) {
		for (int i=0; i<numActions; i++){
			for (int j=0; j<actionNumFrames[i]; j++){
				glDeleteLists (actionLists[i][j], 1);
				for (int k=0; k<numSubParts; k++)
					delete actionPartBox[i][j][k];
				delete actionPartBox[i][j];
			}
			delete actionLists[i];
			delete actionPartBox[i];
		}
		delete actionLists;
		delete []actionNames;
		delete actionNumFrames;
		delete actionFinishTime;
		delete actionPartBox;
	}
}

bool GraphicModel::readFile(const char * path,const char *name){
	
	FILE             *fp;
	int buff[6];
	char fileName[256];
	strcpy(fileName,path);
	strcat(fileName,name);
	if ((fp = fopen(fileName, "rb")) == NULL)
        return (false);
	if (fread(&buff, sizeof(int), 6, fp) < 1){
		return false;
	}
	numMaterials=buff[0];
	numFace_indices=buff[1];
	numVertices=buff[2];
	numNormals=buff[3];
	numTextures=buff[4];
	numMaterial_refs=buff[5];
	
	//materials=(sample_MATERIAL*)malloc(sizeof(sample_MATERIAL)*numMaterials);
	materials= new sample_MATERIAL [numMaterials];
	if (fread(&materials[0], sizeof(sample_MATERIAL), numMaterials, fp) < 1){
		delete materials;
		return false;
	}
	
	//texture_maps=(sample_TEXTURE *)malloc(sizeof(sample_TEXTURE)*numMaterials);
	texture_maps= new sample_TEXTURE [numMaterials];
	if(fread(&texture_maps[0],sizeof(sample_TEXTURE),numMaterials,fp)<1){
		delete materials;
		delete texture_maps;
		return false;
	}
	
	face_indicies=new short[numFace_indices][9];
	if(fread(&face_indicies[0][0],sizeof(short),numFace_indices*9,fp)<1){
		delete materials;
		delete texture_maps;
		delete []face_indicies;
		return false;
	}
	
	vertices=new GLfloat [numVertices][3];
	if(fread(&vertices[0][0],sizeof(GLfloat),numVertices*3,fp)<1){
		delete materials;
		delete texture_maps;
		delete []face_indicies;
		delete []vertices;
		return false;
		
	}

	normals=new GLfloat [numNormals][3];
	if(fread(&normals[0][0],sizeof(GLfloat),numNormals*3,fp)<1){
		delete materials;
		delete texture_maps;
		delete []face_indicies;
		delete []vertices;
		delete []normals;
		return false;
		
	}
	
	textures=new GLfloat[numTextures][2];
	if(fread(&textures[0][0],sizeof(GLfloat),numTextures*2,fp)<1){
		delete materials;
		delete texture_maps;
		delete []face_indicies;
		delete []vertices;
		delete []normals;
		delete []textures;
		return false;
		
	}
	
	material_ref=new int[numMaterial_refs][2];
	if(fread(&material_ref[0][0],sizeof(int),numMaterial_refs*2,fp)<1){
		delete materials;
		delete texture_maps;
		delete []face_indicies;
		delete []vertices;
		delete []normals;
		delete []textures;
		delete []material_ref;
		return false;
		
	}
	if(fclose (fp)!=0){
		return false;
	}
	strcpy(typeName,name);

	list=Gen3DObjectList();

	// load parts from .pr file	
	char fileNameParts[256];
	strcpy(fileNameParts,path);
	strcat(fileNameParts,name);
	int fileNamePartsLen = strlen(fileNameParts);
	fileNameParts[fileNamePartsLen-2] = 'p'; fileNameParts[fileNamePartsLen-1] = 'r';
	
	if ((fp = fopen(fileNameParts, "r")) == NULL)
		numSubParts = 0;
	else {        
		float rotValues[3];
		float boxCoords[6];

		printf ("parsing parts file %s..\n", fileNameParts);
		fscanf (fp, "%d", &numSubParts);		

		vertPartList = new int[numVertices];
		partDependancy = new int[numSubParts];
		partOrigin = new float[numSubParts][3];
		partNames = new char*[numSubParts];
		partBox = new Box*[numSubParts];

		for (int j=0; j<numVertices; j++)
			vertPartList[j] = -1;
		
		
		for (int i=0; i<numSubParts; i++){
			int j;
			partNames[i] = new char[PART_NAME_LENGTH];
			fscanf (fp, "%s", partNames[i]);
			//printf ("%s\n", partNames[i]);
			fscanf (fp, "%d", &partDependancy[i]);
			partDependancy[i]--;  // convert to -1 for none, or correcnt dependancy

			for (j=0; j<3; j++)
				fscanf (fp, "%f", &(partOrigin[i][j]));			
			for (j=0; j<3; j++)
				fscanf (fp, "%f", &rotValues[j]);
			for (j=0; j<6; j++)
				fscanf (fp, "%f", &boxCoords[j]);			

			partBox[i] = new Box;
			*(partBox[i]) = Box(boxCoords);
			partBox[i]->rotateX(rotValues[0]);			
			partBox[i]->rotateY(rotValues[1]);
			partBox[i]->rotateZ(rotValues[2]);						

			for (j=0; j<numVertices; j++){
				//printf ("vertex %d %f %f %f\n",j,vertices[j][0],vertices[j][1],vertices[j][2]);
				if (partBox[i]->collide(vertices[j][0],vertices[j][1],vertices[j][2])){
					vertPartList[j] = i;
					//printf ("vertex %d in part %d\n", j,i);
				}
			}			
		}
		
		fclose (fp);
	}

	// process animation sequences from .an file
	
	if (numSubParts){
		char fileNameAnim[256];
		strcpy(fileNameAnim,path);
		strcat(fileNameAnim,name);
		int fileNameAnimLen = strlen(fileNameAnim);
		fileNameAnim[fileNameAnimLen-2] = 'a'; fileNameAnim[fileNameAnimLen-1] = 'n';
				
		if ((fp = fopen(fileNameAnim, "r")) == NULL)
			numActions = 0;
		else {
			printf ("parsing animation file %s..\n", fileNameAnim);
			fscanf (fp, "%d", &numActions);
			
			actionNames = new char*[numActions];
			actionNumFrames = new int[numActions];
			actionFinishTime = new float[numActions];
			actionLists = new GLint*[numActions];
			actionPartBox = new Box***[numActions];
			
			// parse file
			for (int i=0; i<numActions; i++){
				actionNames[i] = new char[ACTION_NAME_LENGTH];
				fscanf ( fp, "%s", actionNames[i]);
				fscanf ( fp, "%d", &(actionNumFrames[i]));
				fscanf ( fp, "%d",  &(NUM_KEY_FRAMES[i]));

				//printf ( "name %s\n", actionNames[i]);
				//printf ( "num anm frames %d\n", actionNumFrames[i]);
				//printf ( "num key frames %d\n", NUM_KEY_FRAMES[i]);

				for (int j=0; j<NUM_KEY_FRAMES[i]; j++){
					fscanf ( fp, "%f", &(KEY_FRAME_TIMES[i][j]));
					//printf ("time %f\n", KEY_FRAME_TIMES[i][j]);					
					for (int k=0;k<numSubParts; k++){
						for (int l=0;l<6;l++){
							fscanf ( fp, "%f", &(KEY_FRAME_TRANS[i][j][k][l]));
							//printf ( "%f ", KEY_FRAME_TRANS[i][j][k][l]);
						}
						//printf ("\n");
					}
					
				}
			}			
			
			fclose (fp);		
			
			Transform curT;
			Box curPartBox;
			Matrix **tList = new Matrix*[numSubParts];			
			for (i=0; i<numSubParts; i++)
				tList[i] = new Matrix(4,4);

			float curTime = 0;
			float deltaTime = 0;
			float curValue[6];
			int keyFrame;
			float alpha;
			//finishTime;
			
			
			for (i=0; i<numActions; i++){
				printf ("building action %s...\n", actionNames[i]);				
				actionLists[i] = new GLint[actionNumFrames[i]];
				actionPartBox[i] = new Box**[actionNumFrames[i]];
				actionFinishTime[i] = KEY_FRAME_TIMES[i][NUM_KEY_FRAMES[i]-1];
				deltaTime = actionFinishTime[i]/(float)(actionNumFrames[i]); 
				curTime = 0;
				for (int j=0; j<actionNumFrames[i]; j++){
					actionPartBox[i][j] = new Box*[numSubParts];
					keyFrame = -1;
					for (int k=0; k<NUM_KEY_FRAMES[i]-1;k++){
						if ((curTime >= KEY_FRAME_TIMES[i][k]) && (curTime <= KEY_FRAME_TIMES[i][k+1])){
							keyFrame = k;
							alpha = (curTime-KEY_FRAME_TIMES[i][k])/(KEY_FRAME_TIMES[i][k+1]-KEY_FRAME_TIMES[i][k]);
						}
					}
					if (keyFrame < 0) keyFrame = 0;
					for (k=0; k<numSubParts; k++){
						curT.setIdentitiy();
						for (int l=0;l<6;l++){
							curValue[l] = KEY_FRAME_TRANS[i][keyFrame][k][l] +
								alpha* (KEY_FRAME_TRANS[i][keyFrame+1][k][l] - KEY_FRAME_TRANS[i][keyFrame][k][l]);
						}
						curT.addRotationX(curValue[0]);
						curT.addRotationY(curValue[1]);
						curT.addRotationZ(curValue[2]);
						curT.addTranslation(curValue[3],curValue[4],curValue[5]);
						curPartBox = *(partBox[k]);
						curPartBox.transform(curT);
						actionPartBox[i][j][k] = new Box;
						*(actionPartBox[i][j][k]) = curPartBox;
						*(tList[k]) = curT.getTransform();						
					}
					actionLists[i][j] = genTransformList(tList);
					curTime += deltaTime;
				}
			}

			for (i=0;i<numSubParts;i++)
				delete tList[i];
			
			delete tList;
		}
	}
	
	computeBoundingBox();
	return true;
	
	
	
}

const char * GraphicModel::getTypeName(){
	return typeName;
}

void GraphicModel::getUnscaledBoundingBox(vec3 &corner1,vec3 &corner2){
	corner1=minCorner;
	corner2=maxCorner;
}
void GraphicModel::computeBoundingBox(){
	for(int i=0;i<numVertices;i++){
		
		for(int j=0;j<3;j++){
			if(maxCorner[j]<vertices[i][j]){
				maxCorner[j]=vertices[i][j];
			}
			if(minCorner[j]>vertices[i][j]){
				minCorner[j]=vertices[i][j];
			}
			
			
		}
	}
}

void GraphicModel::getBoxCorners(float *f)
{	
	f[0] = minCorner[0];
	f[1] = minCorner[1];
	f[2] = minCorner[2];
	f[3] = maxCorner[0];
	f[4] = maxCorner[1];
	f[5] = maxCorner[2];
}
