/* Aftershock 3D rendering engine
 * Copyright (C) 1999 Stephen C. Taylor
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#ifndef __BSP_H__
#define __BSP_H__


// shader contents
#define	CONTENTS_SOLID			1		// an eye is never valid in a solid
#define	CONTENTS_LAVA			8
#define	CONTENTS_SLIME			16
#define	CONTENTS_WATER			32
#define	CONTENTS_FOG			64
#define	CONTENTS_AREAPORTAL		0x8000
#define	CONTENTS_PLAYERCLIP		0x10000
#define	CONTENTS_MONSTERCLIP	0x20000
//bot specific contents types
#define	CONTENTS_TELEPORTER		0x40000
#define	CONTENTS_JUMPPAD		0x80000
#define CONTENTS_CLUSTERPORTAL	0x100000
#define CONTENTS_DONOTENTER		0x200000
#define	CONTENTS_ORIGIN			0x1000000	// removed before bsping an entity
#define	CONTENTS_BODY			0x2000000	// should never be on a brush, only in game
#define	CONTENTS_CORPSE			0x4000000
#define	CONTENTS_DETAIL			0x8000000	// brushes not used for the bsp
#define	CONTENTS_STRUCTURAL		0x10000000	// brushes used for the bsp
#define	CONTENTS_TRANSLUCENT	0x20000000	// don't consume surface fragments inside
#define	CONTENTS_TRIGGER		0x40000000
#define	CONTENTS_NODROP			0x80000000	// don't leave bodies or items (death fog, lava)

// surface flags
#define	SURF_NODAMAGE			0x1		// never give falling damage
#define	SURF_SLICK				0x2		// effects game physics
#define	SURF_SKY				0x4		// lighting from environment map
#define	SURF_LADDER				0x8
#define	SURF_NOIMPACT			0x10	// don't make missile explosions
#define	SURF_NOMARKS			0x20	// don't leave missile marks
#define	SURF_FLESH				0x40	// make flesh sounds and effects
#define	SURF_NODRAW				0x80	// don't generate a drawsurface at all
#define	SURF_HINT				0x100	// make a primary bsp splitter
#define	SURF_SKIP				0x200	// completely ignore, allowing non-closed brushes
#define	SURF_NOLIGHTMAP			0x400	// surface doesn't need a lightmap
#define	SURF_POINTLIGHT			0x800	// generate lighting info at vertexes
#define	SURF_METALSTEPS			0x1000	// clanking footsteps
#define	SURF_NOSTEPS			0x2000	// no footstep sounds
#define	SURF_NONSOLID			0x4000	// don't collide against curves with this set
#define SURF_LIGHTFILTER		0x8000	// act as a light filter during q3map -light
#define	SURF_ALPHASHADOW		0x10000	// do per-pixel light shadow casting in q3map
#define	SURF_NODLIGHT			0x20000	// don't dlight even if solid (solid lava, skies)
#define SURF_NOPICMIP			0x40000
#define SURF_NOMIPMAP			0x80000

enum
{
    FACETYPE_NORMAL   = 1,
    FACETYPE_MESH     = 2,
    FACETYPE_TRISURF  = 3,
    FACETYPE_FLARE    = 4
};

typedef int bbox_t[6];        /* Integer bounding box (mins, maxs)*/
typedef float bboxf_t[6];     /* Float bounding box */
typedef float texcoord_t[2];  /* Texture s&t coordinates */
typedef byte_t colour_t[4];   /* RGBA */

/* Model 0 is the main map, others are doors, gates, buttons, etc. */
typedef struct
{
    bboxf_t bbox;
    int firstface, numfaces;
    int firstbrush, numbrushes;    
} model_t;

/* Face planes */
typedef struct
{
    vec3_t vec;    /* Normal to plane */
    float offset;  /* Distance to plane along normal */
} plane_t;

/* Nodes in the BSP tree */
typedef struct
{
    int plane;        /* Dividing plane */
    int children[2];  /* Left and right node.  Negatives are leafs */
    bbox_t bbox;
} node_t;

/* Leafs in BSP tree */
typedef struct
{
    int cluster;    /* Visibility cluster number */
    int area;       /* ? */
    bbox_t bbox;
    int firstface, numfaces;
    int firstbrush, numbrushes;
} leaf_t;

/* Faces (or surfaces) */
typedef struct
{
    int shader;      /* Shader reference */
    int unknown1[1];
    int facetype;   /* FACETYPE enum */
    int firstvert, numverts;
    int firstelem, numelems;
    int lm_texnum;    /* lightmap info */
    int lm_offset[2];
    int lm_size[2];
    vec3_t v_orig;   /* FACETYPE_NORMAL only */
    bboxf_t bbox;    /* FACETYPE_MESH only */
    vec3_t v_norm;   /* FACETYPE_NORMAL only */
    int mesh_cp[2];  /* mesh control point dimensions */
} face_t;

/* Shader references (indexed from faces) */
typedef struct
{
    char name[64];
	int flags;
	int contents;
} shaderref_t;

/* Brush side */
typedef struct 
{
	int plane;
	int shader;
} brushside_t;

/* Brush information*/
typedef struct 
{
	int firstside;
	int numsides;
	int shader;
} brush_t;

/* Vertex info */
typedef struct
{
    vec3_t v_point;     /* World coords */
    texcoord_t tex_st;  /* Texture coords */
    texcoord_t lm_st;   /* Lightmap texture coords */
    vec3_t v_norm;      /* Normal */
    colour_t colour;    /* Colour used for vertex lighting ? */
} vertex_t;

/* Potentially visible set (PVS) data */
typedef struct
{
    int numclusters;   /* Number of PVS clusters */
    int rowsize;
    byte_t data[1];
} visibility_t;


/* PVS test macro:  PVS table is a packed single bit array, rowsize
   bytes times numclusters rows */
#define BSP_TESTVIS(from,to) (*(r_visibility->data + (from)*r_visibility->rowsize + ((to)>>3)) & (1 << ((to) & 7)))

#endif /*__BSP_H__*/
