#include "plane.h"

Plane::Plane ()
{
	dist = 0;
	normal[0] = 0;
	normal[1] = 0;
	normal[2] = 0;
}

Plane::Plane (Matrix M)
{
	dist = 0;
	normal[0] = M (0,0);
	normal[1] = M (1,0);
	normal[2] = M (2,0);
}

Plane::Plane (Matrix M, float d)
{
	dist = d;
	normal[0] = M (0,0);
	normal[1] = M (1,0);
	normal[2] = M (2,0);
}
	

Plane::Plane (vec3_t n, float d)
{
	dist = d;
	normal[0] = n[0];
	normal[1] = n[1];
	normal[2] = n[2];
}

Plane::Plane (plane_t p)
{
	dist = p.offset;

	normal[0] = p.vec[0];
	normal[1] = p.vec[1];
	normal[2] = p.vec[2];
}


void Plane::translate ( float dx, float dy, float dz )
{
	dist += dx * normal[0] + dy * normal[1] + dx * normal[2];
}

void Plane::translate ( float d )
{
	dist += d;
}

void Plane::rotateX (float alpha)
{
	mat3_t rot;
	for (int i=0;i<3;i++){
		for (int j=0;j<3;j++)
			rot[i+3*j] = (i==j);
	}
	rot [1+3*1] = cos(alpha);
	rot [1+3*2] =-sin(alpha);
	rot [2+3*1] =-rot [1+3*2];
	rot [2+3*2] = rot [1+3*1];

	mat3_vmult(rot,normal,normal);
}

void Plane::rotateY (float alpha)
{
	mat3_t rot;
	for (int i=0;i<3;i++){
		for (int j=0;j<3;j++)
			rot[i+3*j] = (i==j);
	}
	rot [0+3*0] = cos(alpha);
	rot [0+3*2] =-sin(alpha);
	rot [2+3*0] =-rot [0+3*2];
	rot [2+3*2] = rot [0+3*0];

	mat3_vmult(rot,normal,normal);
}

void Plane::rotateZ (float alpha)
{
	mat3_t rot;
	for (int i=0;i<3;i++){
		for (int j=0;j<3;j++)
			rot[i+3*j] = (i==j);
	}
	rot [0+3*0] = cos(alpha);
	rot [0+3*1] =-sin(alpha);
	rot [1+3*0] =-rot [0+3*1];
	rot [1+3*1] = rot [0+3*0];

	mat3_vmult(rot,normal,normal);
}
	
void Plane::setNormal(Matrix M)
{
	normal[0] = M (0,0);
	normal[1] = M (1,0);
	normal[2] = M (2,0);
}

void Plane::setNormal(vec3_t v)
{
	normal[0] = v[0];
	normal[1] = v[1];
	normal[2] = v[2];
}

void Plane::setNormal(float dx, float dy, float dz)
{
	normal[0] = dx;
	normal[1] = dy;
	normal[2] = dz;
}

void Plane::setDist ( float d )
{
	dist = d;
}

float Plane::distance (float x, float y, float z)
{
	return ( normal[0] * x + normal[1] * y + normal[2] * z - dist );
}

float Plane::distance (Point p)
{
	vec3_t v;
	p.getPosition(v);
	return ( normal[0] * v[0] + normal[1] * v[1] + normal[2] * v[2] - dist );
}

float Plane::distance (Sphere s)
{
	vec3_t v;
	s.getPosition(v);
	return ( normal[0] * v[0] + normal[1] * v[1] + normal[2] * v[2] - dist );
}

void Plane::print ()
{
	printf ("normal %4.4f %4.4f %4.4f  dist %4.4f\n", normal[0], normal[1], normal[2], dist);
}

Matrix Plane::getNormal()
{
	Matrix m(3,1);
	m(0,0) = normal[0];
	m(1,0) = normal[1];
	m(2,0) = normal[2];
	return m;
}

void Plane::getNormal(vec3_t v)
{
	v[0] = normal[0];
	v[1] = normal[1];
	v[2] = normal[2];
}

bool Plane::collide (float x, float y, float z, float dx, float dy, float dz, float ln)
{
	float d1 = distance ( x,y,z);
	float d2 = distance ( x+dx*ln,y+dy*ln,z+dz*ln);
	
	if ((d1<0 && d2>0) || (d1>0 && d2<0)) return true;
	return false;
}

float Plane::collideRatio (float x, float y, float z, float dx, float dy, float dz, float ln)
{
	float d1 = distance ( x,y,z);
	float d2 = distance ( x+dx*ln,y+dy*ln,z+dz*ln);

	if (d1 != d2) return d1/(d1-d2);
	else return -1;
}

