// Player.cpp: implementation of the Player class.
//
//////////////////////////////////////////////////////////////////////

#include "Player.h"
#include <stdio.h>
#include <math.h>
#include "XPModelList.h"
#include "nl.h"
#include "NetworkManager.h"
#include "SoundManager.h"
#include "SoundObjectList.h"
#include "globals.h"
extern XPModelList *xpModelList;
extern XPModelList *projectileList;
extern NetworkManager *networkManager;
#ifdef SOUND_AV
extern SoundManager *soundManager;
#endif
extern SoundObjectList *soundList;

#ifdef WIN32
#include <winsock2.h>
#include <sys/timeb.h>
#else
#include <sys/time.h>
#endif


#include "vec.h"
#include "Box.h"
#include "matrix.h"
//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

static const float PIBY2=3.14159/2;
//static char buff[256];
extern unsigned char packet[MAXPACKETLEN];

Player::Player(){
	
	eyeOrientation[0]=0;
	eyeOrientation[1]=-1;
	eyeOrientation[2]=0;
	orientation[0]=0;
	orientation[1]=0;
	orientation[2]=-90;
	elevation=0;
	setVelocity(vec3(0,-1,0));
	motionState=NOTMOVING;	
	playerID=255;
	initWeapons();	
}


Player::Player(int pID){
	eyeOrientation[0]=0;
	eyeOrientation[1]=-1;
	eyeOrientation[2]=0;
	orientation[0]=0;
	orientation[1]=0;
	orientation[2]=-90;
	elevation=0;
	setVelocity(vec3(0,-1,0));
	motionState=NOTMOVING;	
	playerID=pID;
	initWeapons();
}

Player::~Player()
{

}

void Player::initWeapons()
{
	//pistol
	weapons[0].modelTypeIndex = 11;
	weapons[0].projectileTypeIndex=8;
	weapons[0].ammoTypeIndex=5;
	weapons[0].weaponType=0;
	weapons[0].maxAmmo=40;
	weapons[0].currentAmmo=10;
	weapons[0].clipCount=10;
	weapons[0].clipSize =10;
	weapons[0].clipDelay=1000;
	weapons[0].timeDelay= 250;
	weapons[0].lastTimeFired=0;
	weapons[0].healthDamage=5;
	weapons[0].available=true;
	weapons[0].speed=2000.0;

	//machine gun
	weapons[1].modelTypeIndex = 12;
	weapons[1].projectileTypeIndex=9;
	weapons[1].ammoTypeIndex=6;
	weapons[1].weaponType=1;
	weapons[1].maxAmmo=200;
	weapons[1].currentAmmo=0;
	weapons[1].clipCount=40;
	weapons[1].clipSize =40;
	weapons[1].clipDelay=2000;
	weapons[1].timeDelay=  50;
	weapons[1].lastTimeFired=0;
	weapons[1].healthDamage=5;
	weapons[1].available=true;
	weapons[1].speed=2000.0;

	//rocket launcher
	weapons[2].modelTypeIndex = 13;
	weapons[2].projectileTypeIndex=10;
	weapons[2].ammoTypeIndex=7;
	weapons[2].weaponType=2;
	weapons[2].maxAmmo=20;
	weapons[2].currentAmmo=8;
	weapons[2].clipCount=0;
	weapons[2].clipSize =4;
	weapons[2].clipDelay=4000;
	weapons[2].timeDelay=1000;
	weapons[2].lastTimeFired=0;
	weapons[2].healthDamage=50;
	weapons[2].available=false;
	weapons[2].speed=400.0;

	haveGlasses = 0;
	haveMatrix = 0;
	matrixStartTime = 0;
	activeWeapon=0;
	health = 100;
}


void Player::goForward(float increment)
{
	
	moveAsOriented(increment);

}

void Player::goBack(float increment)
{
	goForward(-increment);
}

void Player::goLeft(float increment)
{
	vec3 direction;
	vec3 temp=getEyeOrientation();

	direction[2]=-temp[1];
	direction[0]=temp[2];
	direction[1]=temp[1];
	direction*=(increment);
	moveInDirection(direction);
}

void Player::goRight(float increment)
{
	goLeft(-increment);
}


void Player::goUp(float increment)
{
	vec3 direction(0,1,0);
	direction*=increment;
	moveInDirection(direction);

	
}

void Player::goDown(float increment)
{
	goUp(-increment);

}

void Player::setEyeOrientation(vec3 eye)
{	
	eye.normalize();
	eyeOrientation=eye;
}

vec3 Player::getEyeOrientation()
{
	vec3_t orient;
	vec_point(orient,orientation[2],elevation);
	vec3 or(orient[0],orient[1],orient[2]);
	return or.normalize();
}

void Player::turnLeft(float increment){


	orientation[2]+=increment;
	if(orientation[2]>360){
		orientation[2]-=360;
	}
	if(orientation[2]<0){
		orientation[2]+=360;
	}
	

}
void Player::turnRight(float increment)
{
	turnLeft(-increment);
}

void Player::lookUp(float increment)
{

	if(((elevation+increment)<90)&&((elevation+increment)>-90)){
		elevation+=increment;
	}
	

}

void Player::lookDown(float increment)
{
	lookUp(-increment);
	
}

void Player::setMotionState(MotionState state){
	motionState=state;
}

int Player::getMotionState(){
	return motionState;
}

void Player::updateState(float delta){
	switch(motionState){

	case	NOTMOVING:
			break;
				
	case	GOFORWARD:
			goForward(delta);
			break;

	case GOBACK:
			goBack(delta);
			break;

	case STRAFELEFT:
			goLeft(-delta);
			break;

	case STRAFERIGHT:
			goRight(-delta);
			break;

	case TURNRIGHT:
			turnRight(-0.1*delta);
			break;

	case TURNLEFT:
			turnLeft(-0.1*delta);
			break;
	case GOUP:
			goUp(-delta);
			break;
	case GODOWN:
			goDown(-delta);
			break;


	}


}
bool Player::fire(){

	unsigned long mcurtime;

	Weapon *weapon=&weapons[activeWeapon];

#ifdef WIN32
	struct timeb tmptime;
	ftime(&tmptime);
	mcurtime = tmptime.time*1000 + tmptime.millitm;
#else
	struct timeval tmptime;
	struct timezone tz;
	gettimeofday(&tmptime, &tz);
	mcurtime = tmptime.tv_sec*1000 + tmptime.tv_usec/1000;
#endif
	
	if(mcurtime<weapon->lastTimeFired || (mcurtime-weapon->lastTimeFired)<weapon->timeDelay){
		return false;
	}
	if(weapon->clipCount==0){		
		if (weapon->currentAmmo > weapon->clipSize ){
			weapon->currentAmmo -= weapon->clipSize;
			weapon->clipCount = weapon->clipSize;
		}
		else {
			weapon->clipCount = weapon->currentAmmo;
			weapon->currentAmmo = 0;
		}
		if (weapon->clipCount == 0) return false;
#ifdef SOUND_AV
		soundManager->playSound(soundList->getSound(17),CHANNEL_MEDIUM_PRIORITY,1,0);
#endif		
		weapon->lastTimeFired += weapon->clipDelay;
		return false;
	}
	weapon->clipCount--;
	weapon->lastTimeFired=mcurtime;
	unsigned char pType=weapons[activeWeapon].projectileTypeIndex;
	XPModel *projectile=new XPModel(pType);

	projectile->setVelocity(weapon->speed*getEyeOrientation());
	projectile->setOrientation(vec3(elevation,0,getOrientation()[2]));
	projectile->setPosition(getPosition());
	
	myProjectiles.addModel(projectile);	
	
	int cursor=0;
	unsigned char charTemp;
	writeByte(packet,cursor,PROJECTILECREATED);
	writeByte(packet,cursor,playerID);
	
	projectile->setOwner(playerID);
	charTemp=myProjectiles.getLength()-1;
	writeByte(packet,cursor,charTemp);
	projectile->setID(charTemp);

	//ModelType
	writeByte(packet,cursor,projectile->getModelTypeIndex());

	
	//Position
	vec3 vec;
	vec=projectile->getPosition();

	writeFloat(packet,cursor,vec[0]);
	writeFloat(packet,cursor,vec[1]);
	writeFloat(packet,cursor,vec[2]);

	//Orientation
	vec=projectile->getOrientation();
	writeFloat(packet,cursor,vec[0]);
	writeFloat(packet,cursor,vec[1]);
	writeFloat(packet,cursor,vec[2]);

	//Velocity
	vec=projectile->getVelocity();
	writeFloat(packet,cursor,vec[0]);
	writeFloat(packet,cursor,vec[1]);
	writeFloat(packet,cursor,vec[2]);

	//Acceleration
	vec=projectile->getAcceleration();
	writeFloat(packet,cursor,vec[0]);
	writeFloat(packet,cursor,vec[1]);
	writeFloat(packet,cursor,vec[2]);
		
	if(networkManager->sendPacket(packet,cursor) < 0)
		return false; 
	return true;

}
unsigned char Player::getPlayerID(){
	return playerID;
}

void Player::setPlayerID(unsigned char pID){
	playerID=pID;
}

bool Player::sendUpdateToServer(){
	
	int cursor=0;
	unsigned char charTemp;

	//PlayerUpdate command
	writeByte(packet,cursor,PLAYERUPDATE );

	

	//PlayerID
	writeByte(packet,cursor,playerID);
	//printf("My id is %d\n",playerID);
	//MotionState
	charTemp=getMotionState();
	writeByte(packet,cursor,charTemp);
	
	//ModelType
	charTemp=getModelTypeIndex();
	writeByte(packet,cursor,charTemp);

	//Position
	vec3 vec;
	vec=getPosition();

	writeFloat(packet,cursor,vec[0]);
	writeFloat(packet,cursor,vec[1]);
	writeFloat(packet,cursor,vec[2]);

	//Orientation
	vec=getOrientation();
	writeFloat(packet,cursor,vec[0]);
	writeFloat(packet,cursor,vec[1]);
	writeFloat(packet,cursor,vec[2]);

	//Animation info
	writeByte(packet,cursor,action);
	writeByte(packet,cursor,previousAction);
	writeFloat(packet,cursor,animationTime);

	
	if (networkManager->sendPacket(packet,cursor) < 0)
		return false; 
	return true;
}

void Player::sendPickUpRequest(unsigned char modelID){
	int cursor=0;
	writeByte(packet,cursor,PICKUPOBJECTREQUEST);

	writeByte(packet,cursor,playerID);

	writeByte(packet,cursor,modelID);

	networkManager->sendPacket(packet,cursor);
	if(cursor!=PICKUPOBJECTREQUESTLENGTH){
		printf("Error in pickupobject pkt length\n");
	}

}

void Player::removeProjectile(XPModel *projectile){
	if (projectile == NULL) return;
	projectileList->removeModel(projectile);
	int cursor=0;
	writeByte(packet,cursor,REMOVEPROJECTILE);
	writeByte(packet,cursor,playerID);

	writeByte(packet,cursor,projectile->getID());
	networkManager->sendPacket(packet,cursor);
	delete projectile;

}

void Player::updateProjectile(XPModel *projectile){
	int cursor=0;
	unsigned char charTemp;
	writeByte(packet,cursor,PROJECTILEUPDATE);
	writeByte(packet,cursor,playerID);
	charTemp=projectile->getID();
	writeByte(packet,cursor,charTemp);
	

	//ModelType
	writeByte(packet,cursor,projectile->getModelTypeIndex());

	
	//Position
	vec3 vec;
	vec=projectile->getPosition();

	writeFloat(packet,cursor,vec[0]);
	writeFloat(packet,cursor,vec[1]);
	writeFloat(packet,cursor,vec[2]);

	//Orientation
	vec=projectile->getOrientation();
	writeFloat(packet,cursor,vec[0]);
	writeFloat(packet,cursor,vec[1]);
	writeFloat(packet,cursor,vec[2]);

	//Velocity
	vec=projectile->getVelocity();
	writeFloat(packet,cursor,vec[0]);
	writeFloat(packet,cursor,vec[1]);
	writeFloat(packet,cursor,vec[2]);

	//Acceleration
	vec=projectile->getAcceleration();
	writeFloat(packet,cursor,vec[0]);
	writeFloat(packet,cursor,vec[1]);
	writeFloat(packet,cursor,vec[2]);
		
	networkManager->sendPacket(packet,cursor);
}

void Player::removeProjectile(){
	if(myProjectiles.getLength()!=0){
		XPModel *projectile=myProjectiles.getModel(0);
		myProjectiles.removeModel(projectile);
		XPModel *p2=projectileList->getModel(projectile->getOwner(),projectile->getID());
		removeProjectile(p2);
		delete projectile;
		
		
	}
}

void Player::updateProjectile(){
	if(myProjectiles.getLength()!=0){
		XPModel *projectile=myProjectiles.getModel(0);
	
		projectile->setModelTypeIndex(4);
		projectile=projectileList->getModel(projectile->getOwner(),projectile->getID());
		if(projectile!=NULL){
			projectile->setModelTypeIndex(4);
			updateProjectile(projectile);
		}
		else{
			printf("update projectile inconsistency\n");
		}
	}

}

int Player::getHealth(){
	return health;
}

void Player::setHealth(int hl){
	health=hl;
	if(health>100){
		health=100;
	}
	if(health<0){
		health=0;
	}
}

void Player::changeHealth(int increment){
	health+=increment;
	if(health>100){
		health=100;
	}
	if(health<0){
		health=0;
	}
}
