#include "BSPDataModel.h"
#include "mapent.h"


BSPDataModel::BSPDataModel()
{
#ifndef SERVERCODE
	r_lodbias = 0;
	r_gamma = 1.8;
#endif
	calls = 0;
}

#ifndef SERVERCODE

void BSPDataModel::setMapQuality(int lodbias, float gamma)
{
	r_lodbias = lodbias;
	r_gamma = gamma;

}

#endif

BSPDataModel::~BSPDataModel()
{
	free(r_models);
    free(r_shaderrefs);
    free(r_verts);
    free(r_planes);
    free(r_leafs);
    free(r_nodes);
    free(r_faces);
    free(r_lfaces);
    free(r_elems);
    free(r_visibility);

    entity_free();
#ifndef SERVERCODE
	mesh_free_all();
	skybox_free();
#endif
	mapent_freeall();
#ifndef SERVERCODE
	shader_freeall();
	lightmap_freeobjs();
	tex_freeobjs();
#endif
}

int BSPDataModel::load(const char *absPath, const char* relMapPath)
{
	FILE *bsp_file;
	struct stat stat_buf;
	char tmpPath[PATHLEN];

	strcpy(rootPath, absPath);
	strcpy(tmpPath, absPath);
	strcat(tmpPath, relMapPath);
	if ( (bsp_file = fopen(tmpPath, "rb")) == NULL)
		return -1; //Error("Could not open bsp file %s", fname);

	if ( stat(tmpPath, &stat_buf) < 0)
		return -1; //Error("Couldn't determine length of bsp file %s", fname);

    bspdata = (byte_t*)malloc(stat_buf.st_size);

	fread(bspdata, 1, stat_buf.st_size, bsp_file);
    fclose(bsp_file);
    
    bspheader = (struct header*)bspdata;
    
    if (bspheader->id != BSPHEADERID)
		return -1; //Error("Not a bsp file: %s", fname);
    if (bspheader->ver != BSPVERSION)
	 return -1; //Error("Bad bsp file version");

    /* Make additional room for shader refs to be added later */
    r_numshaders = bspheader->lump[SHADERREFS].filelen / sizeof(shaderref_t);
    r_shaderrefs = (shaderref_t*)malloc((r_numshaders + SHADERS_ADD)*sizeof(shaderref_t));
    memcpy(r_shaderrefs, bspdata + bspheader->lump[SHADERREFS].fileofs,
	   r_numshaders * sizeof(shaderref_t));
    r_addshaderstart = r_numshaders;
	
#ifdef SERVERCODE
	READLUMP(PLANESERVER,	planes);
#else
    READLUMP(PLANES,	planes);
#endif
    READLUMP(NODES,		nodes);
    READLUMP(LEAFS,		leafs);
    READLUMP(LFACES,	lfaces);
    READLUMP(MODELS,	models);
    READLUMP(VERTS,		verts);
    READLUMP(ELEMS,		elems);
    READLUMP(FACES,		faces);
	READLUMP(BRUSHES,	brushes);
	READLUMP(BRUSH_SIDES,brushsides);
	READLUMP(BRUSH_LIST,lbrushes);

    r_lightmapsize = readlump(LIGHTMAPS, (void**)&r_lightmapdata, 1);
    (void)readlump(VISIBILITY, (void**)&r_visibility, 1);


    entity_parse(bspheader->lump[ENTITIES].filelen,
		 (char*)(bspdata + bspheader->lump[ENTITIES].fileofs));
    
    free((void *)bspdata);

#ifndef SERVERCODE

	mesh_create_all();
	skybox_create();

#endif

	//Added map entities
	mapent_loadall(this);

#ifndef SERVERCODE
	shader_readall();
	lightmap_loadobjs();
	tex_loadobjs();
#endif

	return 0;
}


int BSPDataModel::readlump(int lump, void** mem, size_t elem)
{
    int len = bspheader->lump[lump].filelen;
    int num = len / elem;
    *mem = malloc(len);

    memcpy(*mem, bspdata + bspheader->lump[lump].fileofs, num * elem);
    return num;
}
#ifndef SERVERCODE

void BSPDataModel::inspect()
{
    printf("Contents of BSP dataModel:\n\n");
    printf("num entities     %d\n", g_numentities);
    printf("num models       %d\n", r_nummodels);
    printf("num shaders      %d\n", r_numshaders);
    printf("num planes       %d\n", r_numplanes);
    printf("num verts        %d\n", r_numverts);
    printf("num vertex elems %d\n", r_numelems);
    printf("num leafs        %d\n", r_numleafs);
    printf("num nodes        %d\n", r_numnodes);
    printf("num faces        %d\n", r_numfaces);    
    printf("num lfaces       %d\n", r_numlfaces);
	printf("num brushes      %d\n", r_numbrushes);
	printf("num brush sides  %d\n", r_numbrushsides);
    printf("vis. clusters    %d\n", r_visibility->numclusters);
    
};    



/* Add shader to the shaderref list (from md3 loads) */
int BSPDataModel::addshaderref(const char *shadername)
{
    int i;

    /* Check for shader already in the list */
    for (i=0; i < r_numshaders; i++)
    {
	if (!strcmp(r_shaderrefs[i].name, shadername))
	    return i;
    }
    
    if (++calls > SHADERS_ADD)
	Error("Too many additional shaders");
        
    strcpy(r_shaderrefs[r_numshaders++].name, shadername);    
    return r_numshaders-1;
}

#endif

//entity methods

int BSPDataModel::getNumEntities(){
	return g_numentities;
}

void BSPDataModel::entity_parse(int buflen, char *buf)
{
    int i, newlines, pair;
    char *c;
    
    /* Save local copy of buf */
    entity_buf = (char*)malloc(buflen);
    memcpy(entity_buf, buf, buflen);

    /* Count entities and pairs */
    newlines = g_numentities = 0;
    for (i = 0; i < buflen; i++)
    {
	if (entity_buf[i] == '{') g_numentities++;
	if (entity_buf[i] == '\n') newlines++;
    }
    numepairs = newlines - (2 * g_numentities);

    /* Alloc structures */
    epairs = (epair_t*)malloc(numepairs * sizeof(epair_t));
    entities = (entity_t*)malloc(g_numentities * sizeof(entity_t));

    c = entity_buf;
    pair = 0;
    for (i = 0; i < g_numentities; i++)
    {
	entities[i].firstpair = pair;
	entities[i].numpairs = 0;
	
	/* Skip to leading quote */
	while (*c != '"') c++;

	while (*c != '}')
	{
	    epairs[pair].key = ++c;
	    while (*c != '"') c++;
	    *c = '\0';
	    c += 3;

	    epairs[pair].val = c;
	    while (*c != '"') c++;
	    *c = '\0';
	    c += 2;
	    pair++;
	    entities[i].numpairs++;
	}
    }
}

void BSPDataModel::entity_free()
{
    free(entities);
    free(epairs);
    free(entity_buf);
}

const char* BSPDataModel::entity_value(int entity, const char *key)
{
    epair_t *pair;
    int i;

    pair = &epairs[entities[entity].firstpair];
    for (i = 0; i < entities[entity].numpairs; i++, pair++)
    {
	if (!strcmp(key, pair->key))
	    return pair->val;
    }
    return "";
}

float BSPDataModel::entity_float(int entity, const char *key)
{
    return atof(entity_value(entity, key));
}

void BSPDataModel::entity_vec3(int entity, const char *key, vec3_t vec)
{
    const char *val;

    val = entity_value(entity, key);
    sscanf(val, "%f %f %f", &vec[0], &vec[1], &vec[2]);
}


#ifndef SERVERCODE

//skybox
void BSPDataModel::skybox_create()
{
    int i;

    /* Alloc space for skybox verts, etc. */
    r_skybox = (skybox_t*)malloc(sizeof(skybox_t));
    r_skybox->points[0] = (vec3_t*)malloc(5 * POINTS_LEN * sizeof(vec3_t));
    r_skybox->tex_st[0] = (texcoord_t*)malloc(5 * POINTS_LEN *
					      sizeof(texcoord_t));
    r_skybox->elems = (uint_t*)malloc(ELEM_LEN * sizeof(uint_t));

    r_skybox->numpoints = POINTS_LEN;
    r_skybox->numelems = ELEM_LEN;
    
    for (i=1; i < 5; i++)
    {
	r_skybox->points[i] = r_skybox->points[i-1] + POINTS_LEN;
	r_skybox->tex_st[i] = r_skybox->tex_st[i-1] + POINTS_LEN;
    }

    gen_box();
    gen_elems();
}
    

void BSPDataModel::skybox_free()
{
    free(r_skybox->points[0]);
    free(r_skybox->tex_st[0]);
    free(r_skybox->elems);
    free(r_skybox);    
}

void BSPDataModel::gen_elems()
{
    int u, v;
    uint_t *e;

    /* Box elems in tristrip order */
    e = r_skybox->elems;
    for (v = 0; v < SIDE_SIZE-1; ++v)
    {
	for (u = 0; u < SIDE_SIZE-1; ++u)
	{
	    *e++ = v * SIDE_SIZE + u;
	    *e++ = (v+1) * SIDE_SIZE + u;
	    *e++ = v * SIDE_SIZE + u + 1;
	    *e++ = v * SIDE_SIZE + u + 1;
	    *e++ = (v+1) * SIDE_SIZE + u;
	    *e++ = (v+1) * SIDE_SIZE + u + 1;	    
	}
    }
}
    
void BSPDataModel::gen_box()
{
    vec3_t orig, drow, dcol;
    float size = 1.0f;
    float step = 0.25f;
    
    /* Top */
    orig[0] = -size;
    orig[1] = size;
    orig[2] = size;
    drow[0] = 0.0;
    drow[1] = -step;
    drow[2] = 0.0;
    dcol[0] = step;
    dcol[1] = 0.0;
    dcol[2] = 0.0;
    gen_box_side(SKYBOX_TOP, orig, drow, dcol);

    /* Front */
    orig[0] = size;
    orig[1] = size;
    orig[2] = size;
    drow[0] = 0.0;
    drow[1] = 0.0;
    drow[2] = -step;
    dcol[0] = -step;
    dcol[1] = 0.0;
    dcol[2] = 0.0;
    gen_box_side(SKYBOX_FRONT, orig, drow, dcol);

    /* Right */
    orig[0] = size;
    orig[1] = -size;
    orig[2] = size;
    drow[0] = 0.0;
    drow[1] = 0.0;
    drow[2] = -step;
    dcol[0] = 0.0;
    dcol[1] = step;
    dcol[2] = 0.0;
    gen_box_side(SKYBOX_RIGHT, orig, drow, dcol);

    /* Back */
    orig[0] = -size;
    orig[1] = -size;
    orig[2] = size;
    drow[0] = 0.0;
    drow[1] = 0.0;
    drow[2] = -step;
    dcol[0] = step;
    dcol[1] = 0.0;
    dcol[2] = 0.0;
    gen_box_side(SKYBOX_BACK, orig, drow, dcol);

    /* Left */
    orig[0] = -size;
    orig[1] = size;
    orig[2] = size;
    drow[0] = 0.0;
    drow[1] = 0.0;
    drow[2] = -step;
    dcol[0] = 0.0;
    dcol[1] = -step;
    dcol[2] = 0.0;
    gen_box_side(SKYBOX_LEFT, orig, drow, dcol);
}

void BSPDataModel::gen_box_side(int side, vec3_t orig, vec3_t drow, vec3_t dcol)
{
    vec3_t pos, w, row, *v;
    texcoord_t *tc;
    float p;
    int r, c;
    float d, b, t;

    /* I don't know exactly what Q3A does for skybox texturing, but this is
     * at least fairly close.  We tile the texture onto the inside of
     * a large sphere, and put the camera near the top of the sphere.
     * We place the box around the camera, and cast rays through the
     * box verts to the sphere to find the texture coordinates.
     */
    
    d = EYE_RAD;     /* Sphere center to camera distance */ 
    b = SPHERE_RAD;  /* Sphere radius */
    
    v = &r_skybox->points[side][0];
    tc = &r_skybox->tex_st[side][0];
    vec_copy(orig, row);
    for (r = 0; r < SIDE_SIZE; ++r)
    {
	vec_copy(row, pos);
	for (c = 0; c < SIDE_SIZE; ++c)
	{
	    /* pos points from eye to vertex on box */
	    vec_copy(pos, (*v));
	    vec_copy(pos, w);

	    /* Normalize pos -> w */
	    p = sqrt(vec_dot(w, w));
	    w[0] /= p;
	    w[1] /= p;
	    w[2] /= p;

	    /* Find distance along w to sphere */
	    t = sqrt(d*d*(w[2]*w[2]-1.0) + b*b) - d*w[2];
	    w[0] *= t;
	    w[1] *= t;

	    /* Use x and y on sphere as s and t */
	    (*tc)[0] = w[0] / (2.0 * SCALE_S);
	    (*tc)[1] = w[1] / (2.0 * SCALE_T);
	    
	    vec_add(pos, dcol, pos);
	    v++;
	    tc++;
	}
	vec_add(row, drow, row);
    }
}


/* end of skybox, begin mesh processing*/

void BSPDataModel::mesh_create_all()
{
    int i;    
    
    /* Count meshes */
    for (r_nummeshes=0; r_nummeshes < r_numfaces; r_nummeshes++)
	if (r_faces[r_nummeshes].facetype != FACETYPE_MESH)
	    break;

    r_meshes = (mesh_t*)malloc(r_nummeshes * sizeof(mesh_t));

    for (i=0; i < r_nummeshes; i++)
    {
	mesh_create(&r_faces[i], &r_meshes[i]);
    }
}

void BSPDataModel::mesh_free_all()
{
    int i;

    for (i=0; i < r_nummeshes; i++)
    {
	free(r_meshes[i].points);
	/* tex_st and lm_st are part of points: don't free */
	free(r_meshes[i].elems);
    }
    free(r_meshes);
}

int BSPDataModel::mesh_find_level(vec3_t *v)
{
    int level;
    vec3_t a, b, dist;

    /* Subdivide on the left until tolerance is reached */
    for (level=0; level < r_maxmeshlevel-1; level++)
    {
	/* Subdivide on the left */
	vec_avg(v[0], v[1], a);
	vec_avg(v[1], v[2], b);
	vec_avg(a, b, v[2]);

	/* Find distance moved */
	vec_sub(v[2], v[1], dist);

	/* Check for tolerance */
	if (vec_dot(dist, dist) < r_subdivisiontol * r_subdivisiontol)
	    break;

	/* Insert new middle vertex */
	vec_copy(a, v[1]);
    }

    return level;
}

void BSPDataModel::mesh_find_size(int *numcp, vec3_t *cp, int *size)
{
    int u, v, found, level;
    float *a, *b;
    vec3_t test[3];
    
    /* Find non-coincident pairs in u direction */
    found = 0;
    for (v=0; v < numcp[1]; v++)
    {
	for (u=0; u < numcp[0]-1; u += 2)
	{
	    a = cp[v * numcp[0] + u];
	    b = cp[v * numcp[0] + u + 2];
	    if (!vec_cmp(a,b))
	    {
		found = 1;
		break;
	    }
	}
	if (found) break;
    }
    if (!found) Error("Bad mesh control points");

    /* Find subdivision level in u */
    vec_copy(a, test[0]);
    vec_copy((a+3), test[1]);
    vec_copy(b, test[2]);
    level = mesh_find_level(test);
    size[0] = (LEVEL_WIDTH(level) - 1) * ((numcp[0]-1) / 2) + 1;
    
    /* Find non-coincident pairs in v direction */
    found = 0;
    for (u=0; u < numcp[0]; u++)
    {
	for (v=0; v < numcp[1]-1; v += 2)
	{
	    a = cp[v * numcp[0] + u];
	    b = cp[(v + 2) * numcp[0] + u];
	    if (!vec_cmp(a,b))
	    {
		found = 1;
		break;
	    }
	}
	if (found) break;
    }
    if (!found) Error("Bad mesh control points");

    /* Find subdivision level in v */
    vec_copy(a, test[0]);
    vec_copy((a+numcp[0]*3), test[1]);
    vec_copy(b, test[2]);
    level = mesh_find_level(test);
    size[1] = (LEVEL_WIDTH(level) - 1)* ((numcp[1]-1) / 2) + 1;    
}

void BSPDataModel::mesh_fill_curve_3(int numcp, int size, int stride, vec3_t *p)
{
    int step, halfstep, i, mid;
    vec3_t a, b;

    step = (size-1) / (numcp-1);

    while (step > 0)
    {
	halfstep = step / 2;
	for (i=0; i < size-1; i += step*2)
	{
	    mid = (i+step)*stride;
	    vec_avg(p[i*stride], p[mid], a);
	    vec_avg(p[mid], p[(i+step*2)*stride], b);
	    vec_avg(a, b, p[mid]);

	    if (halfstep > 0)
	    {
		vec_copy(a, p[(i+halfstep)*stride]);
		vec_copy(b, p[(i+3*halfstep)*stride]);
	    }
	}
	
	step /= 2;
    }
}

void BSPDataModel::mesh_fill_curve_2(int numcp, int size, int stride, vec2_t *p)
{
    int step, halfstep, i, mid;
    vec2_t a, b;

    step = (size-1) / (numcp-1);

    while (step > 0)
    {
	halfstep = step / 2;
	for (i=0; i < size-1; i += step*2)
	{
	    mid = (i+step)*stride;
	    vec2_avg(p[i*stride], p[mid], a);
	    vec2_avg(p[mid], p[(i+step*2)*stride], b);
	    vec2_avg(a, b, p[mid]);

	    if (halfstep > 0)
	    {
		vec2_copy(a, p[(i+halfstep)*stride]);
		vec2_copy(b, p[(i+3*halfstep)*stride]);
	    }
	}
	
	step /= 2;
    }
}

void BSPDataModel::mesh_fill_curve_c(int numcp, int size, int stride, colour_t *p)
{
    int step, halfstep, i, mid;
    colour_t a, b;

    step = (size-1) / (numcp-1);

    while (step > 0)
    {
	halfstep = step / 2;
	for (i=0; i < size-1; i += step*2)
	{
	    mid = (i+step)*stride;
	    colour_avg(p[i*stride], p[mid], a);
	    colour_avg(p[mid], p[(i+step*2)*stride], b);
	    colour_avg(a, b, p[mid]);

	    if (halfstep > 0)
	    {
		colour_copy(a, p[(i+halfstep)*stride]);
		colour_copy(b, p[(i+3*halfstep)*stride]);
	    }
	}
	
	step /= 2;
    }
}

void BSPDataModel::mesh_fill_patch_3(int *numcp, int *size, vec3_t *p)
{
    int step, u, v;

    /* Fill in control points in v direction */
    step = (size[0]-1) / (numcp[0]-1);    
    for (u = 0; u < size[0]; u += step)
    {
	mesh_fill_curve_3(numcp[1], size[1], size[0], p + u);
    }

    /* Fill in the rest in the u direction */
    for (v = 0; v < size[1]; v++)
    {
	mesh_fill_curve_3(numcp[0], size[0], 1, p + v * size[0]);
    }
}

void BSPDataModel::mesh_fill_patch_2(int *numcp, int *size, vec2_t *p)
{
    int step, u, v;

    /* Fill in control points in v direction */
    step = (size[0]-1) / (numcp[0]-1);    
    for (u = 0; u < size[0]; u += step)
    {
	mesh_fill_curve_2(numcp[1], size[1], size[0], p + u);
    }

    /* Fill in the rest in the u direction */
    for (v = 0; v < size[1]; v++)
    {
	mesh_fill_curve_2(numcp[0], size[0], 1, p + v * size[0]);
    }
}

void BSPDataModel::mesh_fill_patch_c(int *numcp, int *size, colour_t *p)
{
    int step, u, v;

    /* Fill in control points in v direction */
    step = (size[0]-1) / (numcp[0]-1);    
    for (u = 0; u < size[0]; u += step)
    {
	mesh_fill_curve_c(numcp[1], size[1], size[0], p + u);
    }

    /* Fill in the rest in the u direction */
    for (v = 0; v < size[1]; v++)
    {
	mesh_fill_curve_c(numcp[0], size[0], 1, p + v * size[0]);
    }
}

void BSPDataModel::mesh_create(face_t *face, mesh_t *mesh)
{
    int step[2], size[2], len, i, u, v, p;
    vec3_t *cp;
    vertex_t *vert;

    cp = (vec3_t*)malloc(face->numverts * sizeof(vec3_t));
    vert = &r_verts[face->firstvert];
    for (i=0; i < face->numverts; i++)
    {
	vec_copy(vert->v_point, cp[i]);
	vert++;
    }

    /* Find the degree of subdivision in the u and v directions */
    mesh_find_size(face->mesh_cp, cp, size);
    free(cp);

    /* Allocate space for mesh */
    len = size[0] * size[1];
    mesh->size[0] = size[0];
    mesh->size[1] = size[1];
    mesh->points = (vec3_t*)malloc(len * (sizeof(vec3_t) +
					  2 * sizeof(texcoord_t) +
					 sizeof(colour_t)));
    mesh->colour = (colour_t*)(mesh->points + len);
    mesh->tex_st = (texcoord_t*)(mesh->colour + len);
    mesh->lm_st = mesh->tex_st + len;

    /* Fill in sparse mesh control points */
    step[0] = (size[0]-1) / (face->mesh_cp[0]-1);
    step[1] = (size[1]-1) / (face->mesh_cp[1]-1);
    vert = &r_verts[face->firstvert];
    for (v = 0; v < size[1]; v += step[1])
    {
	for (u = 0; u < size[0]; u += step[0])
	{
	    p = v * size[0] + u;
	    vec_copy(vert->v_point, mesh->points[p]);
	    colour_copy(vert->colour, mesh->colour[p]);
	    vec2_copy(vert->tex_st, mesh->tex_st[p]);
	    vec2_copy(vert->lm_st, mesh->lm_st[p]);
	    vert++;
	}
    }

    /* Fill in each mesh */
    mesh_fill_patch_3(face->mesh_cp, size, mesh->points);
    mesh_fill_patch_c(face->mesh_cp, size, mesh->colour);
    mesh_fill_patch_2(face->mesh_cp, size, (vec2_t*)mesh->tex_st);
    mesh_fill_patch_2(face->mesh_cp, size, (vec2_t*)mesh->lm_st);

    /* Allocate and fill element table */
    mesh->numelems = (size[0]-1) * (size[1]-1) * 6;
    mesh->elems = (uint_t*)malloc(mesh->numelems * sizeof(uint_t));

    i = 0;
    for (v = 0; v < size[1]-1; ++v)
    {
	for (u = 0; u < size[0]-1; ++u)
	{
	    mesh->elems[i++] = v * size[0] + u;
	    mesh->elems[i++] = (v+1) * size[0] + u;
	    mesh->elems[i++] = v * size[0] + u + 1;
	    mesh->elems[i++] = v * size[0] + u + 1;
	    mesh->elems[i++] = (v+1) * size[0] + u;
	    mesh->elems[i++] = (v+1) * size[0] + u + 1;
	}
    }
}


/* end of mesh methods, begin shader methods ***************************/
void BSPDataModel::shader_readall()
{
    int numfiles, len, fixlen;
    char shaderlist[4096], *fname;
	char tmpPath[PATHLEN];

    printf("Initializing Shaders\n");

    r_numtextures = 0;
    r_shaders = (shader_t*)malloc(r_numshaders * sizeof(shader_t));
    r_texfiles = (texfile_t*)malloc(MAX_NUM_TEXTURES * sizeof(texfile_t));
    shaderfound = (int*)malloc(r_numshaders * sizeof(int));    
    shaderbuf = (char*)malloc(SHADERBUF_SIZE);
    memset(shaderfound, 0, r_numshaders * sizeof(int));

    /* Get a list of shader script files */
    numfiles = shader_listshaders(4096, shaderlist);

	strcpy(tmpPath, rootPath);
#ifdef WIN32
	strcat(tmpPath, "scripts\\");
#else
	strcat(tmpPath, "scripts/");
#endif
	fixlen = strlen(tmpPath);

    /* Parse each file */
    fname = shaderlist;
    while (numfiles--)
	{
		strcpy(&tmpPath[fixlen], fname);
		len = shader_readfile(tmpPath, SHADERBUF_SIZE, (unsigned char *)shaderbuf);
		curpos = shaderbuf;
		endpos = curpos + len;
		printf("...loading '%s'\n", fname);
		shader_read();
		fname += strlen(fname) + 1;
	}
    free(shaderbuf);

    /* Make default shaders for those that weren't found */
    shader_makedefaults();
    free(shaderfound);
    
    printf("done.\n");
}

uint_t BSPDataModel::shader_listshaders(uint_t bufsize, char *buf)
{
    uint_t num = 0, len;
    int status;
	DIR *shaderDir;
	struct dirent* curFile;
	char tmpPath[PATHLEN];
    char *end = buf + bufsize;

	strcpy(tmpPath, rootPath);
#ifdef WIN32
	strcat(tmpPath, "scripts\\");
#else
	strcat(tmpPath, "scripts/");
#endif

	if ( (shaderDir = opendir(tmpPath)) == NULL)
		return 0;
	

    curFile = readdir(shaderDir);
    while (curFile)
    {
		len = strlen(curFile->d_name);
		if (len >= 7 && strcmp(&(curFile->d_name[len-7]), ".shader") == 0)
		{
			strcpy(buf, curFile->d_name);
			num++;
			buf += len + 1;
			if (buf > end)
			Error("shader_listshaders(): buffer overrun");
		}
		curFile = readdir(shaderDir);
    }

	closedir(shaderDir);
    return num;
}


uint_t BSPDataModel::shader_readfile(const char *path, uint_t bufsize, byte_t *buf)
{
    FILE *fptr;
	struct stat stat_buf;

    if ( (fptr = fopen(path, "rb")) == NULL )
		return 0;

	if (stat(path, &stat_buf) < 0)
		return 0;

    if ( stat_buf.st_size != 0)
		fread(buf, 1, stat_buf.st_size, fptr);
    
    fclose(fptr);    
    return stat_buf.st_size;
}

void BSPDataModel::shader_freeall()
{
    int i;

    for (i=0; i < r_numtextures; i++)
	free(r_texfiles[i].fname);
    free(r_texfiles);
    free(r_shaders);
}

int BSPDataModel::shader_lookup(const char *name)
{
    int i, id = -1;

    /* FIXME: This should be done with a hash table! */

    for (i=0; i < r_numshaders; ++i)
    {
	if (!strcmp(name, r_shaderrefs[i].name))
	{
	    id = i;
	    break;
	}
    }
    return id;
}

void BSPDataModel::shader_read()
{
    int id;
    char *tok;

    while ((tok = nexttok()) != NULL)
    {
	id = shader_lookup(tok);
	if (id < 0)
	{
	    shader_skip();
	    continue;
	}

	/* Mark shaderref as 'found' */
	shaderfound[id] = 1;
	
	/* Set defaults */
	r_shaders[id].flags = 0;
	r_shaders[id].numpasses = 0;
	
	/* Opening brace */
	tok = nexttok();
	if (tok[0] != '{') Syntax();

	shaderkey_t shaderkeys[] =
	{
		{"cull", 1, 1, shader_cull},
		{"surfaceparm", 1, 1, shader_surfaceparm},
		{"skyparms", 3, 3, shader_skyparms},
		{"nomipmaps", 0, 0, shader_nomipmaps},
		{"deformvertexes", 1, 9, shader_deformvertexes},
		{NULL, 0, 0, NULL}  /* Sentinel */
	};
	while ((tok = nexttok()) != NULL)
	{
	    if (tok[0] == '{') /* Start new pass */
	    {
		int pass = r_shaders[id].numpasses++;
		shader_readpass(&r_shaders[id], &r_shaders[id].pass[pass]);
	    }

	    else if (tok[0] == '}') /* End of shader */
		break;

	    else
		shader_parsetok(&r_shaders[id], NULL, shaderkeys, tok);
	}

	/* Explicit depth write for first pass */
	/* FIXME: is this how we handle transparent ? */
	if (! (r_shaders[id].flags & SHADER_DEPTHWRITE) &&
	    ! (r_shaders[id].flags & SHADER_TRANSPARENT) &&
	    ! (r_shaders[id].flags & SHADER_SKY) &&
	    r_shaders[id].numpasses > 0)
	{
	    r_shaders[id].pass[0].flags |= SHADER_DEPTHWRITE;
	}
    }
}

void BSPDataModel::shader_skip()
{
    char *tok;
    int brace_count;

    /* Opening brace */
    tok = nexttok();
    if (tok[0] != '{') Syntax();

    for (brace_count = 1; brace_count > 0 && curpos < endpos; curpos++)
    {
	if (*curpos == '{')
	    brace_count++;
	else if (*curpos == '}')
	    brace_count--;
    }
}

void BSPDataModel::shader_readpass(shader_t *shader, shaderpass_t *pass)
{
    char *tok;

    /* Set defaults */
    pass->flags = 0;
    pass->texref = -1;
    pass->depthfunc = GL_LEQUAL;
    pass->rgbgen = SHADER_GEN_IDENTITY;
    pass->tcmod = 0;

	shaderkey_t shaderpasskeys[] =
	{
		{"map", 1, 1, shaderpass_map},
		{"rgbgen", 1, 6, shaderpass_rgbgen},
		{"blendfunc", 1, 2, shaderpass_blendfunc},
		{"depthfunc", 1, 1, shaderpass_depthfunc},
		{"depthwrite", 0, 0, shaderpass_depthwrite},
		{"alphafunc", 1, 1, shaderpass_alphafunc},
		{"tcmod", 2, 7, shaderpass_tcmod},
		{"animmap", 3, SHADER_ARGS_MAX, shaderpass_animmap},
		{"clampmap", 1, 1, shaderpass_clampmap},
		{"tcgen", 1, 10, shaderpass_tcgen},
		{NULL, 0, 0, NULL}  /* Sentinel */
	};
    while ((tok = nexttok()) != NULL)
    {
	if (tok[0] == '}') /* End of pass */
	    break;

	else
	    shader_parsetok(shader, pass, shaderpasskeys, tok);
    }
}

void BSPDataModel::shader_parsetok(shader_t *shader, shaderpass_t *pass, shaderkey_t *keys,
		char *tok)
{
    shaderkey_t *key;
    char *c, *args[SHADER_ARGS_MAX];
    int numargs;

    /* Lowercase the token */
    c = tok;
    while (*c++) *c =  LOWERCASE(*c);
    
    /* FIXME: This should be done with a hash table! */

    for (key = keys; key->keyword != NULL; key++)
    {
	if (strcmp(tok, key->keyword) == 0)
	{
	    for (numargs=0; (c = nextarg()) != NULL; numargs++)
	    {
		/* Lowercase the argument */
		args[numargs] = c;
		while (*c) {*c = LOWERCASE(*c); c++;}
	    }
	    if (numargs < key->minargs || numargs > key->maxargs)
		Syntax();
	    
	    if (key->func)
		key->func(this, shader, pass, numargs, args);
	    return;
	}
    }

    /* Unidentified keyword: no error for now, just advance to end of line */
    while (*curpos != '\n')
	if (++curpos == endpos) break;    
}

void BSPDataModel::shader_makedefaults()
{
    int i, f, firsttrisurf, lasttrisurf, trisurf, md3;
    char fname[128];

    /* Find first and last trisurf */
    firsttrisurf = -1;
    for (f = 0; f < r_numfaces; f++)
    {
	if (r_faces[f].facetype == FACETYPE_TRISURF)
	{
	    firsttrisurf = f;
	    break;
	}
    }
    if (firsttrisurf >= 0)
    {
	for (f = firsttrisurf; f < r_numfaces; f++)
	{
	    if (r_faces[f].facetype != FACETYPE_TRISURF)
	    {
		lasttrisurf = f;
		break;
	    }
	}
    }
	
    for (i=0; i < r_numshaders; ++i)
    {
	if (shaderfound[i]) continue;

	/* Special exception: noshader */
	if (!strcmp(r_shaderrefs[i].name, "noshader"))
	{
	    r_shaders[i].numpasses = 0;
	    continue;
	}
	
	/* Append tga to get file name */
	strcpy(fname, r_shaderrefs[i].name);
	strcat(fname, ".tga");

	/* Check if shader is for an md3 */
	md3 = 0;
	trisurf = 0;
	if (i >= r_addshaderstart)
	{
	    md3 = 1;
	}
	else
	{
	    /* Check if shader is for a trisurf */
	    if (firsttrisurf >= 0)
	    {
		for (f=firsttrisurf; f <= lasttrisurf; f++)
		{
		    if (r_faces[f].shader == i)
		    {
			trisurf = 1;
			break;
		    }
		}
	    }
	}

	if (md3)
	{
	    r_shaders[i].flags = SHADER_NOCULL;
	    r_shaders[i].numpasses = 1;
	    r_shaders[i].pass[0].flags = SHADER_DEPTHWRITE;
	    r_shaders[i].pass[0].texref = shader_gettexref(fname);
	    r_shaders[i].pass[0].depthfunc = GL_LEQUAL;
	    r_shaders[i].pass[0].rgbgen = SHADER_GEN_IDENTITY;
	}
	else if (trisurf)
	{
	    r_shaders[i].flags = SHADER_NOCULL | SHADER_NEEDCOLOURS;
	    r_shaders[i].numpasses = 1;
	    r_shaders[i].pass[0].flags = SHADER_DEPTHWRITE;
	    r_shaders[i].pass[0].texref = shader_gettexref(fname);
	    r_shaders[i].pass[0].depthfunc = GL_LEQUAL;
	    r_shaders[i].pass[0].rgbgen = SHADER_GEN_VERTEX;	    
	}
	else
	{
	    r_shaders[i].flags = 0;
	    r_shaders[i].numpasses = 2;
	    r_shaders[i].pass[0].flags = SHADER_LIGHTMAP | SHADER_DEPTHWRITE;
	    r_shaders[i].pass[0].texref = -1;
	    r_shaders[i].pass[0].depthfunc = GL_LEQUAL;
	    r_shaders[i].pass[0].rgbgen = SHADER_GEN_IDENTITY;
	    
	    r_shaders[i].pass[1].flags = SHADER_BLEND;
	    r_shaders[i].pass[1].texref = shader_gettexref(fname);
	    r_shaders[i].pass[1].blendsrc = GL_DST_COLOR;
	    r_shaders[i].pass[1].blenddst = GL_ZERO;
	    r_shaders[i].pass[1].depthfunc = GL_LEQUAL;
	    r_shaders[i].pass[1].rgbgen = SHADER_GEN_IDENTITY;
	}
    }
}

int BSPDataModel::shader_gettexref(const char *fname)
{
    int i;
    
    /* FIXME: hash table again! */
    for (i=0; i < r_numtextures; ++i)
    {
	if (!strcmp(fname, r_texfiles[i].fname))
	    return i;
    }
    
    if (r_numtextures == MAX_NUM_TEXTURES)
	Error("Texture count exceeded");
    r_texfiles[r_numtextures].flags = 0;
    r_texfiles[r_numtextures].fname = strdup(fname);
    return r_numtextures++;
}

char* BSPDataModel::nexttok()
{
    char *tok;
    
    while (curpos < endpos)
    {
	/* Skip leading whitespace */
	while (*curpos == ' ' || *curpos == '\t' || *curpos == '\n' ||
	      *curpos == '\r')
	    if (++curpos == endpos) return NULL;

	/* Check for comment */
	if (curpos[0] == '/' && curpos[1] == '/')
	{
	    /* Skip to end of comment line */
	    while (*curpos++ != '\n')
		if (curpos == endpos) return NULL;
	    /* Restart with leading whitespace */
	    continue;
	}

	/* Seek to end of token */
	tok = curpos;
	while (*curpos != ' ' && *curpos != '\t' && *curpos != '\n' &&
	      *curpos != '\r')
	    if (++curpos == endpos) break;

	/* Zero whitespace character and advance by one */
	*curpos++ = '\0';
	return tok;
    }
    return NULL;
}

char* BSPDataModel::nextarg()
{
    char *arg;

    while (curpos < endpos)
    {
	/* Skip leading whitespace */
	while (*curpos == ' ' || *curpos == '\t')
	    if (++curpos == endpos) return NULL;

	/* Check for newline or comment */
	if (*curpos == '\n' || *curpos == '\r' ||
	    (curpos[0] == '/' && curpos[1] == '/'))
	    return NULL;
	
	/* Seek to end of token */
	arg = curpos;
	while (*curpos != ' ' && *curpos != '\t' && *curpos != '\n' &&
	      *curpos != '\r')
	    if (++curpos == endpos) break;

	/* Zero whitespace character and advance by one */
	*curpos++ = '\0';
	return arg;
    }
    return NULL;
}


inline void BSPDataModel::Syntax()
{
    printf("Syntax error in shader\n");
}

void BSPDataModel::shader_parsefunc(char **args, shaderfunc_t *func)
{
	if (!strcmp(args[0], "sin"))
	    func->func = SHADER_FUNC_SIN;
	else if (!strcmp(args[0], "triangle"))
	    func->func = SHADER_FUNC_TRIANGLE;
	else if (!strcmp(args[0], "square"))
	    func->func = SHADER_FUNC_SQUARE;
	else if (!strcmp(args[0], "sawtooth"))
	    func->func = SHADER_FUNC_SAWTOOTH;
	else if (!strcmp(args[0], "inversesawtooth"))
	    func->func = SHADER_FUNC_INVERSESAWTOOTH;
	else
	    Syntax();

	func->args[0] = atof(args[1]);
	func->args[1] = atof(args[2]);
	func->args[2] = atof(args[3]);
	func->args[3] = atof(args[4]);
}

/****************** shader keyword functions ************************/


void shader_cull(BSPDataModel *bspDM, shader_t *shader, shaderpass_t *pass, int numargs, char **args)
{
    if (!strcmp(args[0], "disable") || !strcmp(args[0], "none"))
	shader->flags |= SHADER_NOCULL;
}

void shader_surfaceparm(BSPDataModel *bspDM, shader_t *shader, shaderpass_t *pass, int numargs,
		   char **args)
{
    if (!strcmp(args[0], "trans"))
	shader->flags |= SHADER_TRANSPARENT;
    else if (!strcmp(args[0], "sky"))
	shader->flags |= SHADER_SKY;
}

void shader_skyparms(BSPDataModel *bspDM, shader_t *shader, shaderpass_t *pass, int numargs,
		char **args)
{
    shader->skyheight = atof(args[1]);
}

void shader_nomipmaps(BSPDataModel *bspDM, shader_t *shader, shaderpass_t *pass, int numargs,
		 char **args)
{
    shader->flags |= SHADER_NOMIPMAPS;
}

void shader_deformvertexes(BSPDataModel *bspDM, shader_t *shader, shaderpass_t *pass, int numargs,
		      char **args)
{
    if (!strcmp(args[0], "wave"))
    {
	if (numargs != 7)
	    bspDM->Syntax();
	shader->flags |= SHADER_DEFORMVERTS;
	shader->deformv_wavesize = atof(args[1]);
	bspDM->shader_parsefunc(&args[2], &shader->deformv_wavefunc);
    }
}

/****************** shader pass keyword functions *******************/

void shaderpass_map(BSPDataModel *bspDM, shader_t *shader, shaderpass_t *pass, int numargs, char **args)
{
    if (!strcmp(args[0], "$lightmap"))
	pass->flags |= SHADER_LIGHTMAP;
    else
    {
	pass->texref = bspDM->shader_gettexref(args[0]);
	if (shader->flags & SHADER_NOMIPMAPS)
	    bspDM->r_texfiles[pass->texref].flags |= TEXFILE_NOMIPMAPS;
    }
}

void shaderpass_rgbgen(BSPDataModel *bspDM, shader_t *shader, shaderpass_t *pass, int numargs,
		  char **args)
{
    if (!strcmp(args[0], "identity"))
	return; /* Default */
    else if (!strcmp(args[0], "wave"))
    {
	if (numargs != 6)
	    bspDM->Syntax();

	pass->rgbgen = SHADER_GEN_WAVE;
	bspDM->shader_parsefunc(&args[1], &pass->rgbgen_func);
    }
    else if (!strcmp(args[0], "vertex") || !strcmp(args[0], "lightingdiffuse"))
    {
	shader->flags |= SHADER_NEEDCOLOURS;
	pass->rgbgen = SHADER_GEN_VERTEX;
    }
}

void shaderpass_blendfunc(BSPDataModel *bspDM, shader_t *shader, shaderpass_t *pass, int numargs,
		     char **args)
{
    pass->flags |= SHADER_BLEND;
    
    if (numargs == 1)
    {
	if (!strcmp(args[0], "blend"))
	{
	    pass->blendsrc = GL_SRC_ALPHA;
	    pass->blenddst = GL_ONE_MINUS_SRC_ALPHA;
	}
	else if (!strcmp(args[0], "filter"))
	{
	    pass->blendsrc = GL_DST_COLOR;
	    pass->blenddst = GL_ZERO;
	}
	else if (!strcmp(args[0], "add"))
	{
	    pass->blendsrc = pass->blenddst = GL_ONE;
	}
	else
	    bspDM->Syntax();
    }
    else
    {
	int i;
	uint_t *blend;
	for (i=0; i < 2; ++i)
	{
	    blend = i == 0 ? &pass->blendsrc : &pass->blenddst;
	    if (!strcmp(args[i], "gl_zero"))
		*blend = GL_ZERO;
	    else if (!strcmp(args[i], "gl_one"))
		*blend = GL_ONE;
	    else if (!strcmp(args[i], "gl_dst_color"))
		*blend = GL_DST_COLOR;
	    else if (!strcmp(args[i], "gl_one_minus_src_alpha"))
		*blend = GL_ONE_MINUS_SRC_ALPHA;
	    else if (!strcmp(args[i], "gl_src_alpha"))
		*blend = GL_SRC_ALPHA;
	    else if (!strcmp(args[i], "gl_src_color"))
		*blend = GL_SRC_COLOR;
	    else if (!strcmp(args[i], "gl_one_minus_dst_color"))
		*blend = GL_ONE_MINUS_DST_COLOR;
	    else if (!strcmp(args[i], "gl_one_minus_src_color"))
		*blend = GL_ONE_MINUS_SRC_COLOR;
	    else if (!strcmp(args[i], "gl_dst_alpha"))
		*blend = GL_DST_ALPHA;
	    else if (!strcmp(args[i], "gl_one_minus_dst_alpha"))
		*blend = GL_ONE_MINUS_DST_ALPHA;
	    else
		bspDM->Syntax();
	}
    }
}

void shaderpass_depthfunc(BSPDataModel *bspDM, shader_t *shader, shaderpass_t *pass, int numargs,
		     char **args)
{
    if (!strcmp(args[0], "equal"))
		pass->depthfunc = GL_EQUAL;
    else
		bspDM->Syntax();
}

void shaderpass_depthwrite(BSPDataModel *bspDM, shader_t *shader, shaderpass_t *pass, int numargs,
		      char **args)
{
    /* FIXME: Why oh why is depthwrite enabled in the sky shaders ???? */
    if (shader->flags & SHADER_SKY) return;
    
    shader->flags |= SHADER_DEPTHWRITE;
    pass->flags |= SHADER_DEPTHWRITE;
}

void shaderpass_alphafunc(BSPDataModel *bspDM, shader_t *shader, shaderpass_t *pass, int numargs,
		     char **args)
{
    pass->flags |= SHADER_ALPHAFUNC;
    
    if (!strcmp(args[0], "gt0"))
    {
		pass->alphafunc = GL_GREATER;
		pass->alphafuncref = 0.0f;
    }
    else if (!strcmp(args[0], "ge128"))
    {
		pass->alphafunc = GL_GEQUAL;
		pass->alphafuncref = 0.5f;
    }
    else
		bspDM->Syntax();
}

void shaderpass_tcmod(BSPDataModel *bspDM, shader_t *shader, shaderpass_t *pass, int numargs,
		 char **args)
{
    pass->flags |= SHADER_TCMOD;
    
    if (!strcmp(args[0], "scale"))
    {
	if (numargs != 3) bspDM->Syntax();
	pass->tcmod |= SHADER_TCMOD_SCALE;
	pass->tcmod_scale[0] = atof(args[1]);
	pass->tcmod_scale[1] = atof(args[2]);
    }
    else if (!strcmp(args[0], "rotate"))
    {
	pass->tcmod |= SHADER_TCMOD_ROTATE;
	pass->tcmod_rotate = atof(args[1]);
    }
    else if (!strcmp(args[0], "scroll"))
    {
	if (numargs != 3) bspDM->Syntax();
	pass->tcmod |= SHADER_TCMOD_SCROLL;
	pass->tcmod_scroll[0] = atof(args[1]);
	pass->tcmod_scroll[1] = atof(args[2]);
    }
    else if (!strcmp(args[0], "stretch"))
    {
	if (numargs != 6) bspDM->Syntax();
	pass->tcmod |= SHADER_TCMOD_STRETCH;
	bspDM->shader_parsefunc(&args[1], &pass->tcmod_stretch);
    }
    else if (!strcmp(args[0], "transform"))
    {
	int i;
	if (numargs != 7) bspDM->Syntax();
	pass->tcmod |= SHADER_TCMOD_TRANSFORM;
	for (i=0; i < 6; ++i)
	    pass->tcmod_transform[i] = atof(args[i+1]);
    }
    else if (!strcmp(args[0], "turb"))
    {
	int i, a1;
	if (numargs == 5)
	    a1 = 1;
	else if (numargs == 6)
	    a1 = 2;
	else
	    bspDM->Syntax();
	pass->tcmod |= SHADER_TCMOD_TURB;
	for (i=0; i < 4; ++i)
	    pass->tcmod_turb[i] = atof(args[i+a1]);
    }
    else
	bspDM->Syntax();
}

void shaderpass_animmap(BSPDataModel *bspDM, shader_t *shader, shaderpass_t *pass, int numargs,
		   char **args)
{
    int i;
    pass->flags |= SHADER_ANIMMAP;
    pass->anim_fps = atof(args[0]);
    pass->anim_numframes = numargs - 1;
    for (i=1; i < numargs; ++i)
	pass->anim_frames[i-1] = bspDM->shader_gettexref(args[i]);
}

void shaderpass_clampmap(BSPDataModel *bspDM, shader_t *shader, shaderpass_t *pass, int numargs,
		    char **args)
{
    pass->texref = bspDM->shader_gettexref(args[0]);
    bspDM->r_texfiles[pass->texref].flags |= TEXFILE_CLAMP;
    if (shader->flags & SHADER_NOMIPMAPS)
	bspDM->r_texfiles[pass->texref].flags |= TEXFILE_NOMIPMAPS;    
}

void shaderpass_tcgen(BSPDataModel *bspDM, shader_t *shader, shaderpass_t *pass, int numargs,
		 char **args)
{
    if (!strcmp(args[0], "environment"))
    {
	if (numargs != 1)
	    bspDM->Syntax();
	pass->flags |= SHADER_TCGEN_ENV;
    }
}

/* end of shader methods, begin texture methods */

void BSPDataModel::tex_loadobjs(void)
{
    int i;
    byte_t *rgb;
    int width, height, format, len;
	char tmpPath[PATHLEN];
	int fixlen;

    imgbuf = malloc(IMG_BUFSIZE);

	//satyam
	strcpy(tmpPath, MATRIXDIR);
	fixlen = strlen(tmpPath);
	glGenTextures(NUM_MATRIX_TEXTURES, matrix_textures);
	for ( i = 0; i < NUM_MATRIX_TEXTURES; i++)
	{
		sprintf(&tmpPath[fixlen], "%d.jpg", i);
		printf(tmpPath);
	    if (!jpg_readtex(tmpPath, &rgb, &width, &height, &format))
			Error("Could not load the matrix.\n");
		glBindTexture(GL_TEXTURE_2D, matrix_textures[i]);
		tex_loadtexture(rgb, width, height, format, 0);
		free((void *)rgb);

	}

    r_textures = (uint_t*)malloc(r_numtextures * sizeof(uint_t));
    glGenTextures(r_numtextures, r_textures);
    
	strcpy(tmpPath, rootPath);
	fixlen = strlen(tmpPath);

    for (i=0; i < r_numtextures; ++i)
    {
	printf("loading...%s\n", r_texfiles[i].fname);

	strcpy(&tmpPath[fixlen], r_texfiles[i].fname);

	len = strlen(tmpPath);
	if (!strcmp(&tmpPath[len-4], ".jpg"))
	{
	    if (!jpg_readtex(tmpPath, &rgb, &width, &height,
			     &format))
		Error("Could not open file %s", tmpPath);
	}
	else if (!strcmp(&tmpPath[len-4], ".tga"))
	{
	    if (!tga_readtex(tmpPath, &rgb, &width, &height,
			     &format))
	    {
		/* Might still be a jpg file !!! (compatibility with old
		   shader scripts?) */
		strcpy(&tmpPath[len-3], "jpg");
		if (!jpg_readtex(tmpPath, &rgb, &width,
				 &height, &format))
		{
		    /* FIXME: This should be an error, but still happens
		       in the demo levels ! */
		    strcpy(&tmpPath[len-3], "tga");
		    printf("Could not open file %s\n", tmpPath);
		    continue;
		}
	    }
	}
	else
	    Error("Unknown format for %s", tmpPath);
	
	glBindTexture(GL_TEXTURE_2D, r_textures[i]);
	tex_loadtexture(rgb, width, height, format, r_texfiles[i].flags);
	free(rgb);
    }
    free(imgbuf);
}

void BSPDataModel::tex_freeobjs(void)
{
    glDeleteTextures(r_numtextures, r_textures);
    free(r_textures);
}

void BSPDataModel::tex_loadtexture(byte_t *rgb, int w, int h, int format, uint_t flags)
{
    byte_t *tex = rgb;
    int width = w, height = h;
    int size = width*height* (format == GL_RGB ? 3 : 4);

    /* Scale image down for biased level of detail (lowered texture quality) */
    if (r_lodbias > 0)
    {
	width /= 1 << r_lodbias;
	height /= 1 << r_lodbias;
	tex = (byte_t *)malloc(size);

	gluScaleImage(format, w, h, GL_UNSIGNED_BYTE, rgb,
		      width, height, GL_UNSIGNED_BYTE, tex);
    }

    /* Not really a gamma: prelighten the texture to compensate for
       darkening after lightmap application. */
    /* FIXME: should alpha be brightened too? */
    if (r_gamma != 1.0)
    {
	int i, val;

	for (i=0; i<size; ++i)
	{
	    val = tex[i] * r_gamma;
	    if (val > 255) val = 255;
	    tex[i] = val;
	}
    }

    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER,
		    GL_LINEAR_MIPMAP_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

    if (flags & TEXFILE_CLAMP)
    {
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    }
    else
    {
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
    }

    if (flags & TEXFILE_NOMIPMAPS)
    {
	glTexImage2D(GL_TEXTURE_2D, 0, format, width, height, 0, format,
		     GL_UNSIGNED_BYTE, tex);
    }
    else
    {
	gluBuild2DMipmaps(GL_TEXTURE_2D, format, width, height, format,
			  GL_UNSIGNED_BYTE, tex);
    }

    if (r_lodbias > 0)
	free(tex);
}

int BSPDataModel::tga_readtex(const char *fname, byte_t **rgb, int *w, int *h, int *format)
{
    tgaheader_t *tgahead;
    byte_t *img, *tga, *tgacur, *tgaend;
    int tgalen, len, depth = 0;

    tgalen = shader_readfile(fname, IMG_BUFSIZE, (byte_t *)imgbuf);
    if (!tgalen) return 0;

    tga = (byte_t*)imgbuf;
    tgaend = tga + tgalen;
    
    tgahead = (tgaheader_t*)tga;
    if (tgahead->imgtype != 2 && tgahead->imgtype != 10)
	Error("Bad tga image type");

    if (tgahead->pixsize == 24)
	depth = 3;
    else if (tgahead->pixsize == 32)
	depth = 4;
    else
	Error("Non 24 or 32 bit tga image");
    
    len = tgahead->width * tgahead->height * depth;
    img = (byte_t *)malloc(len);

    tgacur = tga + sizeof(tgaheader_t) + tgahead->idlen;
    if (tgahead->imgtype == 10)
    {
	int i, j, packetlen;
	byte_t packethead;
	byte_t *c = img, *end = img + len;
	byte_t rlc[4];
	
	while (c < end)
	{
	    packethead = *tgacur;
	    if (++tgacur > tgaend)
		Error("Unexpected end of tga file");
	    if (packethead & 0x80)
	    {
		/* Run-length packet */
		packetlen = (packethead & 0x7f) + 1;
		memcpy(rlc, tgacur, depth);
		if ((tgacur += depth) > tgaend)
		    Error("Unexpected end of tga file");
		for (j=0; j < packetlen; ++j)
		    for(i=0; i < depth; ++i)
			*c++ = rlc[i];
	    }
	    else
	    {
		/* Raw data packet */
		packetlen = packethead + 1;
		memcpy(c, tgacur, depth * packetlen);
		if ((tgacur += depth * packetlen) > tgaend)
		    Error("Unexpected end of tga file");
		c += packetlen * depth;
	    }
	}

	/* Flip image in y */
	{
	    int i, linelen;
	    byte_t *temp;
	    
	    linelen = tgahead->width * depth;
	    temp = (byte_t *)malloc(linelen);
	    for (i=0; i < tgahead->height/2; ++i)
	    {
		memcpy(temp, &img[i * linelen], linelen);
		memcpy(&img[i * linelen], &img[(tgahead->height - i - 1)
					      * linelen], linelen);
		memcpy(&img[(tgahead->height - i - 1) * linelen], temp,
		       linelen);
	    }
	    free(temp);
	}	
    }
    else
    {
	int i, linelen;
	
	if (tgaend - tgacur + 1 < len)
	    Error("Bad tga image data length");

	/* Flip image in y */
	linelen = tgahead->width * depth;
	for (i=0; i < tgahead->height; ++i)
	    memcpy(&img[i * linelen],
		   &tgacur[(tgahead->height - i - 1) * linelen], linelen);
    }    

    /* Exchange B and R to get RGBA ordering */
    {
	int i;
	byte_t temp;

	for (i=0; i < len; i += depth)
	{
	    temp = img[i];
	    img[i] = img[i+2];
	    img[i+2] = temp;
	}
    }
    
    *rgb = img;
    *w = tgahead->width;
    *h = tgahead->height;
    *format = (depth == 3) ? GL_RGB : GL_RGBA;
    return 1;
}

void jpg_noop(j_decompress_ptr cinfo)
{
}

boolean jpg_fill_input_buffer(j_decompress_ptr cinfo)
{
    Error("Premeture end of jpeg file");
    return TRUE;
}

void jpg_skip_input_data(j_decompress_ptr cinfo, long num_bytes)
{
        
    cinfo->src->next_input_byte += (size_t) num_bytes;
    cinfo->src->bytes_in_buffer -= (size_t) num_bytes;
    if (cinfo->src->bytes_in_buffer < 0)
	Error("Premeture end of jpeg file");
}

void jpeg_mem_src(j_decompress_ptr cinfo, byte_t *mem, int len)
{
    cinfo->src = (struct jpeg_source_mgr *)
	(*cinfo->mem->alloc_small)((j_common_ptr) cinfo,
				   JPOOL_PERMANENT,
				   sizeof(struct jpeg_source_mgr));
    cinfo->src->init_source = jpg_noop;
    cinfo->src->fill_input_buffer = jpg_fill_input_buffer;
    cinfo->src->skip_input_data = jpg_skip_input_data;
    cinfo->src->resync_to_restart = jpeg_resync_to_restart;
    cinfo->src->term_source = jpg_noop;
    cinfo->src->bytes_in_buffer = len;
    cinfo->src->next_input_byte = mem;
}
	
int BSPDataModel::jpg_readtex(const char *fname, byte_t **rgb, int *w, int *h, int *format)
{
    struct jpeg_decompress_struct cinfo;
    struct jpeg_error_mgr jerr;
    byte_t *img, *c;
    int jpglen;

	jpglen = shader_readfile(fname, IMG_BUFSIZE, (byte_t *)imgbuf);
    if (!jpglen) return 0;

    cinfo.err = jpeg_std_error(&jerr);
    jpeg_create_decompress(&cinfo);
    jpeg_mem_src(&cinfo, (byte_t *)imgbuf, jpglen);
    jpeg_read_header(&cinfo, TRUE);
    jpeg_start_decompress(&cinfo);

    if (cinfo.output_components != 3)
	Error("Bad number of jpg components");

    img = c = (byte_t *)malloc(cinfo.output_width * cinfo.output_height * 3);
    while (cinfo.output_scanline < cinfo.output_height)
    {
	jpeg_read_scanlines(&cinfo, &c, 1);
	c += cinfo.output_width * 3;
    }

    *rgb = img;
    *w = cinfo.output_width;
    *h = cinfo.output_height;
    *format = GL_RGB;

    jpeg_finish_decompress(&cinfo);
    jpeg_destroy_decompress(&cinfo);
    
    return 1;
}

/* end of texture methods, begin lightmaps */
void BSPDataModel::lightmap_loadobjs(void)
{
    int i, texsize = (128*128*3);
    
    r_numlightmaptex = r_lightmapsize / texsize;
    r_lightmaptex = (uint_t *)malloc(r_numlightmaptex * sizeof(uint_t));
    glGenTextures(r_numlightmaptex, r_lightmaptex);
    
    for (i=0; i < r_numlightmaptex; ++i)
    {
	if (r_gamma > 1.0)
	{
	    int j, val;
	    byte_t *c;

	    c = &r_lightmapdata[i * texsize];
	    for (j=0; j < texsize; j++, c++)
	    {
		val = *c * r_gamma;
		if (val > 255) val = 255;
		*c = val;
	    }
	}
	
	glBindTexture(GL_TEXTURE_2D, r_lightmaptex[i]);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);

	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, 128, 128, 0, GL_RGB,
		     GL_UNSIGNED_BYTE, &r_lightmapdata[i * texsize]);
    }

    /* We don't need this data any more */
    free(r_lightmapdata);
}

void BSPDataModel::lightmap_freeobjs(void)
{
    glDeleteTextures(r_numlightmaptex, r_lightmaptex);
    free(r_lightmaptex);
}

/* end of lightmaps */

#endif //#ifndef SERVERCODE

const char * BSPDataModel::getRootPath(){
	return rootPath;
}

#ifdef SERVERCODE

int BSPDataModel::find_cluster(vec3_t pos)
{
    node_t *node;
    int cluster = -1;
    int leaf = -1;
    
    node = &(r_nodes[0]);

    /* Find the leaf/cluster containing the given position */
    
    while (1)
    {
	if (classify_point(pos, node->plane) > 0)
	{
	    if (node->children[0] < 0)
	    {
		leaf = -(node->children[0] + 1);
		break;
	    }
	    else
	    {
		node = &(r_nodes[node->children[0]]);
	    }
	}
	else
	{
	    if (node->children[1] < 0)
	    {
		leaf = -(node->children[1] + 1);
		break;
	    }
	    else
	    {
		node = &(r_nodes[node->children[1]]);
	    }
	}	    
    }

    if (leaf >= 0)
	cluster = r_leafs[leaf].cluster;
    return cluster;
}


int BSPDataModel::classify_point(vec3_t p, int plane_n)
{
    /* Determine which side of plane p is on */
    plane_t *plane = &(r_planes[plane_n]);

    return (vec_dot(p, plane->vec) < plane->offset ? -1 : 1);
}
#endif