// PPokemonAI.cpp: implementation of the PPokemonAI class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "PPokemonAI.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

/* this component of the AI is a neural net which is used to recognize patterns in the player's
   actions.  Observations of the environment "fire" certain event nodes (input nodes), which 
   are then processed by the hidden action nodes via the Query function.  The result of the action 
   nodes is then processed by the output node, which returns the action with the maximum activation
   level.  The network is represented as a 3 dimensional array, with each action node connected
   to NUM_EVENTS * MAX_DIFF_VALUES event nodes.
*/


PPokemonAI::PPokemonAI()
{
	for(int i=0; i<NUM_ACTIONS; i++) {
		for(int j=0; j<NUM_EVENTS; j++) {
			for(int k=0; k<MAX_DIFF_VALUES; k++) {
				aProbTable[i][j][k] = 0.5f; // initialize to equal probabilities for all moves
			}
		}
	}

	pModel = 0;
	bLearn = true;
}

PPokemonAI::~PPokemonAI()
{

}


void PPokemonAI::SetModel(PPokemon *pModel)
{
	this ->pModel = pModel;
}

// returns the action with the highest score given the observed environment
AI_MOVE PPokemonAI::GuessMove()
{
	ObserveEvents();

	AI_MOVE mcMaxAction = AI_MOVE_NOTHING;
	float fMaxScore = Query(mcMaxAction);

	float score;
	for(int i=mcMaxAction - 1; i>=0; i--) {
		score = Query((AI_MOVE) i);
		if(score > fMaxScore) {
			fMaxScore = score;
			mcMaxAction = (AI_MOVE) i;
		}
	}
	
	return mcMaxAction;

}

// observe events in the environment, firing the appropriate nodes in the net
void PPokemonAI::ObserveEvents()
{
	PPokemon * pTarget = pModel ->GetTarget();
	if(!pTarget) return;
	float fEnemyDistance = pModel ->Distance(pTarget);

	PVector vToTarget = pTarget ->GetCoordinates();
	vToTarget.subtract(&pModel ->GetCoordinates());

	PVector pDirection = pModel ->GetDirectionVector();
	float fAngleToEnemy = pDirection.AngleTo(&vToTarget);

	// orientation of the pokemon to its target in varying angle intervals
	if(fAngleToEnemy <= 0.175f || fAngleToEnemy > 6.11f) {
		aEvents[0] = 0;
	} else if(fAngleToEnemy > 0.175f && fAngleToEnemy <= 1.571f) {
		aEvents[0] = 1;
	} else if(fAngleToEnemy > 1.571f && fAngleToEnemy <= 3.14f) {
		aEvents[0] = 2;
	} else if(fAngleToEnemy > 3.14f && fAngleToEnemy <= 4.712f) {
		aEvents[0] = 3;
	} else if(fAngleToEnemy > 4.712f && fAngleToEnemy <= 6.11f) {
		aEvents[0] = 4;
	} else {
		aEvents[0] = 5;
	}
	
	// distance from target in varying intervals
	if(fEnemyDistance <= 2.5f) {
		aEvents[1] = 0;
	} else if(fEnemyDistance > 2.5f && fEnemyDistance <= 7.5f) {
		aEvents[1] = 1;
	} else if(fEnemyDistance > 7.5 && fEnemyDistance <= 12.5f) {
		aEvents[1] = 2;
	} else {
		aEvents[1] = 3;
	}

	PVector pTargetDirection = pTarget ->GetDirectionVector();
	vToTarget.invert();
	float fAngleFromEnemy = pTargetDirection.AngleTo(&vToTarget);
	// direction target is facing in varying intervals
	if(fAngleFromEnemy <= 0.523f || fAngleFromEnemy > 5.76f) {
		aEvents[2] = 0;
	} else if(fAngleFromEnemy > 0.523f && fAngleFromEnemy <= 1.571f) {
		aEvents[2] = 1;
	} else if(fAngleFromEnemy > 1.571f && fAngleFromEnemy <= 2.618f) {
		aEvents[2] = 2;
	} else if(fAngleFromEnemy > 2.618f && fAngleFromEnemy <= 3.665f) {
		aEvents[2] = 3;
	} else if(fAngleFromEnemy > 3.665f && fAngleFromEnemy <= 4.712f) {
		aEvents[2] = 4;
	} else if(fAngleFromEnemy > 4.712f && fAngleFromEnemy <= 5.76f) {
		aEvents[2] = 5;
	}

	float fProjectileIntersect = pTarget ->GetBoundVolume() ->RaySphereIntersect(
		& pModel ->GetCoordinates(), & pModel ->GetDirectionVector());
	
	// distance of potential attack collision in varying intervals
	if(fProjectileIntersect == -1.0f) {
		aEvents[3] = 0;
	} else if(fProjectileIntersect <= 1.0f) {
		aEvents[3] = 1;
	} else if(fProjectileIntersect > 1.0f && fProjectileIntersect <= 5.0f) {
		aEvents[3] = 2;
	} else if(fProjectileIntersect > 5.0f && fProjectileIntersect < 10.0f) {
		aEvents[3] = 3;
	} else {
		aEvents[3] = 4;
	}

}

// guess the move given the current network, compare to player's actual moves, 
// and update the weights as necessary
void PPokemonAI::Evaluate(AI_MOVE aiMove)
{
	if(!bLearn) return;
	if(!pModel) return;
	if(rand() % 3) return; // learning isn't perfectly elastic to help with noise

	AI_MOVE guess = GuessMove();

	if(guess == aiMove) { // increase weights if answer is correct
		AdjustProb(aiMove, true);
	} else {
		AdjustProb(guess, false); // decrease weights for the incorrect guess
		AdjustProb(aiMove, true); // increase weights for the actual move
	}
}

// the hidden nodes representing each possible action: returns the total score,
// which is a multiplication of all the weights
float PPokemonAI::Query(AI_MOVE aiMove)
{
	float fProb = 1.0f;

	for(int i=0; i<NUM_EVENTS; i++) {
		fProb *= aProbTable[aiMove][i][aEvents[i]];
	}

	return fProb;
}

// updates weights the environment links to the action node
void PPokemonAI::AdjustProb(AI_MOVE aiMove, bool correct)
{
	if(aiMove == AI_MOVE_NOTHING) return;
	float fScore = correct ? 0.01f: -0.01f;
	for(int i=0; i<NUM_EVENTS; i++) {
		aProbTable[aiMove][i][aEvents[i]] += fScore;
		if(aProbTable[aiMove][i][aEvents[i]] > 1.0f) aProbTable[aiMove][i][aEvents[i]] = 1.0f;
		else if(aProbTable[aiMove][i][aEvents[i]] < 0.0f) aProbTable[aiMove][i][aEvents[i]] = 0.0f;
	}
}

void PPokemonAI::ToggleLearning(bool bLearn)
{
	this ->bLearn = bLearn;
}

bool PPokemonAI::IsLearning()
{
	return bLearn;
}
