// PPokemon.cpp: implementation of the PPokemon class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "PPokemon.h"
#include "PProjectile.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

PPokemon::PPokemon()
{
	otObjType = OBJ_TYPE_POKEMON;
	fCurrentLifeEnergy = fLifeEnergy = fAttackEnergy = fSpeedEnergy = 100.0f;
	fDamage = fLifeRecovery = 0.0f;
	fDamageRate = fAttackRecovery = fSpeedRecovery = 20.0f;
	
	groundBound = true;
	sideBound = true;
	pTarget = 0;

	fRollRotation = fFlipRotation = 0.0f;
	for(int i=0; i<NUM_MOVES; i++) aMoves[i] = 0;
	SetAI(false);
	strcpy(strName, "");
}

PPokemon::~PPokemon()
{

}

char * PPokemon::GetName()
{
	return strName;
}

int PPokemon::GetStat(POKEMON_STAT stat)
{
	switch(stat) {
	case PS_POKEMON: return iCharacter;
	case PS_LEVEL: return iLevel;
	case PS_EXPERIENCE: return iExperience;
	case PS_NEXTLEVEL: return iNextLevel;
	case PS_HP: return iHP;
	case PS_ATTACK: return iAttack;
	case PS_SPEED: return iSpeed;
	default:
		return -1;
	}
}

void PPokemon::SetStat(POKEMON_STAT stat, int value)
{
	switch(stat) {
	case PS_LEVEL: iLevel = value; break;
	case PS_EXPERIENCE: iExperience = value; break;
	case PS_NEXTLEVEL: iNextLevel = value; break;
	case PS_HP: iHP = value; fLifeEnergy = iHP; fCurrentLifeEnergy = iHP; break;
	case PS_ATTACK: iAttack = value; break;
	case PS_SPEED: iSpeed = value; break;
	}
}

GLuint PPokemon::GetTextureID()
{
	return iTextureID;
}

PPokemon * PPokemon::GetTarget()
{
	return pTarget;
}

float PPokemon::GetFlipRotation()
{
	return fFlipRotation;
}

bool PPokemon::IsBlocking()
{
	return aMoves[MC_MOVE_BLOCK];
}


void PPokemon::Move()
{
	if(aMoves[MC_MOVE_FLIP]) Move(MC_MOVE_FLIP);
	else if(aMoves[MC_MOVE_LEFT]) Move(MC_MOVE_LEFT);
	else if(aMoves[MC_MOVE_RIGHT]) Move(MC_MOVE_RIGHT);
	else if(aMoves[MC_MOVE_AI]) MoveAI();
	else if(aMoves[MC_MOVE_BLOCK]) {
		ChangeEnergy(ET_ATTACK, -50.0f * pGame ->GetTimeElapsed());
		return;
	} 

	if(!bAIOn) {
		if(aMoves[MC_MOVE_FORWARD]) {
			if(aMoves[MC_ROTATE_LEFT]) pGame ->EvaluateAI(AI_FORWARD_LEFT);
			else if(aMoves[MC_ROTATE_RIGHT]) pGame ->EvaluateAI(AI_FORWARD_RIGHT);
			else pGame ->EvaluateAI(AI_MOVE_FORWARD);
		} else if(aMoves[MC_MOVE_BACK]) {
			if(aMoves[MC_ROTATE_RIGHT]) pGame ->EvaluateAI(AI_BACK_RIGHT);
			else if(aMoves[MC_ROTATE_LEFT]) pGame ->EvaluateAI(AI_BACK_LEFT);
		} else if(aMoves[MC_ROTATE_LEFT]) pGame ->EvaluateAI(AI_ROTATE_LEFT);
		else if(aMoves[MC_ROTATE_RIGHT]) pGame ->EvaluateAI(AI_ROTATE_RIGHT);
		else pGame ->EvaluateAI(AI_MOVE_NOTHING);

	}

	if(aMoves[MC_MOVE_FORWARD]) Move(MC_MOVE_FORWARD);
	else if(aMoves[MC_MOVE_BACK]) Move(MC_MOVE_BACK);

	if(aMoves[MC_ROTATE_LEFT]) Move(MC_ROTATE_LEFT);
	else if(aMoves[MC_ROTATE_RIGHT]) Move(MC_ROTATE_RIGHT);
}

void PPokemon::Move(MOVE_CODE mcMove)
{
	float fTimeElapsed = pGame ->GetTimeElapsed();
	switch(mcMove) {
	case MC_MOVE_FLIP:
		ChangeEnergy(ET_SPEED, -50.0f * fTimeElapsed);
		fVelocity = fRunVelocity;
		if(fFlipRotation + 500.0f * fTimeElapsed >= 180.0f) {
			fFlipRotation = 0.0f;
			aMoves[MC_MOVE_FLIP] = false;
			Rotate(AXIS_Z, 180.0f);
		} else {
			fFlipRotation += 500.0f * fTimeElapsed;
			PMovableObject::Move(MC_MOVE_FORWARD);
		}
		break;
	case MC_MOVE_RIGHT:
		ChangeEnergy(ET_SPEED, -50.0f * fTimeElapsed);
		fVelocity = fRunVelocity;
		if(fRollRotation - 1000.0f * fTimeElapsed <= -360.0f) {
			fRollRotation = 0.0f;
			aMoves[MC_MOVE_RIGHT] = false;
		} else {
			fRollRotation -= 1000.0f * fTimeElapsed;
			PMovableObject::Move(MC_MOVE_RIGHT);
		}
		break;
	case MC_MOVE_LEFT:
		ChangeEnergy(ET_SPEED, -50.0f * fTimeElapsed);
		fVelocity = fRunVelocity;
		if(fRollRotation + 1000.0f * fTimeElapsed >= 360.0f) {
			fRollRotation = 0.0f;
			aMoves[MC_MOVE_LEFT] = false;
		} else {
			fRollRotation += 1000.0f * fTimeElapsed;
			PMovableObject::Move(MC_MOVE_LEFT);
		}
		break;
	case MC_MOVE_BACK:
		fVelocity = 0.5f * fRunVelocity;
		PMovableObject::Move(MC_MOVE_BACK);
		break;
	case MC_MOVE_FORWARD:
		ChangeEnergy(ET_SPEED, -50.0f * fTimeElapsed);
		if(fSpeedEnergy == 0.0f) fVelocity = 0.5f * fRunVelocity + 5.0f * 0.005f * iSpeed;
		else fVelocity = fRunVelocity + 5.0f * 0.005f * iSpeed;
	default: PMovableObject::Move(mcMove);
	}
}


void PPokemon::RecoverEnergy()
{
	float fTimeElapsed = pGame ->GetTimeElapsed();
	
	// damage application = to life recovery rate, can change this
	float fAppliedDamage = fDamageRate * fTimeElapsed;
	if(fDamage - fAppliedDamage < 0) fAppliedDamage = fDamage;
	if(fCurrentLifeEnergy - fAppliedDamage < fLifeEnergy) {
		fCurrentLifeEnergy = fLifeEnergy;
	} else {
		fCurrentLifeEnergy -= fAppliedDamage;
	}
	fDamage -= fAppliedDamage;

	ChangeEnergy(ET_ATTACK, fAttackRecovery * fTimeElapsed);
	ChangeEnergy(ET_SPEED, fSpeedRecovery * fTimeElapsed);
}

float PPokemon::GetEnergy(ENERGY_TYPE eType)
{
	switch(eType) {
	case ET_LIFE: return fLifeEnergy;
	case ET_CURRENT_LIFE: 
		return fCurrentLifeEnergy;
	case ET_ATTACK: return fAttackEnergy;
	case ET_SPEED: return fSpeedEnergy;
	default:
		return 0.0;
	}
}

void PPokemon::ChangeEnergy(ENERGY_TYPE eType, float value)
{
	float max;
	float * ePtr = 0;
	switch(eType) {
	case ET_LIFE: ePtr = & fLifeEnergy; max = iHP; break;
	case ET_ATTACK: ePtr = & fAttackEnergy; max = 100.0f; break;
	case ET_SPEED: ePtr = & fSpeedEnergy; max = 100.0f; break;
	}

	if(!ePtr) return;
	float tmp = * ePtr;
	tmp += value;
	if(tmp > 100.0f) tmp = 100.0f;
	if(tmp < 0.0f) tmp = 0.0f;
	* ePtr = tmp;
}

bool PPokemon::AIOn()
{
	return bAIOn;
}

void PPokemon::SetAI(bool bAIOn)
{
	this ->bAIOn = bAIOn;
	aMoves[MC_MOVE_AI] = bAIOn;
	if(bAIOn) iAgression = rand() % (iLevel * 2);
}

void PPokemon::TakeDamage(DAMAGE_TYPE dtType, float fDamage)
{
	fCurrentLifeEnergy = fLifeEnergy;
	if(aMoves[MC_MOVE_BLOCK] && fAttackEnergy > 0.0f) fDamage = 0.0f;
	this ->fDamage += fDamage;
	if(fLifeEnergy - fDamage <= 0.0) {
		fLifeEnergy = 0.0f;
		pGame ->CheckGameStatus();
	}
	else fLifeEnergy -= fDamage;
}


void PPokemon::ToggleMove(MOVE_CODE mcMove, bool value)
{
	if((mcMove == MC_MOVE_FLIP || mcMove == MC_MOVE_LEFT || mcMove == MC_MOVE_RIGHT) 
		&& value == true && fSpeedEnergy < 5.0f) return;
	aMoves[mcMove] = value;
}


void PPokemon::SetTarget(PPokemon *pTarget)
{
	this ->pTarget = pTarget;
}

bool PPokemon::IsAlive()
{
	return fLifeEnergy > 0.0f;
}

void PPokemon::GainLevel()
{
	if(iLevel < 99) {
		iLevel++;
		if(iLevel < 99) {
			iNextLevel += iLevel * 100;
		} else {
			iNextLevel = -1;
		}
		iHP += 1 + rand() % 5;
		iAttack += 1 + rand() % 2;
		iSpeed += 1 + rand() % 2;
	} else {
		iNextLevel = -1;
	}
}

// AI functions

void PPokemon::MoveAI() {
	if(aMoves[MC_MOVE_BLOCK] && (clock() - tBlock) / CLK_TCK < 0.05f) return;

	AI_GOAL goal = GoalAI();
	AI_MOVE guess = pGame ->GuessMove();

	PVector vTargetLoc = AdjustTargetLoc(guess);

	float fEnemyDistance = Distance(pTarget);
	
	PVector vToTarget = pTarget ->GetCoordinates();
	vToTarget.subtract(&loc);
	float fAngleToTarget = pDirection.AngleTo(&vToTarget);

	aMoves[MC_MOVE_BLOCK] = false;
	if(goal == AI_ATTACK) {
		// first look to attack
		if(AttackAI(&vTargetLoc)) {
			MoveTowardsTargetAI(&vTargetLoc);
			return;
		} 
		// next look to block
		else if(guess == AI_MOVE_ATTACK && EnemyThreatAI()) {
			if(fAttackEnergy > 15.0f && fEnemyDistance < 2.0f && MoveAllowedAI(MC_MOVE_FLIP)) {
				MoveSpecialAI(MC_MOVE_BLOCK);
				tBlock = clock();
			}
			else {
				if(!(rand() % 2) && MoveAllowedAI(MC_MOVE_LEFT)) MoveSpecialAI(MC_MOVE_LEFT);
				else if(MoveAllowedAI(MC_MOVE_RIGHT)) MoveSpecialAI(MC_MOVE_RIGHT);
				MoveTowardsTargetAI(&vTargetLoc);
			}
			return;
		}
		
		if(fEnemyDistance < 3.0f) {
			if(fAngleToTarget > 0.70 && fAngleToTarget < 0.873) {
				if(MoveAllowedAI(MC_MOVE_RIGHT)) {
					MoveSpecialAI(MC_MOVE_RIGHT);
					return;
				}
			} 
			if(fAngleToTarget > 5.41 && fAngleToTarget < 5.59) {
				if(MoveAllowedAI(MC_MOVE_LEFT)) {
					MoveSpecialAI(MC_MOVE_LEFT);
					return;
				}
			}
		}
		if(fEnemyDistance < 1.0f && fAngleToTarget > 2.61 && fAngleToTarget < 3.665f) {
			if(MoveAllowedAI(MC_MOVE_FLIP)) {
				MoveSpecialAI(MC_MOVE_FLIP);
				return;
			}
		} else { 
			MoveTowardsTargetAI(&vTargetLoc);
		}
	} else {
		// defense strategy

		// first look to defend against attack
		if(guess == AI_MOVE_ATTACK && EnemyThreatAI()) { // randomly dodge
			if(!(rand() % 2) && MoveAllowedAI(MC_MOVE_LEFT)) MoveSpecialAI(MC_MOVE_LEFT);
			else if(MoveAllowedAI(MC_MOVE_RIGHT)) MoveSpecialAI(MC_MOVE_RIGHT);
			return;
		}

		if(goal == AI_WAIT) {
			AttackAI(&vTargetLoc);
			// turn to face enemy
			if(fAngleToTarget > 2.61 && fAngleToTarget < 3.665f) {
				if(MoveAllowedAI(MC_MOVE_FLIP)) {
					MoveSpecialAI(MC_MOVE_FLIP);
					return;
				}
			} else if(fAngleToTarget > M_PI) {
				Move(MC_ROTATE_LEFT);
			} else if(fAngleToTarget < M_PI) {
				Move(MC_ROTATE_RIGHT);
			}
			return;
		}
		// next look to run away
		if(goal == AI_DEFEND && fSpeedEnergy > 0.0f && fEnemyDistance < 10.0f) {
			if(fSpeedEnergy > 20.0f && vToTarget.dotProduct(&pDirection2D) > 0.0f) {
				if(MoveAllowedAI(MC_MOVE_FLIP)) {
					MoveSpecialAI(MC_MOVE_FLIP);
					return;
				}
			} else {
				switch(rand()%3) {
				case 0: Move(MC_MOVE_FORWARD);
				case 1: Move(MC_ROTATE_RIGHT); Move(MC_MOVE_FORWARD);
				case 2: Move(MC_ROTATE_LEFT); Move(MC_MOVE_FORWARD);
				}
			}
		} else { // finally, just want to face enemy
			if(fAngleToTarget > 2.61 && fAngleToTarget < 3.665f) {
				if(MoveAllowedAI(MC_MOVE_FLIP)) {
					MoveSpecialAI(MC_MOVE_FLIP);
					return;
				}
			} else if(fAngleToTarget > M_PI) {
				Move(MC_ROTATE_LEFT);
			} else if(fAngleToTarget < M_PI) {
				Move(MC_ROTATE_RIGHT);
			}
		}
	}
}

// returns AI "strategy" based on current conditions and agression level
AI_GOAL PPokemon::GoalAI()
{
	float fTotal = fAttackEnergy + 0.5f * fSpeedEnergy + iAgression;

	if(fTotal >= 125.0f) {
		return AI_ATTACK;
	}
	if(fTotal >= 100.0f) {
		return AI_WAIT;
	}
	return AI_DEFEND;
}

// use move forwards and rotations to move towards target
void PPokemon::MoveTowardsTargetAI(PVector * vTarget)
{
	PVector vTargetDirection = FindPathAI(vTarget);

	float fAngleToTarget = pDirection2D.AngleTo(& vTargetDirection);

	if(fAngleToTarget > M_PI && fAngleToTarget > 0.017f) {
		Move(MC_ROTATE_LEFT);
	} else if(fAngleToTarget < M_PI) {
		Move(MC_ROTATE_RIGHT);
	} 
	
	fAngleToTarget = pDirection2D.AngleTo(&vTargetDirection);
	if(fAngleToTarget < 0.131f || fAngleToTarget > 6.152f) {
		Move(MC_MOVE_FORWARD);
	}
}



bool PPokemon::AttackAI(PVector * vTarget)
{
	if(pTarget ->IsBlocking()) return false;
	float fTargetDistance = loc.dist(vTarget);
	int bRandAttack = (int) fTargetDistance * 2 + 1; // closer you get the higher the probability of attack
	// don't attack if you see enemy blocking
	if(!pTarget ->IsBlocking() && rand()%bRandAttack) return false;  

	// if enemy in line of fire, attack
	PBoundVolume pTargetVolume = *pTarget ->GetBoundVolume();
	pTargetVolume.SetCoordinates(vTarget ->x, vTarget ->y, vTarget ->z);
	if(pTargetVolume.RaySphereIntersect(&loc, & pDirection, 0.5f) > 0) {
		if(fTargetDistance < 1.0f && fAttackEnergy >= 20.0f) {
			Attack(AC_CLOSE_ATTACK);
		} else if(fTargetDistance < 7.5f) {
			Attack(AC_NORMAL_ATTACK);
		}
	}
	return true;
}

// adjust the target location depending on the move guessed.
PVector PPokemon::AdjustTargetLoc(AI_MOVE move)
{
	float dir;
	float fVelocity = pTarget ->GetVelocity();
	float fTimeElapsed = pGame ->GetTimeElapsed();
	PVector tLoc = pTarget ->GetCoordinates();
	PVector pDirection = pTarget ->GetDirectionVector();

	switch(move) {
	case AI_ROTATE_RIGHT:
	case AI_ROTATE_LEFT:
	case AI_MOVE_NOTHING:
	case AI_MOVE_BLOCK:
	case AI_MOVE_ATTACK:
		// location does not change
		break;
	case AI_MOVE_BACK:
		dir = -0.5f;
	case AI_MOVE_FORWARD:
		tLoc.x += pDirection.x * fVelocity * dir * fTimeElapsed;
		tLoc.y += pDirection.y * fVelocity * dir * fTimeElapsed;
		break;
	case AI_BACK_RIGHT:
		dir = - 0.5f;
	case AI_FORWARD_RIGHT:
		pDirection.Rotate(AXIS_Z, fRotationRate * fTimeElapsed);
		tLoc.x += pDirection.x * fVelocity * dir * fTimeElapsed;
		tLoc.y += pDirection.y * fVelocity * dir * fTimeElapsed;
		break;
	case AI_BACK_LEFT:
		dir = - 0.5f;
	case AI_FORWARD_LEFT:
		pDirection.Rotate(AXIS_Z, -fRotationRate * fTimeElapsed);
		tLoc.x += pDirection.x * fVelocity * dir * fTimeElapsed;
		tLoc.y += pDirection.y * fVelocity * dir * fTimeElapsed;
		break;
	case AI_MOVE_RIGHT:
		dir = -1.0f;
	case AI_MOVE_LEFT:
		tLoc.x -= pDirection.y * fVelocity * dir * 2.0f;
		tLoc.y += pDirection.x * fVelocity * dir * 2.0f;
		break;
	case AI_MOVE_FLIP:
		tLoc.x += pDirection.x * fVelocity * 2.0f;
		tLoc.y += pDirection.y * fVelocity * 2.0f;
		break;
	}
	tLoc.z = pGame->ground->GetHeight(tLoc.x, tLoc.y);

	return tLoc;
}

// uses a heuristic (A*) search to find path around obstacles to target
// takes one right, and one left branch from the forward direction, taking unit steps on each
// branch trying to turn towards the target whenever possible
PVector PPokemon::FindPathAI(PVector * vTarget)
{
	CPtrArray * pProjectiles = pGame ->GetObjects(OBJ_TYPE_PROJECTILE);
	PProjectile * pProjectile;
	
	int iRightSteps = 0;
	int iLeftSteps = 0;
	float hWidth = width * 0.5f;

	PVector vTempTarget = * vTarget;	// copy of target for height adjustment
	PVector vLeftLoc = loc;				// current location in left search
	PVector vRightLoc = loc;			// current location in right search
	//PVector vLeftShifted;				// shifted locations so back intersection is ignored
	//PVector vRightShifted;				

	PVector vLeftPath;					// result vector towards left path
	PVector vRightPath;					// result vector towards right path

	PVector vCurLeft;					// current direction on left path
	PVector vCurRight;					// current direction on right path

	float fIntersectLeft;				// distance to intersection
	float fIntersectRight;			
	
	bool collisionLeft ;				
	bool collisionRight;

	bool leftIncremented;
	bool rightIncremented;

	float rotation;

	while(true) {
		vCurLeft = *vTarget;
		vCurLeft.subtract(& vLeftLoc);
		vCurLeft.z = 0.0f;	// collision tests will be horizontal on plane of the possible obstacle
		vCurLeft.makeUnit();
	
		vCurRight = *vTarget;
		vCurRight.subtract(& vRightLoc);
		vCurRight.z = 0.0f;
		vCurRight.makeUnit();
	
		//vLeftShifted = vLeftLoc;
		//vLeftShifted.Translate(hWidth * vCurLeft.x, hWidth * vCurLeft.y, 0.0f);
		//vRightShifted = vRightLoc;
		//vRightShifted.Translate(hWidth * vCurRight.x, hWidth * vCurRight.y, 0.0f);

		rotation = 0.0f;
		
		leftIncremented = false;
		rightIncremented = false;

		// start with vector pointing towards target, and rotate outwards until a legal step is found
		while(rotation < 360.0f) {
			collisionLeft = false;
			collisionRight = false;

			for(int i=0; i<pProjectiles ->GetSize(); i++) {
				// if collisions detected on both path for this angle, advance to next angle
				if(collisionLeft && collisionRight) break;

				// current obstacle to test
				pProjectile = (PProjectile *) pProjectiles ->GetAt(i);
				// don't test against attacks, these are transient obstacles
				if(pProjectile ->GetSource() == this || pProjectile ->GetSource() == pTarget) continue;

				// test left path
				// if collision already found or obstacles is too far away, skip test
				if(!collisionLeft && vLeftLoc.dist(&pProjectile ->GetCoordinates()) < 2.0f) {
					vLeftLoc.z = pProjectile ->GetCoordinate(AXIS_Z);
					vTempTarget.z = pProjectile ->GetCoordinate(AXIS_Z);
					fIntersectLeft = pProjectile ->GetBoundVolume() ->RaySphereIntersect(&vLeftLoc, &vCurLeft, hWidth);
					if(fIntersectLeft > 0.0f && fIntersectLeft + hWidth < vLeftLoc.dist(&vTempTarget)) {
						collisionLeft = true;
					}
				}
				// test right path
				if(!collisionRight && vRightLoc.dist(&pProjectile ->GetCoordinates())
					< 2.0f) {
					vRightLoc.z = pProjectile ->GetCoordinate(AXIS_Z);
					vTempTarget.z = pProjectile ->GetCoordinate(AXIS_Z);
					fIntersectRight = pProjectile ->GetBoundVolume() ->RaySphereIntersect(&vRightLoc, &vCurRight, hWidth);
					if(fIntersectRight > 0.0f && fIntersectRight + hWidth < vRightLoc.dist(& vTempTarget)) {
						collisionRight = true;
					}
				}
			}
		
			if(!collisionLeft) {
				// if rotation is 0, there was no collision on the vector from the current node to target,
				// so path is complete
				if(iLeftSteps == 0) {
					if(rotation == 0.0f) return vCurLeft;
					vLeftPath = vCurLeft;
					iLeftSteps ++;
				} else {
					if(rotation == 0.0f) {
						// make sure that this is the best path in event of tie
						if(collisionRight && 
							pDirection2D.angleSubtended(AXIS_Z, & vLeftPath) <
							pDirection2D.angleSubtended(AXIS_Z, & vRightPath)) {
							return vLeftPath;
						} 
					}
					if(!leftIncremented) {
						leftIncremented = true;
						vLeftLoc.add(& vCurLeft);
						iLeftSteps ++;
					}
				}
			} else {
				if(iLeftSteps == 0) vCurLeft.Rotate(AXIS_Z, -5.0f);
				else vCurLeft.Rotate(AXIS_Z, 5.0f);
			}
			if(!collisionRight && !rightIncremented) {
				rightIncremented = true;
				if(iRightSteps == 0) {
					if(rotation == 0.0f) return vCurRight;
					vRightPath = vCurRight;
					iRightSteps ++;
				} else {
					// make sure this is best path in event of tie
					if(collisionLeft && 
						pDirection2D.angleSubtended(AXIS_Z, & vRightPath) <
						pDirection2D.angleSubtended(AXIS_Z, & vLeftPath)) {
						return vLeftPath;
					}
					if(!rightIncremented) {
						rightIncremented = true;
						vRightLoc.add(& vCurRight);
						iRightSteps ++;
					}
				}
			} else {
				if(iRightSteps == 0) vCurRight.Rotate(AXIS_Z, 5.0f);
				else vCurRight.Rotate(AXIS_Z, -5.0f);
			}
			if(!collisionLeft && !collisionRight) break;
			else rotation += 5.0f;
		}
		if(iLeftSteps > 20) {
			return vLeftPath;
		}
		if(rotation >= 360.0f) 	break;
	}
		
	return PVector(0.0f, 0.0f, 0.0f); // if all else fails....
}

// toggles one of the special moves if none of them are currently active
void PPokemon::MoveSpecialAI(MOVE_CODE mcMove)
{
	if(aMoves[MC_MOVE_BLOCK] || aMoves[MC_MOVE_FLIP] || aMoves[MC_MOVE_LEFT] || aMoves[MC_MOVE_RIGHT] || fSpeedEnergy < 5.0f)
		return;
	else aMoves[mcMove] = true;
}

void PPokemon::GenerateStatsAI(int iLevel)
{
	this ->iLevel = iLevel;
	iHP = 30;
	iAttack = 10;
	iSpeed = 10;
	for(int i=0; i<iLevel; i++) {
		iHP += 1 + rand() % 5;
		iAttack += 1 + rand() % 2;
		iSpeed += 1 + rand()%2;
	}
	fLifeEnergy = iHP;
	fCurrentLifeEnergy = iHP;
}

// if the move is not outside the bounds and does not result in a collision
bool PPokemon::MoveAllowedAI(MOVE_CODE mcMove)
{
	float dir = 1.0f;
	float x, y;
	switch(mcMove) {
		case MC_MOVE_RIGHT:
			dir = -1.0f;
		case MC_MOVE_LEFT:
			x = loc.x - pDirection.y * fVelocity * dir * 1.5f;
			y = loc.y + pDirection.x * fVelocity * dir * 1.5f;
			break;
		case MC_MOVE_FLIP:
			x = pDirection.x * fVelocity * 1.5f;
			y = pDirection.y * fVelocity * 1.5f;
			break;
	}
	if(pGame ->ground ->GetHeight(x,y)<0.1) {
		return false;
	} else {
		PBoundVolume tVolume = pVolume;
		CPtrArray aCollisions;
		pVolume.SetCoordinates(x,y,pGame ->ground ->GetHeight(x,y));
		pGame ->DetectCollisions(this, &tVolume, &aCollisions);
		if(aCollisions.GetSize()>0) return false;
		else return true;
	}
}

// if enemy is facing us
bool PPokemon::EnemyThreatAI()
{
	PVector vToThis = loc;
	vToThis.subtract(&pTarget ->GetCoordinates());
	if(pTarget ->GetDirectionVector().angleSubtended(AXIS_Z,&vToThis) < 0.523) 
		return true;
	else 
		return false;
}
