// PGame.cpp: implementation of the PGame class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "PGame.h"

#include "PStartMenu.h"
#include "PCharacterMenu.h"
#include "POptionsMenu.h"

#include "PCamera.h"

#include "PProjectile.h"
#include "Pokemon/PPikachu.h"
#include "Pokemon/PCharmander.h"
#include "Pokemon/PSquirtle.h"

#include "PSound.h"
#include "gltext.h"

#include "PPokemonAI.h"
#include "PFlameObstacle.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

PStartMenu * pStartMenu;
PCharacterMenu * pCharMenu;
POptionsMenu * pOptionsMenu;
PCamera * pCamera;
PSound * pSound;
PPokemonAI pAI;

HWND hActive;

PGame::PGame()
{
	culler = NULL;

	bGameOn = false;
	glCullFace(GL_BACK);
	gsState = GS_START_MENU;
	goOptions.difficulty=1;
	goOptions.numEnemies=1;
	goOptions.sound=1;

	pCamera = 0;
	SetUpMenus();
	ground = 0;
	stage=0;
	goOptions.difficulty = 2;
	goOptions.lastLevel = -1;

	hActive=FindWindow(NULL,"Pokemon Battle Ground!");
	pSound= new PSound(hActive);
	pSound->InitSounds();
	pSound->PlaySegment("battle.mid",true);
}

PGame::~PGame()
{
	delete pStartMenu;
	delete pCharMenu;
	delete pOptionsMenu;
	delete pSound;
}

void PGame::SetUpMenus()
{
	pStartMenu = new PStartMenu(this);
	pCharMenu = new PCharacterMenu(this);
	pOptionsMenu = new POptionsMenu(this);
}

void PGame::Draw()
{
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	int i;

	switch(gsState) {
	case GS_START_MENU: pStartMenu ->Draw(); break;
	case GS_CHARACTER_MENU: pCharMenu ->Draw(); break;
	case GS_OPTIONS_MENU: pOptionsMenu ->Draw(); break;
	case GS_GAME_COUNTDOWN:
		if(GetTimer() >= 3) {
			for(i=0; i<aPokemon.GetSize(); i++) {
				((PPokemon * ) aPokemon.GetAt(i)) ->Move(MC_MOVE_NOTHING);
			}
			gsState = GS_GAME;
			tLast = clock();
			bGameOn = true;
			glDrawText(iWindowWidth * 0.45f, iWindowHeight * 0.25f, "Fight!", FONT_COUNTDOWN);
		}
		UpdateGameState();
		for(i=0; i<aPokemon.GetSize(); i++) {
			((PPokemon * ) aPokemon.GetAt(i)) ->Move(MC_MOVE_FALL);
		}
		pCamera ->Draw(); break;
	case GS_GAME_VICTORY:
		if(GetTimer() >= 1) {
			gsState = GS_GAME_EXPERIENCE;
		}
	case GS_GAME:
	case GS_GAME_EXPERIENCE:
		UpdateGameState();
		pCamera ->Draw(); break;

	}
	
	if(!pAI.IsLearning()) {
		glColor3f(1.0, 1.0, 1.0);
		glDrawText(5, iWindowHeight - 20, "Learning off", FONT_POKEMON_NAME);
	}
	glutSwapBuffers();
}

void PGame::UpdateGameState()
{
	PPokemon * pCurPokemon;
	PProjectile * pCurProjectile;
	
	clock_t curTime = clock();

	fTimeElapsed = (curTime - tLast) / (float) CLOCKS_PER_SEC;
	tLast = curTime;
	if(fTimeElapsed > 0.15f) fTimeElapsed = 0.1f;

	
	for(int i=0; i < aPokemon.GetSize(); i++) {
		pCurPokemon = (PPokemon *)aPokemon.GetAt(i);
		pCurPokemon->RecoverEnergy();
		if(pCurPokemon ->IsAlive()) {
			if(bGameOn) pCurPokemon ->Move();
		}
	}

	for(i=0; i < aProjectiles.GetSize(); i++) {
		pCurProjectile = (PProjectile *)aProjectiles.GetAt(i);
		pCurProjectile ->Move(MC_MOVE_FORWARD);
	}

	PMovableObject * pCurObject;	
	for(i=0; i < aEnvironmentObjects.GetSize(); i++) {
		pCurObject = (PMovableObject *)aEnvironmentObjects.GetAt(i);
		if(stage == 2) pCurObject ->Move(MC_MOVE_FLOAT);
		else pCurObject ->Move(MC_MOVE_RAIN);
	}

	EmptyGarbage();
}


void PGame::AddObject(PDrawableObject * pObject)
{
	switch(pObject ->GetObjectType()) {
	case OBJ_TYPE_POKEMON: 
		aPokemon.Add(pObject); 
		break;
	case OBJ_TYPE_PROJECTILE: 
		aProjectiles.Add(pObject);
		break;
	case OBJ_TYPE_ENVIRONMENT:
		aEnvironmentObjects.Add(pObject);
		break;
	default: 
		aWorldObjects.Add(pObject);
	}

}

void PGame::RemoveObject(PDrawableObject *pObject)
{
	CPtrArray * pObjectArray;

	switch(pObject ->GetObjectType()) {
	case OBJ_TYPE_POKEMON:
		pObjectArray = &aPokemon; break;
	case OBJ_TYPE_PROJECTILE:
		pObjectArray = &aProjectiles; break;
	case OBJ_TYPE_ENVIRONMENT:
		pObjectArray = &aEnvironmentObjects; break;
	default:
		pObjectArray = &aWorldObjects; break;
	}
	
	if(pObjectArray == 0) return;
	for(int i=0; i<pObjectArray ->GetSize(); i++) {
		if(pObjectArray ->GetAt(i) == pObject) {
			pObjectArray ->RemoveAt(i);
			if(pObject ->GetObjectType() == OBJ_TYPE_PROJECTILE ||
			   pObject ->GetObjectType() == OBJ_TYPE_ENVIRONMENT) {
				aGarbage.Add(pObject);
			}
			break;
		}
	}
}

CPtrArray * PGame::GetObjects(OBJECT_TYPE ocType)
{
	switch(ocType) {
	case OBJ_TYPE_POKEMON: return &aPokemon;
	case OBJ_TYPE_PROJECTILE: return &aProjectiles;
	case OBJ_TYPE_ENVIRONMENT: return &aEnvironmentObjects;
	default: return &aWorldObjects;
	}
}

float abs(float num) {
	if(num>0) return num;
	return -num;

}


void PGame::DetectCollisions(PDrawableObject * pObject, PBoundVolume *pVolume, CPtrArray *pCollidingObjects)
{
	PPokemon * pCurPokemon;
	PBoundVolume * pCurVolume;

	for(int i=0; i<aPokemon.GetSize(); i++) {
		pCurPokemon = (PPokemon *) aPokemon.GetAt(i); 
		if(pObject ->GetObjectType() == OBJ_TYPE_PROJECTILE && ((PProjectile *) pObject) ->GetSource() == pCurPokemon) continue;
		pCurVolume = pCurPokemon ->GetBoundVolume();
		if(pCurPokemon == pObject)
			continue;
		else if(pVolume ->GetMaxLength() + pCurVolume ->GetMaxLength() < pVolume ->Distance(pCurVolume))
			continue; 
		else if(pVolume ->DetectIntersection(pCurVolume))
			pCollidingObjects ->Add(pCurPokemon);
	}	
	PProjectile * pCurProjectile;
	for(i=0; i<aProjectiles.GetSize(); i++) {
		pCurProjectile = (PProjectile *) aProjectiles.GetAt(i); 
		if(pObject ->GetObjectType() == OBJ_TYPE_POKEMON && pCurProjectile ->GetSource() == pObject) continue;
		pCurVolume = pCurProjectile ->GetBoundVolume();
		if(pCurProjectile == pObject)
			continue;
		else if(pVolume ->GetMaxLength() + pCurVolume ->GetMaxLength() < pVolume ->Distance(pCurVolume))
			continue; 
		else if(pVolume ->DetectIntersection(pCurVolume))
			pCollidingObjects ->Add(pCurProjectile);
	}

	PDrawableObject * pCurObject;
	for(i=0; i<aWorldObjects.GetSize(); i++) {
		pCurObject = (PDrawableObject *) aWorldObjects.GetAt(i);
		pCurVolume = pCurObject ->GetBoundVolume();
		if(pCurObject == pObject)
			continue;
		else if(pVolume ->GetMaxLength() + pCurVolume ->GetMaxLength() < pVolume ->Distance(pCurVolume))
			continue; 
		else if(pVolume ->DetectIntersection(pCurVolume))
			pCollidingObjects ->Add(pCurObject);
	}
}

void PGame::StopGame()
{
	pSound->Stop();
	pSound->DestroySounds();
	bGameOn = false;
	if(goOptions.sound==1) pSound->PlaySegment("battle.mid",true);
}

void PGame::Keyboard(unsigned char key, int x, int y)
{
	switch(gsState) {
	case GS_START_MENU: pStartMenu ->Keyboard(key, x, y); break;
	case GS_CHARACTER_MENU: pCharMenu ->Keyboard(key, x, y); break;
	case GS_OPTIONS_MENU: pOptionsMenu ->Keyboard(key, x, y); break;
	case GS_GAME: 
	case GS_GAME_VICTORY:
	case GS_GAME_EXPERIENCE:
		pCamera ->Keyboard(key, x, y); break;
	}
}

void PGame::KeyboardUp(unsigned char key, int x, int y)
{
	switch(gsState) {
		case GS_GAME: pCamera ->KeyboardUp(key, x, y); break;
	}
}

void PGame::SetGameState(GAME_STATE gsState)
{
	this ->gsState = gsState;
}

PGame::SetGameOption(GAME_OPTION goOption, int val)
{
	switch(goOption) {
	case GO_DIFFICULTY: goOptions.difficulty = val; break;
	case GO_NUM_ENEMIES: goOptions.numEnemies = val; break;
	case GO_SOUND: 
		pSound->Stop();
		if(val==1) pSound->PlaySegment("battle.mid",true);
		goOptions.sound = val; 
		break;
	}
}

int PGame::GetGameOption(GAME_OPTION goOption)
{
	switch(goOption) {
	case GO_DIFFICULTY: return goOptions.difficulty;
	case GO_NUM_ENEMIES: return goOptions.numEnemies;
	case GO_SOUND: return goOptions.sound;
	default: return -1;
	}
}


void PGame::StartGame()
{
	CleanUp();
	
	int r;
	while( (r=rand()%4)==goOptions.lastLevel) {
		// can't be same level as last one
	}
	
	goOptions.lastLevel = r;
	switch(r) {
	case 0: stage=0; break;
	case 1: stage=1; break;
	case 2: stage=2; break;
	case 3: stage=3; break;
	}

	culler = new PFrustumCuller();
	ground = new PTerrain(this);
	
	if(stage==2) {
		PFlameObstacle * pFlame;
		int x, y;
		for(int i=0; i<5; i++) {
			pFlame = new PFlameObstacle(this);
			x = 5 + rand() % 20;
			y = 5 + rand() % 20;
			pFlame ->SetCoordinates(x,y,ground ->GetHeight(x,y));
			AddObject(pFlame);
		}
	}

	PPokemon * pPlayer;
	PPikachu *pPikachu;
	PCharmander *pCharmander;
	PSquirtle *pSquirtle;

	srand(time(NULL));
	gsState = GS_GAME_COUNTDOWN;
	StartTimer();
	if(goOptions.sound==1) {
		pSound->Stop();
		if(stage==0) {
			pSound->PlaySound(SOUND_STREAM,true,100);
			pSound->PlaySegment("fighting.mid",true);
		} else if(stage==1) {
			pSound->PlaySound(SOUND_WIND,true,5);
			pSound->PlaySegment("fighting3.mid",true);
		} else if(stage==2) {
			pSound->PlaySound(SOUND_LAVA,true,100);
			pSound->PlaySegment("fighting2.mid",true);
		} else if(stage==3) {
			pSound->PlaySound(SOUND_RAIN,true,15);
			pSound->PlaySegment("fighting4.mid",true);
		}
	}
	if(csStats.character==0) {
		pPikachu =new PPikachu(this, 10.0, 10.0);
		pPlayer = pPikachu;
	} else if(csStats.character==1) {
		pCharmander=new PCharmander(this, 10.0,10.0);
		pPlayer = pCharmander;
	} else if(csStats.character==2) {
		pSquirtle =new PSquirtle(this, 10.0, 10.0);
		pPlayer = pSquirtle;
	}
	pPlayer ->SetCoordinate(AXIS_Z, 20.0f);
	pPlayer ->SetStat(PS_LEVEL, csStats.level);
	pPlayer ->SetStat(PS_EXPERIENCE, csStats.experience);
	pPlayer ->SetStat(PS_NEXTLEVEL, csStats.next);
	pPlayer ->SetStat(PS_HP, csStats.hp);
	pPlayer ->SetStat(PS_ATTACK, csStats.attack);
	pPlayer ->SetStat(PS_SPEED, csStats.speed);

	pCamera = new PCamera(this, pPlayer, w, h);
	pAI.SetModel(pPlayer);
	AddObject(pPlayer);

	int iLevel;
	PPokemon * pEnemy;
	float rx,ry;
	
	rx=5 + rand()%20;
	ry=5 + rand()%20;
	while(rx>8 && rx<12) rx=rand()%20;
	while(ry>8 && ry<12) ry=rand()%20;
	for(int i=0; i<goOptions.numEnemies; i++) {
		int iEnemy = rand() % 3;
		switch(iEnemy) {
		case 0:
			pPikachu = new PPikachu(this, rx, ry);
			pEnemy = pPikachu;
			break;
		
		case 1:
			pCharmander = new PCharmander(this, rx, ry);
			pEnemy = pCharmander;
			break;
		case 2:
			pSquirtle = new PSquirtle(this, rx, ry);
			pEnemy = pSquirtle;
			break;
		}
		switch(goOptions.difficulty) {
		case 1: // any level at or below 
			iLevel = rand() % csStats.level + 1;
			break;
		case 2: // up to five levels above
			iLevel = csStats.level + rand() % 5; 
			break;
		case 3: // 5 to 20 levels above
			iLevel = 5 + csStats.level + rand() % 15;
			break;
		}
		if(iLevel > 99) iLevel = 99;
		pEnemy ->SetCoordinate(AXIS_Z, 20.0f);
		pEnemy ->GenerateStatsAI(iLevel);
		pEnemy ->SetAI(true);
		AddObject(pEnemy);
		pPlayer ->SetTarget(pEnemy);
		pEnemy ->SetTarget(pPlayer);
	}

	PVector vToEnemy = pEnemy ->GetCoordinates();
	PVector vToPlayer = pPlayer ->GetCoordinates();
	vToEnemy.subtract(&vToPlayer);
	float fAngle = pPlayer ->GetDirectionVector().AngleTo(&vToEnemy);
	pPlayer ->Rotate(AXIS_Z, DEGREE(fAngle));
	vToEnemy.invert();
	fAngle = pEnemy ->GetDirectionVector().AngleTo(&vToEnemy);
	pEnemy ->Rotate(AXIS_Z, DEGREE(fAngle));

}

void PGame::ResizeWindow(int w, int h)
{
	this ->w = w; this ->h = h;
	if(pCamera != 0) pCamera ->ResizeWindow(w, h);
}


PSound *PGame::GetSoundObject() {
	return pSound;
}

PDrawableObject *PGame::GetFocusObject() {
	return pCamera->GetFocusObject();
}

void PGame::CheckGameStatus()
{
	PPokemon * pPokemon;

	int numAlive = 0;
	int numHuman = 0;

	for(int i=0; i<aPokemon.GetSize(); i++) {
		pPokemon = (PPokemon *) aPokemon.GetAt(i);
		if(pPokemon ->IsAlive()) {
			numAlive ++;
			if(!pPokemon ->AIOn()) 
				numHuman ++;
		}
	}
	if(numAlive <= 1 || numHuman ==0) {
		gsState = GS_GAME_VICTORY;
		StopGame();
		pCamera ->SetExperience();
		StartTimer();
	}
}

int PGame::GetStage() {
	return stage;
}

int PGame::GetTimer()
{	
	return ( (int) (clock() - countStart) / CLK_TCK);
}

void PGame::StartTimer()
{	
	countStart = clock();
}

GAME_STATE PGame::GetGameState()
{
	return gsState;
}

float PGame::GetTimeElapsed()
{
	return fTimeElapsed;
}

void PGame::CleanUp()
{
	pAI.SetModel(0);
	PDrawableObject * pObject;
	for(int i=aPokemon.GetSize()-1; i>=0; i--) {
		pObject = (PDrawableObject *) aPokemon.GetAt(i);
		aPokemon.RemoveAt(i);
		if(pObject) delete pObject;
	}

	for(i=aProjectiles.GetSize()-1; i>=0; i--) {
		pObject = (PDrawableObject *) aProjectiles.GetAt(i);
		aProjectiles.RemoveAt(i);
		if(pObject) delete pObject;
	}

	for(i=aGarbage.GetSize()-1; i>=0; i--) {
		pObject = (PDrawableObject *) aGarbage.GetAt(i);
		aGarbage.RemoveAt(i);
		if(pObject) delete pObject;
	}

	for(i=aWorldObjects.GetSize()-1; i>=0; i--) {
		pObject = (PDrawableObject *) aWorldObjects.GetAt(i);
		aWorldObjects.RemoveAt(i);
		if(pObject) delete pObject;
	}

	for(i=aEnvironmentObjects.GetSize()-1; i>=0; i--) {
		pObject = (PDrawableObject *) aEnvironmentObjects.GetAt(i);
		aEnvironmentObjects.RemoveAt(i);
		if(pObject) delete pObject;
	}

	if(ground) delete ground;
	if(culler) delete culler;
}

void PGame::QuitGame()
{
	CleanUp();
	exit(0);
}

void PGame::EvaluateAI(AI_MOVE aiMove)
{
	pAI.Evaluate(aiMove);
}


AI_MOVE PGame::GuessMove()
{
	return pAI.GuessMove();
}

void PGame::ToggleLearning()
{
	pAI.ToggleLearning(!pAI.IsLearning());
}

void PGame::SetCharacter(POKEMON_STAT stat, int value)
{
	switch(stat) {
	case PS_POKEMON: csStats.character = value; break;
	case PS_LEVEL: csStats.level = value; break;
	case PS_EXPERIENCE: csStats.experience = value; break;
	case PS_NEXTLEVEL: csStats.next = value; break;
	case PS_HP: csStats.hp = value; break;
	case PS_ATTACK: csStats.attack = value; break;
	case PS_SPEED: csStats.speed = value; break;
	}
}

void PGame::Save()
{
	FILE *fp;

	fp = fopen("pokemon.sav", "w");

	fprintf(fp, "%d\n", csStats.character);
	fprintf(fp, "%d\n", csStats.level);
	fprintf(fp, "%d\n", csStats.experience);
	fprintf(fp, "%d\n", csStats.next);
	fprintf(fp, "%d\n", csStats.hp);
	fprintf(fp, "%d\n", csStats.attack);
	fprintf(fp, "%d\n", csStats.speed);
	
	fclose(fp);
}

void PGame::EmptyGarbage()
{
	if(aGarbage.GetSize() < 100) return;
	PObject * pObject;
	
	for(int i=aGarbage.GetSize()-1; i>=0; i--) {
		pObject = (PObject *) aGarbage.GetAt(i);
		delete pObject;
	}
	aGarbage.RemoveAll();
}

