// PCamera.cpp: implementation of the PCamera class.
//
//////////////////////////////////////////////////////////////////////
// Object representing the camera that follows the player's character
// The camera determines what the world looks like.  In other words,
// it is in charge of all the transformations for rendering the world correctly
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "PCamera.h"
#include "PPokemon.h"
#include "gltext.h"
//#include "PExperienceWindow.h"

#define CAMERA_HEIGHT 3.0f
#define CAMERA_DISTANCE -5.0f
#define CAMERA_FOCUS_POINT 10.0f

//PExperienceWindow pWindow;
//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

PCamera::PCamera(PGame * pGame, PPokemon * pFocusObject, int w, int h):
	pExpWind(pGame)
{
	this ->pGame = pGame;
	this ->pFocusObject = pFocusObject;
	this ->w = w;
	this ->h = h;
}

PCamera::~PCamera()
{

}

void PCamera::Draw()
{

	float x, y, z;
	x = pFocusObject ->GetCoordinate(AXIS_X);
	y = pFocusObject ->GetCoordinate(AXIS_Y);
	z = pFocusObject ->GetCoordinate(AXIS_Z);

	GLfloat fAspectRatio = (GLfloat) w/h;



	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glFrustum(-1.0f, 1.0f, -1.0f / fAspectRatio, 1.0f / fAspectRatio, 1.5f, 200.0f);
		
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	
	// move camera behind and slightly above character and points towards distant horizon
	gluLookAt(0.0f, CAMERA_DISTANCE, CAMERA_HEIGHT, 0.0f, CAMERA_FOCUS_POINT, 0.0f, 0.0f, 1.0f, 0.0f);

	// rotate world so the character is facing forward
	glRotatef((90.0f + (GLfloat) pFocusObject ->GetDirection()), 0.0f, 0.0f, 1.0f);
	if(pFocusObject ->GetFlipRotation()) 
		glRotatef(pFocusObject ->GetFlipRotation(), 0.0f, 0.0f, 1.0f);

	// translate world so the character is in the center of the screen
	if(pGame ->GetGameState() == GS_GAME_COUNTDOWN) {
		glTranslatef( -x, -y, -pGame ->ground ->GetHeight(x,y));
	} else {
		glTranslatef( -x, -y, -z);
	}
	
	GLfloat light_ambient[] = { 1.0, 1.0, 1.0, 1.0 };
	GLfloat light_diffuse[] = { 1.0, 1.0, 1.0, 1.0 };
	if(pGame->GetStage()==1) {
		light_ambient[0]=.8;
		light_ambient[1]=.8;
		light_ambient[2]=.8;
		light_ambient[3]=1.0;
		light_diffuse[0]=.8;
		light_diffuse[1]=.8;
		light_diffuse[2]=.8;
		light_diffuse[3]=.8;
	} else if(pGame->GetStage()==2) {
		light_ambient[0]=1.0;
		light_ambient[1]=.6;
		light_ambient[2]=.6;
		light_ambient[3]=1.0;
		light_diffuse[0]=1.0;
		light_diffuse[1]=.6;
		light_diffuse[2]=.6;
		light_diffuse[3]=.8;
	} else if(pGame->GetStage()==3) {
		light_ambient[0]=.6;
		light_ambient[1]=.6;
		light_ambient[2]=.6;
		light_ambient[3]=1.0;
		light_diffuse[0]=.6;
		light_diffuse[1]=.6;
		light_diffuse[2]=.6;
		light_diffuse[3]=.8;
	}

	GLfloat light_specular [] = { 1.0, 1.0, 1.0, 1.0 };
	GLfloat light_position[] = {0.5, 0.5, 10, 0.0};

	glLightModelfv(GL_LIGHT_MODEL_AMBIENT, light_ambient);
	glLightfv (GL_LIGHT0, GL_POSITION, light_position);
	glLightfv(GL_LIGHT0, GL_AMBIENT_AND_DIFFUSE, light_diffuse);
	glLightfv(GL_LIGHT0, GL_SPECULAR, light_specular);
	glLightfv(GL_LIGHT0, GL_POSITION, light_position);
	
	glEnable(GL_LIGHTING);
	glEnable(GL_DEPTH_TEST);
	glEnable(GL_LIGHT0);
	
	pGame->culler->DrawDecision();

	pGame->ground->Draw();

	DrawObjectArray(pGame ->GetObjects(OBJ_TYPE_POKEMON));
	DrawObjectArray(pGame ->GetObjects(OBJ_TYPE_PROJECTILE));
	DrawObjectArray(pGame ->GetObjects(OBJ_TYPE_WORLD));
	DrawObjectArray(pGame ->GetObjects(OBJ_TYPE_ENVIRONMENT));

	glDisable(GL_LIGHTING);
	glDisable(GL_LIGHT0);
	glDisable(GL_DEPTH_TEST);
	DrawRadar();
	DrawEnergyBars();


	GAME_STATE gsState = pGame ->GetGameState();
	if(pGame ->GetStage() != 0) {
		glColor3f(1.0f, 1.0f, 1.0f);
	} else {
		glColor3f(0.0f, 0.0f, 0.25f);
	}
	if(gsState == GS_GAME_COUNTDOWN) {
		int iSeconds = pGame ->GetTimer();
		if(iSeconds < 3) {
			char buffer[20];
			sprintf(buffer, "%d", 3 - iSeconds);
			glDrawText(iWindowWidth * 0.5f, iWindowHeight * 0.25f, buffer, FONT_COUNTDOWN);
		} else {
			glDrawText(iWindowWidth * 0.45f, iWindowHeight * 0.25f, "Fight!", FONT_COUNTDOWN);
		}
	} else if(gsState == GS_GAME_VICTORY) {
		if(pFocusObject ->IsAlive()) glDrawText(iWindowWidth * 0.4f, iWindowHeight * 0.2f, "Victory!", FONT_GAME_RESULT);
		else glDrawText(iWindowWidth * 0.4f, iWindowHeight * 0.2f, "You Lose!", FONT_GAME_RESULT);
	} else if(gsState == GS_GAME_EXPERIENCE) {
		if(pFocusObject ->IsAlive()) pExpWind.Draw();
		else {
			glDrawText(iWindowWidth * 0.4f, iWindowHeight * 0.45f, "Game Over", FONT_GAME_RESULT);
		}
	}
}

void PCamera::DrawObjectArray(CPtrArray *pObjects)
{
	PDrawableObject * pObject;
	
	for(int i=0; i<pObjects ->GetSize(); i++) {
		pObject = (PDrawableObject *) pObjects ->GetAt(i);

		glPushMatrix();
		if(pObject->GetObjectType()==OBJ_TYPE_PROJECTILE || pObject->GetObjectType()==OBJ_TYPE_ENVIRONMENT) {
			pObject->TranslateForDrawing();
			glRotatef(-(pFocusObject->GetDirection()),0.0,0.0,1.0);
		}
		pObject ->Draw();
		glPopMatrix();
	}		
}


void PCamera::ResizeWindow(int w, int h)
{
	this ->w = w;
	this ->h = h;
}

PMovableObject * PCamera::GetFocusObject()
{
	return pFocusObject;
}

int PCamera::GetWidth()
{
	return w;
}

int PCamera::GetHeight()
{
	return h;
}

void PCamera::DrawRadar()
{
	float x, y, z;
	x = pFocusObject ->GetCoordinate(AXIS_X);
	y = pFocusObject ->GetCoordinate(AXIS_Y);
	z = pFocusObject ->GetCoordinate(AXIS_Z);

	GLfloat fAspectRatio = (GLfloat) h / w;

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluOrtho2D(-1.0, 1.0, -1.0, 1.0 * fAspectRatio);
	
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	// translate to top-right corner
	glTranslatef(0.65, 0.65 * fAspectRatio, 0.0);
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glColor4f(0.0, 0.0, 1.0, 0.25);
	glutSolidSphere(0.20, 25, 5);
	glDisable(GL_BLEND);
	// rotate so objects in front are above center, and objects below are behind
	glRotatef((90.0f + (GLfloat) pFocusObject ->GetDirection()), 0.0f, 0.0f, 1.0f);
	// translate so focus object is in center
	glTranslatef( -x * 0.01, -y * 0.01, -z * 0.01);

	CPtrArray * pObjects = pGame ->GetObjects(OBJ_TYPE_POKEMON);
	PDrawableObject * pObject;

	for(int i=0; i<pObjects ->GetSize(); i++) {
		pObject = (PDrawableObject *) pObjects ->GetAt(i);
		if(pFocusObject ->Distance(pObject) < 19.5) {
			glPushMatrix();
			glTranslatef(pObject ->GetCoordinate(AXIS_X) * 0.01,
						pObject ->GetCoordinate(AXIS_Y) * 0.01, 
						pObject ->GetCoordinate(AXIS_Z) * 0.01);
			if(pObject == pFocusObject) glColor3f(1.0, 1.0, 1.0);
			else if(pObject == pFocusObject ->GetTarget()) glColor3f(1.0f, 1.0f, 0.0f);
			else glColor3f(1.0, 0.0, 0.0);
			glRectf(-.01, -.01, 0.01, 0.01);
			glPopMatrix();
		}
	}	
}


void PCamera::DrawEnergyBars()
{
	GLfloat fAspectRatio = (GLfloat) h / w;

	GLfloat lineX = -0.85f;
	GLfloat lineY = 0.8f * fAspectRatio;
	if(pFocusObject ->GetObjectType() != OBJ_TYPE_POKEMON) return;
	PPokemon * pFocusPokemon = (PPokemon *) pFocusObject;

	float fScaler = 1.0f / 175.0f;
	float fLife;
	char buffer[20];

	glLoadIdentity();
	
	// player icon
	glEnable(GL_TEXTURE_2D);
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);
	glBindTexture(GL_TEXTURE_2D, textures[pFocusObject ->GetTextureID()]);
	glBegin(GL_QUADS);
		glTexCoord2f(0.0f,0.0f); glVertex2f(lineX - 0.1f,lineY - 0.06f * fAspectRatio);
		glTexCoord2f(1.0f,0.0f); glVertex2f(lineX - 0.01f,lineY - 0.06f * fAspectRatio);
		glTexCoord2f(1.0f,1.0f); glVertex2f(lineX - 0.01f,lineY + 0.04f * fAspectRatio);
		glTexCoord2f(0.0f,1.0f); glVertex2f(lineX - 0.1f,lineY + 0.04f * fAspectRatio);
	glEnd();
	glDisable(GL_TEXTURE_2D);

	//life bar
	glLineWidth(0.0208f * iWindowHeight);
	glBegin(GL_LINES);
		glColor3f(1.0f, 0.5f, 0.5f);
		glVertex2f(lineX, lineY);
		glVertex2f(lineX + pFocusPokemon ->GetEnergy(ET_CURRENT_LIFE) / 
					pFocusPokemon ->GetStat(PS_HP) * 100 * fScaler, lineY);
		fLife = pFocusPokemon ->GetEnergy(ET_LIFE) / pFocusPokemon ->GetStat(PS_HP);
		if(fLife < 0.25f) glColor3f(1.0f, 0.0f, 0.0f);
		else glColor3f(1.0f, 1.0f, 0.0f);
		glVertex2f(lineX, lineY);
		glVertex2f(lineX + fLife * 100 * fScaler, lineY);
	glEnd();
	
	glColor3f(0.0f, 0.0f, 0.25f);
	glDrawText(iWindowWidth * 0.08f, iWindowHeight * 0.073f, pFocusPokemon ->GetName(), FONT_POKEMON_NAME);
	if(pGame ->GetStage() != 0) {
		glColor3f(1.0f, 1.0f, 1.0f);
	}
	sprintf(buffer, "Lv. %d", pFocusPokemon ->GetStat(PS_LEVEL));
	glDrawText(iWindowWidth * 0.03f, iWindowHeight * 0.11f, buffer, FONT_POKEMON_LEVEL);

	lineX += 0.7f;

	// enemy life bar
	PPokemon * pTarget = pFocusPokemon ->GetTarget();
	if(pTarget && pTarget ->IsAlive()) {
		glEnable(GL_TEXTURE_2D);
		glBindTexture(GL_TEXTURE_2D, textures[pFocusObject ->GetTarget() ->GetTextureID()]);
		glBegin(GL_QUADS);
			glTexCoord2f(0.0f,0.0f); glVertex2f(lineX - 0.1f,lineY - 0.06f * fAspectRatio);
			glTexCoord2f(1.0f,0.0f); glVertex2f(lineX - 0.01f,lineY - 0.06f * fAspectRatio);
			glTexCoord2f(1.0f,1.0f); glVertex2f(lineX - 0.01f,lineY + 0.04f * fAspectRatio);
			glTexCoord2f(0.0f,1.0f); glVertex2f(lineX - 0.1f,lineY + 0.04f * fAspectRatio);
		glEnd();
		glDisable(GL_TEXTURE_2D);

		glBegin(GL_LINES);
			glColor3f(1.0f, 0.5f, 0.5f);
			glVertex2f(lineX, lineY);
			glVertex2f(lineX + pTarget ->GetEnergy(ET_CURRENT_LIFE) / 
					pTarget ->GetStat(PS_HP) * 100 * fScaler, lineY);
			fLife = pTarget ->GetEnergy(ET_LIFE) / pTarget ->GetStat(PS_HP);
			if(fLife < 0.25f) glColor3f(1.0, 0.0, 0.0);
			else glColor3f(1.0f, 1.0f, 0.0f);
			glVertex2f(lineX, lineY);
			glVertex2f(lineX + fLife * 100 * fScaler, lineY);
		glEnd();
	
		glColor3f(0.0f, 0.0f, 0.25f);
		glDrawText(iWindowWidth * 0.43f, iWindowHeight * 0.073f, pTarget ->GetName(), FONT_POKEMON_NAME);
		if(pGame ->GetStage() != 0) {
			glColor3f(1.0f, 1.0f, 1.0f);
		} 
		sprintf(buffer, "Lv. %d", pTarget ->GetStat(PS_LEVEL));
		glDrawText(iWindowWidth * 0.38f, iWindowHeight * 0.11f, buffer, FONT_POKEMON_LEVEL);
	} //else {
	//	GetNextTarget();
	//}
	
	// speed and attack bars
	lineX -= 0.7f;
	lineY -= 0.05f * fAspectRatio;
	glLineWidth(0.00417 * iWindowHeight);
	glBegin(GL_LINES);
		glColor3f(0.75f, 0.0f, 1.0f);
		glVertex2f(lineX, lineY);
		glVertex2f(lineX + pFocusPokemon ->GetEnergy(ET_ATTACK) * fScaler, lineY);
		lineY -= 0.02f * fAspectRatio;
		glColor3f(0.0f, 0.0f, 1.0f);
		glVertex2f(lineX, lineY);
		glVertex2f(lineX + pFocusPokemon ->GetEnergy(ET_SPEED) * fScaler, lineY);
	glEnd();
}

void PCamera::Keyboard(unsigned char key, int x, int y)
{
	GAME_STATE gsState = pGame ->GetGameState();

	if(gsState == GS_GAME_EXPERIENCE) {
		if(pFocusObject ->IsAlive()) {
			pExpWind.Keyboard(key, x, y);
		} else {
			pGame ->SetGameState(GS_START_MENU);
		}
		return;
	}

	if(gsState != GS_GAME) return;

	PPokemon * pObject = (PPokemon *) pFocusObject;
	switch(key) {
	case 'a':
	case 52: 
		pObject ->ToggleMove(MC_ROTATE_LEFT, true); 
		break;
	case 'd':
	case 54: 
		pObject ->ToggleMove(MC_ROTATE_RIGHT, true);
		break;
	case 'w':
	case 56: 
		pObject->ToggleMove(MC_MOVE_FORWARD, true);
		break;
	case 's':
	case 'x':
	case 50: 
		pObject->ToggleMove(MC_MOVE_BACK, true);
		break;
	case 49:
	case 'z':
		pGame ->EvaluateAI(AI_MOVE_LEFT);
		pObject->ToggleMove(MC_MOVE_LEFT, true);
		break;
	case 51:
	case 'c': 
		pGame ->EvaluateAI(AI_MOVE_RIGHT);
		pObject->ToggleMove(MC_MOVE_RIGHT, true);
		break;
	case 55:
	case 57:
	case 'q':
	case 'e':
		pGame ->EvaluateAI(AI_MOVE_FLIP);
		pObject ->ToggleMove(MC_MOVE_FLIP, true);
		break;
	case 13:
		pGame ->EvaluateAI(AI_MOVE_ATTACK);
		pObject ->Attack(AC_NORMAL_ATTACK); 
		break;
	case 48:
	case '\\':
		pGame ->EvaluateAI(AI_MOVE_ATTACK);
		pObject->Attack(AC_CLOSE_ATTACK);
		break;
	case '\'':
		pGame ->EvaluateAI(AI_MOVE_BLOCK);
		pObject ->ToggleMove(MC_MOVE_BLOCK, true);
		break;
	case ']':
		GetNextTarget(); break;
	case 'l':
		pGame ->ToggleLearning(); break;
	case 'f':
		pFocusObject ->GetTarget() ->SetAI(!pFocusObject ->GetTarget() ->AIOn()); break;
	case 27: pGame ->StopGame(); pGame ->SetGameState(GS_START_MENU);
	}
}

void PCamera::KeyboardUp(unsigned char key, int x, int y)
{
	PPokemon * pObject = (PPokemon *) pFocusObject;

	switch(key) {
	case 'a':
	case 52: 
		pObject ->ToggleMove(MC_ROTATE_LEFT, false); 
		break;
	case 'd':
	case 54: 
		pObject ->ToggleMove(MC_ROTATE_RIGHT, false);
		break;
	case 'w':
	case 56: 
		pObject->ToggleMove(MC_MOVE_FORWARD, false);
		break;
	case 's':
	case 'x':
	case 50: 
		pObject->ToggleMove(MC_MOVE_BACK, false);
		break;
	case 49:
	case 'z': 
		break;
	case 51:
	case 'c': 
		break;
	case '\'':
		pObject ->ToggleMove(MC_MOVE_BLOCK, false);
		break;
	}
}

// for cycling through multiple possible targets, not used in final version
void PCamera::GetNextTarget()
{
	CPtrArray * pTargets = pGame ->GetObjects(OBJ_TYPE_POKEMON);
	PPokemon * pTarget = pFocusObject ->GetTarget();

	int start = 0;
	for(int i=0; i<pTargets ->GetSize(); i++) {
		if(pTargets ->GetAt(i) == pTarget) {
			start = i;
			break;
		}
	}

	PPokemon * pCurTarget;
	PPokemon * pNewTarget = 0;
	for(i=start+1;;i++) {
		if(i>=pTargets ->GetSize()) {
			i=0; 
		}
		if(i==start) break;
		else {
			pCurTarget = (PPokemon *) pTargets ->GetAt(i);
			if(pCurTarget ->IsAlive() && pCurTarget!=pTarget && pCurTarget!=pFocusObject) {
				pNewTarget = pCurTarget;
				break;
			}
		}
	}
	
	pFocusObject ->SetTarget(pNewTarget);
}

void PCamera::SetExperience()
{
	if(!pFocusObject ->IsAlive()) return;
	int temp, iExperience = 0;
	CPtrArray * pPokemon = pGame ->GetObjects(OBJ_TYPE_POKEMON);
	PPokemon * pCurPokemon;
	for(int i=0; i<pPokemon ->GetSize(); i++) {
		pCurPokemon = (PPokemon *) pPokemon ->GetAt(i);
		if(pCurPokemon != pFocusObject) {
			temp = pCurPokemon ->GetStat(PS_LEVEL) - pFocusObject ->GetStat(PS_LEVEL) + 1;
			if(temp < 0) { // if Pokemon is of lower level
				temp = (pCurPokemon ->GetStat(PS_LEVEL) + rand() % 5) * 10 / -temp;
			} else {
				temp *= (pCurPokemon ->GetStat(PS_LEVEL) + rand() % 5) * 10;
			}
			
			iExperience += temp;
		}	
	}
	pExpWind.Initialize(pFocusObject, iExperience);
}
