(* wall.ml: implementation of rendering of walls *)

type t = Stone | Blocks | Tiles | SlimePuddle

let tex = Array.make 4 Nativeint.zero
and texgrey = Array.make 4 Nativeint.zero


(* the basic command for drawing a wall - just a quad *)
let drawwall textures texid =
  if textures then begin
    Gl.enable `texture_2d;
    GlTex.env (`mode `modulate);
    Tex.bind texid
  end else ();
  GlDraw.begins `quads;
  GlDraw.normal ~x:0.95 ~y:0.22079402 ~z:0.22079402 ();
  if textures then GlTex.coord2 (1.0,1.0) else ();
  GlDraw.vertex ~x:0.0 ~y:0.5 ~z:0.5 ();
  GlDraw.normal ~x:0.95 ~y:0.22079402 ~z:(-0.22079402) ();
  if textures then GlTex.coord2 (0.0,1.0) else ();
  GlDraw.vertex ~x:0.0 ~y:0.5 ~z:(-0.5) ();
  GlDraw.normal ~x:0.95 ~y:(-0.22079402) ~z:(-0.22079402) ();
  if textures then GlTex.coord2 (0.0,0.0) else ();
  GlDraw.vertex ~x:0.0 ~y:(-0.5) ~z:(-0.5) ();
  GlDraw.normal ~x:0.95 ~y:(-0.22079402) ~z:0.22079402 ();
  if textures then GlTex.coord2 (1.0,0.0) else ();
  GlDraw.vertex ~x:0.0 ~y:(-0.5) ~z:0.5 ();
  GlDraw.ends ();
  if textures then Gl.disable `texture_2d else ()


(* initialize walls for rendering *)
let init () =
  (* set up textures for walls *)
  let texnames = [| "Data/stone.ppm"; "Data/blocks.ppm"; "Data/tiles.ppm";
                    "Data/slimepuddle.ppm" |] in
  for i = 0 to 3 do
    tex.(i) <- Tex.loadPPM texnames.(i);
    if tex.(i) = Nativeint.zero then begin
      prerr_endline ("Problem loading wall image " ^ texnames.(i));
      exit 0
    end else ()
  done;
  let texnames = [| "Data/stone_grey.ppm"; "Data/blocks_grey.ppm";
                    "Data/tiles_grey.ppm"; "Data/slimepuddle_grey.ppm" |] in
  for i = 0 to 3 do
    texgrey.(i) <- Tex.loadPPM texnames.(i);
    if texgrey.(i) = Nativeint.zero then begin
      prerr_endline ("Problem loading wall image " ^ texnames.(i));
      exit 0
    end else ()
  done


(* render at canonical coordinates: (0,-.5,-.5) to (0,.5,.5) with front
 * facing the positive x axis *)
let render w textureson greyscaleon alpha =
  match w with
    Stone ->
      begin
        if textureson then       GlDraw.color ~alpha (1.0,1.0,1.0)
        else if greyscaleon then GlDraw.color ~alpha (0.45,0.45,0.45)
        else                     GlDraw.color ~alpha (0.45,0.4,0.4);
        GlLight.material ~face:`front (`shininess 1.0);
        if greyscaleon then begin
          GlLight.material ~face:`front
            (`ambient_and_diffuse(0.45,0.45,0.45,alpha));
          GlLight.material ~face:`front (`specular(0.1,0.1,0.1,alpha));
        end else begin
          GlLight.material ~face:`front
            (`ambient_and_diffuse(0.45,0.4,0.4,alpha));
          GlLight.material ~face:`front (`specular(0.1,0.1,0.0,alpha));
        end;
        drawwall textureson (if greyscaleon then texgrey.(0) else tex.(0))
      end
  | Blocks ->
      begin
        if textureson then       GlDraw.color ~alpha (1.0,1.0,1.0)
        else if greyscaleon then GlDraw.color ~alpha (0.65,0.65,0.65)
        else                     GlDraw.color ~alpha (0.65,0.6,0.5);
        GlLight.material ~face:`front (`shininess 3.0);
        if greyscaleon then begin
          GlLight.material ~face:`front
            (`ambient_and_diffuse(0.65,0.65,0.65,alpha));
          GlLight.material ~face:`front (`specular(0.1,0.1,0.1,alpha));
        end else begin
          GlLight.material ~face:`front
            (`ambient_and_diffuse(0.65,0.6,0.5,alpha));
          GlLight.material ~face:`front (`specular(0.2,0.1,0.0,alpha));
        end;
        drawwall textureson (if greyscaleon then texgrey.(1) else tex.(1))
      end
  | Tiles ->
      begin
        if textureson then       GlDraw.color ~alpha (1.0,1.0,1.0)
        else if greyscaleon then GlDraw.color ~alpha (0.8,0.8,0.8)
        else                     GlDraw.color ~alpha (0.85,0.8,0.75);
        GlLight.material ~face:`front (`shininess 3.0);
        if greyscaleon then begin
          GlLight.material ~face:`front
            (`ambient_and_diffuse(0.8,0.8,0.8,alpha));
          GlLight.material ~face:`front (`specular(1.0,1.0,1.0,alpha));
        end else begin
          GlLight.material ~face:`front
            (`ambient_and_diffuse(0.85,0.8,0.75,alpha));
          GlLight.material ~face:`front (`specular(1.0,1.0,1.0,alpha));
        end;
        drawwall textureson (if greyscaleon then texgrey.(2) else tex.(2))
      end
  | SlimePuddle ->
      begin
        if textureson then       GlDraw.color ~alpha (1.0,1.0,1.0)
        else if greyscaleon then GlDraw.color ~alpha (0.85,0.85,0.85)
        else                     GlDraw.color ~alpha (0.8,0.85,0.75);
        GlLight.material ~face:`front (`shininess 5.0);
        if greyscaleon then begin
          GlLight.material ~face:`front
            (`ambient_and_diffuse(0.85,0.85,0.85,alpha));
          GlLight.material ~face:`front (`specular(1.0,1.0,1.0,alpha));
        end else begin
          GlLight.material ~face:`front
            (`ambient_and_diffuse(0.8,0.85,0.75,alpha));
          GlLight.material ~face:`front (`specular(1.0,1.0,1.0,alpha));
        end;
        drawwall textureson (if greyscaleon then texgrey.(3) else tex.(3))
      end

