(* ui.ml: The game user interface *)

open Game
open Tk

(* create the main toplevel window *******************************************)
let maketop =
  let top = openTk () in
  Wm.aspect_set top ~minnum:4 ~mindenom:3 ~maxnum:4 ~maxdenom:3;
  Wm.title_set top "Spaceman Spiff in Escape from Zorg";
  top


(* state values for the UI to keep track of **********************************)
let paused = ref true                   (* is the game paused? *)

let varlightson =                       (* is lighting in use? *)
  let v = Textvariable.create () in
  Textvariable.set v "On";
  v

let lights = ref true

let vartextureson =                     (* are textures in use? *)
  let v = Textvariable.create () in
  Textvariable.set v "On";
  v

let textures = ref true

let varstereoon =                       (* is stereo mode on? *)
  let v = Textvariable.create () in
  Textvariable.set v "Off";
  v

let stereo = ref false


(* the mapping of X events to game events ************************************)
let pause_ev = [`KeyPressDetail("Escape")]
let startfwd_ev =   [`KeyPressDetail("Up")]
let startback_ev =  [`KeyPressDetail("Down")]
let startleft_ev =  [`KeyPressDetail("Left")]
let startright_ev = [`KeyPressDetail("Right")]
let startjump_ev =  [`KeyPressDetail("KP_Insert")]
let endfwd_ev =   [`KeyReleaseDetail("Up")]
let endback_ev =  [`KeyReleaseDetail("Down")]
let endleft_ev =  [`KeyReleaseDetail("Left")]
let endright_ev = [`KeyReleaseDetail("Right")]
let endjump_ev =  [`KeyReleaseDetail("KP_Insert")]


(* reset command *************************************************************)
let resetfun s () =
  Game.restart s


(* helpbox window ************************************************************)
let abouttext =
" Spaceman Spiff\n" ^
"            in\n" ^
"Escape from Zorg!\n\n" ^
"By Robert Bridson\n" ^
"for CS248, Fall 2000\n" ^
"(Prof. Marc Levoy)\n\n" ^
"Dedicated to Jonathan,\n" ^
"who couldn't wait to\n" ^
"see the world"

let bgtext =
"You are the fearless Spaceman Spiff, held captive in the dungeons of " ^
"Zorg after crashlanding (yet again) on the hostile desert planet. After " ^
"a fierce battle with the evil Zorgoids, you were overcome, only to wake " ^
"up later in a smelly prison cell with nothing more than your clothes " ^
"... and your carefully hidden Dematterizer.\n\n" ^
"The Dematterizer, when fired, swings a chunk of matter into the Fifth " ^
"Dimension --- temporarily. It soon swings back into reality, so it's a " ^
"good idea to not be in the space it occupied. In particular be careful " ^
"about falling into dematterized pits, since in your weakened condition " ^
"you won't be able to jump out.\n\n" ^
"Beware of guards patrolling the dungeons, who will " ^
"easily overcome your puny Earthling physique and lock you up again if they " ^
"can catch you. The Zorgoids have a limited patience for escape attempts; " ^
"get caught too many times and they will freeze you in carbonated scum " ^
"forever (a.k.a. Game Over). However, you may find the occasional treasure " ^
"chest, with which you can bribe the guards into giving you another " ^
"chance.\n\n" ^
"To advance from the depths of the dungeon to the surface though the " ^
"intermediate levels, you will need to " ^
"use the teleporters. To activate a " ^
"teleporter, you need to collect 9 charges (they look like batteries)."

let controltext =
"The screen is divided into the world view (what Spiff is seeing), a " ^
"display of the remaining chances for escape on the right, an indicator " ^
"showing the remaining charge in the dematterizer on the bottom, and a " ^
"panel in the bottom right showing teleporter charges collected so far.\n\n" ^
"While in game mode (entered by selecting 'Start' or 'Resume' in the Game " ^
"menu) the following controls are active:\n\n" ^
"   mouse movement - rotate Spiff\n" ^
"   mouse button -       fire dematterizer\n" ^
"   cursor keys -          walk around\n" ^
"   0/Insert -                jump\n" ^
"   Escape -                 pause game\n"


let helpbox name text top () =
  let helptop = Toplevel.create ~name top in
  Wm.title_set helptop name;
  let mesg = Message.create ~text helptop in
  pack [mesg]


(* quit command **************************************************************) let quitfun () = closeTk ()


(* handle the menubar ********************************************************)
let rec makemenu ?(first = false) s top togl =
  if not first then begin
    (* pause game *)
    paused := true;
    Mouse.game_mode_off ();
    bind ~events:pause_ev ~action:(fun ev -> ()) top;
    bind ~events:startfwd_ev   ~action:(fun ev -> ()) top;
    bind ~events:startback_ev  ~action:(fun ev -> ()) top;
    bind ~events:startleft_ev  ~action:(fun ev -> ()) top;
    bind ~events:startright_ev ~action:(fun ev -> ()) top;
    bind ~events:startjump_ev  ~action:(fun ev -> ()) top;
    bind ~events:endfwd_ev   ~action:(fun ev -> ()) top;
    bind ~events:endback_ev  ~action:(fun ev -> ()) top;
    bind ~events:endleft_ev  ~action:(fun ev -> ()) top;
    bind ~events:endright_ev ~action:(fun ev -> ()) top;
    bind ~events:endjump_ev  ~action:(fun ev -> ()) top;
    (* change main window's title bar to something descriptive *)
    Wm.title_set top "Spaceman Spiff in Escape from Zorg (paused)";
  end else ();

  (* create menu bar *)
  let menubar = Frame.create ~borderwidth:2 ~relief:`Raised top in
  let gameb = Menubutton.create ~text:"Game" menubar
  and optionb = Menubutton.create ~text:"Options" menubar
  and helpb = Menubutton.create ~text:"Help" menubar in
  let gamemenu = Menu.create ~tearoff:false gameb
  and optionmenu = Menu.create ~tearoff:false optionb
  and helpmenu = Menu.create ~tearoff:false helpb in

  (* start command - refers to makemenu, so has to be defined here *)
  let startfun () = begin
    destroy menubar;

    (* set up callbacks for game *)
    Mouse.game_mode_on ~poscb:(Spiff.movemouse s.spiff)
                       ~butcb:(Game.pressbutton s);
    bind ~events:pause_ev ~action:(fun ev -> makemenu s top togl) top;
    bind ~events:startfwd_ev   ~action:(fun ev-> Spiff.startfwd s.spiff) top;
    bind ~events:startback_ev  ~action:(fun ev-> Spiff.startback s.spiff) top;
    bind ~events:startleft_ev  ~action:(fun ev-> Spiff.startleft s.spiff) top;
    bind ~events:startright_ev ~action:(fun ev-> Spiff.startright s.spiff) top;
    bind ~events:startjump_ev  ~action:(fun ev-> Spiff.startjump s.spiff) top;
    bind ~events:endfwd_ev   ~action:(fun ev -> Spiff.endfwd s.spiff) top;
    bind ~events:endback_ev  ~action:(fun ev -> Spiff.endback s.spiff) top;
    bind ~events:endleft_ev  ~action:(fun ev -> Spiff.endleft s.spiff) top;
    bind ~events:endright_ev ~action:(fun ev -> Spiff.endright s.spiff) top;
    bind ~events:endjump_ev  ~action:(fun ev-> Spiff.endjump s.spiff) top;

    (* let the user know how to get back to the menu *)
    Wm.title_set top
   "Spaceman Spiff in Escape from Zorg (press Escape for pause and game menu)";

    (* let the game state know we just started *)
    Game.resumeplaying s;
    paused := false;
  end in

  (* create the game menu entries *)
  Menu.add_command gamemenu ~label:(if first then "Start" else "Resume")
                            ~command:startfun;
  Menu.add_command gamemenu ~label:"Reset" ~command:(resetfun s);
  Menu.add_command gamemenu ~label:"Quit" ~command:quitfun;

  (* create the option menu entries *)
  Menu.add_checkbutton optionmenu ~label:"Lighting" ~indicatoron:true
    ~variable:varlightson ~offvalue:"Off" ~onvalue:"On"
    ~command:(fun () -> lights := (Textvariable.get varlightson)="On");
  Menu.add_checkbutton optionmenu ~label:"Textures" ~indicatoron:true
    ~variable:vartextureson ~offvalue:"Off" ~onvalue:"On"
    ~command:(fun () -> textures := (Textvariable.get vartextureson)="On");
  Menu.add_checkbutton optionmenu ~label:"Stereo" ~indicatoron:true
    ~variable:varstereoon ~offvalue:"Off" ~onvalue:"On"
    ~command:(fun () -> stereo := (Textvariable.get varstereoon)="On");

  (* create the help menu entries *)
  Menu.add_command helpmenu ~label:"About..."
                            ~command:(helpbox "About" abouttext top);
  Menu.add_command helpmenu ~label:"Background..."
                            ~command:(helpbox "Background" bgtext top);
  Menu.add_command helpmenu ~label:"Controls..."
                            ~command:(helpbox "Controls" controltext top);

  (* get the menus working *)
  Menubutton.configure gameb ~menu:gamemenu;
  Menubutton.configure optionb ~menu:optionmenu;
  Menubutton.configure helpb ~menu:helpmenu;

  (* display them *)
  pack ~side:`Left [gameb; optionb; helpb];
  place ~height:32 ~x:0 ~y:0 ~relwidth:1.0 menubar


(* configure the GL drawing area in the main canvas **************************)
let configuretogl togl s top =
  Togl.display_func togl ~cb:(fun () -> Game.render s togl !lights !textures
                                                       !stereo);
  Togl.reshape_func togl ~cb:(fun () -> Game.render s togl !lights !textures
                                                       !stereo);
  (* register the callback that will be called every 11 milliseconds (if
   * possible). It updates the game state, handles special actions, and
   * re-renders. *)
  Togl.timer_func ~ms:11
    ~cb:(fun () ->
           if !paused then ()
           else begin
             let result = Game.update s in
             begin match result with
               Game.WonGame ->    makemenu s top togl;
                                  ignore (Dialog.create ~parent:top
                                    ~title:"You win!"
                                    ~message:"Hooray! You escaped!"
                                    ~buttons:["OK"] ())
             | Game.LostGame ->   makemenu s top togl;
                                  ignore (Dialog.create ~parent:top
                                    ~title:"You lose"
                                    ~message:"Sorry... you lost"
                                    ~buttons:["OK"] ())
             | Game.LostChance -> makemenu s top togl;
                                  ignore (Dialog.create ~parent:top
                                    ~title:"Uh-oh"
                                    ~message:("Go back to jail - " ^
                                              "with one less chance")
                                    ~buttons:["OK"] ())
             | Game.NextLevel ->  makemenu s top togl;
                                  ignore (Dialog.create ~parent:top
                                    ~title:"You're getting there!"
                                    ~message:"You made it to the next level!"
                                    ~buttons:["OK"] ())
             | Game.StillPlaying -> ()
             end;
             Game.render s togl !lights !textures !stereo
           end)


(* start the user interface **************************************************)
let startup s = 
  let top = maketop in
  let togl = Togl.create ~width:800 ~height:600 ~rgba:true ~double:true
                         ~depth:true top in
  Game.init togl;
  configuretogl togl s top;
  pack ~fill:`Both ~side:`Bottom ~expand:true [togl];
  makemenu ~first:true s top togl

