(* level.mli: interface for level update and rendering *)

(* variant for whether a cube is solid (with given walls) or not *)
type solidity = Empty | Solid of Wall.t array

(* sorry about all the mutables - though most aren't needed for the game, it
 * makes life a bit more convenient for the editor *)

(* The basic unit of the level: a cube. Can be empty or solid, and can have
 * lists of contained objects and badguys attached to it, as well as a
 * timer indicating dematterization status, and scent which the badguys use
 * to track down Spiff. *)
type cube = {
  mutable w: solidity;
  mutable fixed: Fixedobj.t list;
  mutable free: Freeobj.t list;
  mutable badg: Badguy.t list;
  mutable dmtimer: float;   (* timer to count down dematterization *)
  mutable scent: float;     (* scent of Spiff used by badguys for tracking *)
}


(* the level *)
type t = {
  (* the current time in the level *)
  mutable currtime: float;

  (* the layout of this level *)
  lenx: int;   (* dimensions of level *)
  leny: int;
  lenz: int;
  layout: cube array; (* actually 3d array in FORTRAN order *)
  (* global lists of dematterized things in the level *)
  mutable dmcubes: cube list;
  mutable dmfixed: Fixedobj.t list;
  mutable dmfree: Freeobj.t list;
  (* a list of ALL the free objects in the level *)
  mutable freeobj: Freeobj.t list;
  (* a list of ALL the badguys in the level *)
  mutable badguys: Badguy.t list;

  (* Basic fields of spiff's initial state for this level *)
  mutable initspiffx: float;
  mutable initspiffy: float;
  mutable initspiffz: float;
  mutable initspiffheading: float;
}


(* return the ijk'th cube *)
val getcube: t -> int -> int -> int -> cube

(* set the ijk'th cube *)
val setcube: t -> int -> int -> int -> cube -> unit

(* Attempt to load in a level from the given file *)
val load_level: string -> t

(* Load in a fresh instance of the dungeon (aborting if there's a problem) *)
val load_newgame: unit -> (t array)

(* Put Spiff into the initial position for the given level *)
val init_spiff: t -> Spiff.t -> unit

(* dematterize stuff in the given hull *)
val dematterize: t -> Collision.hull -> unit

(* Update the state of the level given the time increment and Spiff.
 * Return whatever action happens to Spiff that can't be fully handled
 * here. *)
val update: t -> Spiff.t -> float -> Collision.action

(* render the current state of the level, given Spiff (for the viewpoint),
 * the width/height/border of the GL window in pixels, and whether or not
 * to use lighting and textures and stereo, and a displacement in camera's x *)
val render: t -> Spiff.t -> int -> int -> int -> bool -> bool -> bool ->
            float -> unit

