(* game.ml: implementation file for handling general game events *)

type state = {
  mutable prevtime: float;
  mutable spiff: Spiff.t;
  mutable lev: int;
  mutable dungeon: Level.t array;
}

type result = StillPlaying | WonGame | LostGame | LostChance | NextLevel

(* given the ToGL widget the game will be played in, initialize all the
 * objects for rendering. *)
let init togl =
  Tex.setparam ();
  Wall.init ();
  Spiff.init ();
  Fixedobj.init ();
  Freeobj.init ();
  Badguy.init ()


(* find and change to dungeon subdirectory, or raise Exit *)
let find_dungeon () =
  let dungeonpath = if Array.length Sys.argv > 1 then Sys.argv.(1)
                    else "Dungeon" in
  if Sys.file_exists (dungeonpath ^ "/level0")
  && Sys.file_exists (dungeonpath ^ "/Data") then
    Sys.chdir dungeonpath
  else begin
    prerr_endline ("Could not find dungeon (tried " ^ dungeonpath ^ ")");
    prerr_endline "Please specify a valid dungeon path on the command line";
    prerr_endline " (or don't specify anything if the 'Dungeon/' directory";
    prerr_endline "  is in the current directory)";
    exit 1
  end


(* attempt to load a new game in from the level files *)
let load_newgame () =
  find_dungeon ();
  let dungeon = Level.load_newgame () in
  let spiff = Spiff.newspiff () in
  Level.init_spiff dungeon.(0) spiff;
  { prevtime = 0.0;
    spiff = spiff;
    lev = 0;
    dungeon = dungeon}


(* called when gameplay resumes, in order to make sure the game time is
 * correct. *)
let resumeplaying s =
  s.prevtime <- Unix.gettimeofday () 


(* restarts the game at level 0 *)
let restart s =
  let dungeon = Level.load_newgame () in
  let spiff = Spiff.newspiff () in
  Level.init_spiff dungeon.(0) spiff;
  s.spiff <- spiff;
  s.lev <- 0;
  s.dungeon <- dungeon


(* called every time step to advance the state of the game. *)
let update s =
  (* first figure out the advance of time *)
  let newtime = Unix.gettimeofday () in
  let dt = newtime -. s.prevtime in
  s.prevtime <- newtime;

  (* update the level, getting the Spiff-related action *)
  let action = Level.update s.dungeon.(s.lev) s.spiff dt in

  (* respond to action *)
  match action with
    Collision.Teleport ->
      if s.lev = (Array.length s.dungeon)-1 then begin
        restart s;
        WonGame
      end else begin
        s.lev <- s.lev + 1;
        Level.init_spiff s.dungeon.(s.lev) s.spiff;
        NextLevel
      end
  | Collision.LoseGame ->
      (restart s; LostGame)
  | Collision.LoseChance ->
      LostChance
  | _ -> StillPlaying


(* render the current state s of the game in given ToGL widget, with flags
 * for lighting, texture and stereo mode. *)
let render s togl lightson textureson stereoon =
  GlClear.color (0.0, 0.0, 0.0);
  GlClear.clear [`color; `depth];
  let w = Togl.width togl and h = Togl.height togl in
  let b = w/20 in

  let dx = if stereoon then 0.03 else 0.0 in
  if stereoon then
    (* first draw only to red channel if we're in stereo *)
    GlFunc.color_mask ~red:true ~green:false ~blue:false ()
  else ();

  Level.render s.dungeon.(s.lev) s.spiff w h b lightson textureson stereoon dx;

  if stereoon then begin
    (* erase just the depth buffer *)
    GlClear.clear [`depth];
    (* and then draw the dungeon from a different viewpoint only in the
     * blue channel *)
    GlFunc.color_mask ~red:false ~green:false ~blue:true ();
    Level.render s.dungeon.(s.lev) s.spiff w h b lightson textureson stereoon
                 (-.dx);
    (* and finally return colour masking to normal *)
    GlFunc.color_mask ~red:true ~green:true ~blue:true ();
  end else ();

  (* draw the on-screen controls *)
  Spiff.render s.spiff w h b stereoon;

  (* and swap front and back buffers *)
  Togl.swap_buffers togl


(* function for activating the dematterizer. *)
let pressbutton s () =
  if Spiff.fire s.spiff then begin  (* see if Spiff can fire it *)
    (* if so, find out the dematterization hull *)
    let dmhull = Spiff.dematterhull s.spiff in
    (* and go ahead and dematterize whatever in the level *)
    Level.dematterize s.dungeon.(s.lev) dmhull
  end else ()

