(* fixedobj.ml: implementation for handling fixed objects *)

open Collision

type kind = BarsXY | BarsZY | BarsXZ | Step

type t = {
  kind: kind;
  hull: Collision.hull;
  mutable dmtimer: float;
}


(* create a new fixed object *)
let newfixed kind x y z =
  { kind = kind;
    hull = (match kind with
             BarsXY -> {x = x; y = y; z = z; lx = 0.5; ly = 0.5; lz = 0.025}
           | BarsZY -> {x = x; y = y; z = z; lx = 0.025; ly = 0.5; lz = 0.5}
           | BarsXZ -> {x = x; y = y; z = z; lx = 0.5; ly = 0.025; lz = 0.5}
           | Step   -> {x = x; y = y; z = z; lx = 0.5; ly = 0.25; lz = 0.5});
    dmtimer = 0.0
  }


(* stuff for rendering - we use display lists for the usual case of fully
 * solid objects, but we have to re-render anew for partially transparent
 * objects (since the alpha values change)
 *)

type displaylists = NotInitialized | DisplayLists of GlList.base

let plainlists = ref NotInitialized
and texlists = ref NotInitialized
and plaingreylists = ref NotInitialized
and texgreylists = ref NotInitialized

let barstexid = ref Nativeint.zero
and barstexgreyid = ref Nativeint.zero
and steptexid = ref Nativeint.zero
and steptexgreyid = ref Nativeint.zero

(* draw a vertical iron bar at the given x coordinate, with given radius,
 * and number of faces around. *)
let drawybar x r nr =
  (* bottom *)
  GlDraw.begins `triangle_fan;
    GlDraw.normal ~x:0.0 ~y:(-1.0) ~z:0.0 ();
    GlDraw.vertex ~x ~y:(-0.5) ~z:0.0 ();
    for i = 0 to nr do
      let t = 2.0 *. 3.1415926535897 *. (float_of_int (i mod nr))
                                    /. (float_of_int nr) in
      GlDraw.vertex ~x:(x+.r*.cos t) ~y:(-0.5) ~z:(r*.sin t) ();
    done;
  GlDraw.ends ();
  (* middle section *)
  GlDraw.begins `quad_strip;
    for i = 0 to nr do
      let t = 2.0 *. 3.1415926535897 *. (float_of_int ((nr-i) mod nr))
                                     /. (float_of_int nr) in
      let c = cos t and s = sin t in
      GlDraw.normal ~x:c ~y:0.0 ~z:s ();
      GlDraw.vertex ~x:(x+.r*.c) ~y:0.5    ~z:(r*.s) ();
      GlDraw.vertex ~x:(x+.r*.c) ~y:(-0.5) ~z:(r*.s) ();
    done;
  GlDraw.ends ();
  (* top *)
  GlDraw.begins `triangle_fan;
    GlDraw.normal ~x:0.0 ~y:1.0 ~z:0.0 ();
    GlDraw.vertex ~x ~y:0.5 ~z:0.0 ();
    for i = 0 to nr do
      let t = 2.0 *. 3.1415926535897 *. (float_of_int ((nr-i) mod nr))
                                    /. (float_of_int nr) in
      GlDraw.vertex ~x:(x+.r*.cos t) ~y:0.5 ~z:(r*.sin t) ();
    done;
  GlDraw.ends ()

(* draw vertical iron bar with textures *) 
let drawybartex x r nr =
  GlDraw.begins `triangle_fan;
    GlDraw.normal ~x:0.0 ~y:(-1.0) ~z:0.0 ();
    GlDraw.vertex ~x ~y:(-0.5) ~z:0.0 ();
    for i = 0 to nr do
      let t = 2.0 *. 3.1415926535897 *. (float_of_int (i mod nr))
                                    /. (float_of_int nr) in
      GlDraw.vertex ~x:(x+.r*.cos t) ~y:(-0.5) ~z:(r*.sin t) ();
    done;
  GlDraw.ends ();
  Gl.enable `texture_2d;
  GlDraw.begins `quad_strip;
    for i = 0 to nr do
      let t = 2.0 *. 3.1415926535897 *. (float_of_int ((nr-i) mod nr))
                                     /. (float_of_int nr) in
      let c = cos t and s = sin t in
      GlDraw.normal ~x:c ~y:0.0 ~z:s ();
      GlTex.coord2 (6.3*.r*.(float i)/.(float nr),1.0);
      GlDraw.vertex ~x:(x+.r*.c) ~y:0.5    ~z:(r*.s) ();
      GlTex.coord2 (6.3*.r*.(float i)/.(float nr),0.0);
      GlDraw.vertex ~x:(x+.r*.c) ~y:(-0.5) ~z:(r*.s) ();
    done;
  GlDraw.ends ();
  Gl.disable `texture_2d;
  GlDraw.begins `triangle_fan;
    GlDraw.normal ~x:0.0 ~y:1.0 ~z:0.0 ();
    GlDraw.vertex ~x ~y:0.5 ~z:0.0 ();
    for i = 0 to nr do
      let t = 2.0 *. 3.1415926535897 *. (float_of_int ((nr-i) mod nr))
                                    /. (float_of_int nr) in
      GlDraw.vertex ~x:(x+.r*.cos t) ~y:0.5 ~z:(r*.sin t) ();
    done;
  GlDraw.ends ()

(* draw horizontal (aligned with x axis) iron bar *) 
let drawxbar y r nr =
  GlDraw.begins `triangle_fan;
    GlDraw.normal ~x:(-1.0) ~y:0.0 ~z:0.0 ();
    GlDraw.vertex ~x:(-0.5) ~y ~z:0.0 ();
    for i = 0 to nr do
      let t = 2.0 *. 3.1415926535897 *. (float_of_int ((nr-i) mod nr))
                                    /. (float_of_int nr) in
      GlDraw.vertex ~x:(-0.5) ~y:(y+.r*.cos t) ~z:(r*.sin t) ();
    done;
  GlDraw.ends ();
  GlDraw.begins `quad_strip;
    for i = 0 to nr do
      let t = 2.0 *. 3.1415926535897 *. (float_of_int ((nr-i) mod nr))
                                     /. (float_of_int nr) in
      let c = cos t and s = sin t in
      GlDraw.normal ~x:0.0 ~y:c ~z:s ();
      GlDraw.vertex ~x:(-0.5) ~y:(y+.r*.c) ~z:(r*.s) ();
      GlDraw.vertex ~x:0.5    ~y:(y+.r*.c) ~z:(r*.s) ();
    done;
  GlDraw.ends ();
  GlDraw.begins `triangle_fan;
    GlDraw.normal ~x:1.0 ~y:0.0 ~z:0.0 ();
    GlDraw.vertex ~x:0.5 ~y ~z:0.0 ();
    for i = 0 to nr do
      let t = 2.0 *. 3.1415926535897 *. (float_of_int (i mod nr))
                                    /. (float_of_int nr) in
      GlDraw.vertex ~x:0.5 ~y:(y+.r*.cos t) ~z:(r*.sin t) ();
    done;
  GlDraw.ends ()

(* draw horizontal iron bar with textures *)
let drawxbartex y r nr =
  GlDraw.begins `triangle_fan;
    GlDraw.normal ~x:(-1.0) ~y:0.0 ~z:0.0 ();
    GlDraw.vertex ~x:(-0.5) ~y ~z:0.0 ();
    for i = 0 to nr do
      let t = 2.0 *. 3.1415926535897 *. (float_of_int ((nr-i) mod nr))
                                    /. (float_of_int nr) in
      GlDraw.vertex ~x:(-0.5) ~y:(y+.r*.cos t) ~z:(r*.sin t) ();
    done;
  GlDraw.ends ();
  Gl.enable `texture_2d;
  GlDraw.begins `quad_strip;
    for i = 0 to nr do
      let t = 2.0 *. 3.1415926535897 *. (float_of_int ((nr-i) mod nr))
                                     /. (float_of_int nr) in
      let c = cos t and s = sin t in
      GlDraw.normal ~x:0.0 ~y:c ~z:s ();
      GlTex.coord2 (6.3*.r*.(float i)/.(float nr),1.0);
      GlDraw.vertex ~x:(-0.5) ~y:(y+.r*.c) ~z:(r*.s) ();
      GlTex.coord2 (6.3*.r*.(float i)/.(float nr),0.0);
      GlDraw.vertex ~x:0.5    ~y:(y+.r*.c) ~z:(r*.s) ();
    done;
  GlDraw.ends ();
  Gl.disable `texture_2d;
  GlDraw.begins `triangle_fan;
    GlDraw.normal ~x:1.0 ~y:0.0 ~z:0.0 ();
    GlDraw.vertex ~x:0.5 ~y ~z:0.0 ();
    for i = 0 to nr do
      let t = 2.0 *. 3.1415926535897 *. (float_of_int (i mod nr))
                                    /. (float_of_int nr) in
      GlDraw.vertex ~x:0.5 ~y:(y+.r*.cos t) ~z:(r*.sin t) ();
    done;
  GlDraw.ends ()


(* draw BarsXY with the given alpha value, flags to indicate whether to use
 * textures and/or greyscale, and the id of the texture to use. *)
let drawbarsxy alpha textures greyscale texid =
  if textures then begin
    Gl.enable `texture_2d;
    GlTex.env (`mode `modulate);
    Tex.bind texid;
    GlDraw.color ~alpha (1.0,1.0,1.0);
  end else if greyscale then
    GlDraw.color ~alpha (0.6,0.6,0.6)
  else
    GlDraw.color ~alpha (0.7,0.6,0.1);

  if greyscale then begin
    GlLight.material ~face:`front (`ambient_and_diffuse(0.5,0.5,0.5,alpha));
    GlLight.material ~face:`front (`specular(0.9,0.9,0.9,alpha));
  end else begin
    GlLight.material ~face:`front (`ambient_and_diffuse(0.5,0.4,0.1,alpha));
    GlLight.material ~face:`front (`specular(0.9,0.8,0.3,alpha));
  end;
  GlLight.material ~face:`front (`shininess 20.0);

  if textures then begin
    drawybartex (-0.375) 0.025 9;
    drawybartex (-0.125) 0.025 9;
    drawybartex 0.125 0.025 9;
    drawybartex 0.375 0.025 9;
    drawxbartex (-0.375) 0.025 9;
    drawxbartex (-0.125) 0.025 9;
    drawxbartex 0.125 0.025 9;
    drawxbartex 0.375 0.025 9;
  end else begin
    drawybar (-0.375) 0.025 9;
    drawybar (-0.125) 0.025 9;
    drawybar 0.125 0.025 9;
    drawybar 0.375 0.025 9;
    drawxbar (-0.375) 0.025 9;
    drawxbar (-0.125) 0.025 9;
    drawxbar 0.125 0.025 9;
    drawxbar 0.375 0.025 9;
  end;
  if textures then
    Gl.disable `texture_2d
  else ()


(* draw Step *)
let drawstep alpha textures greyscale texid =
  if textures then begin
    Gl.enable `texture_2d;
    GlTex.env (`mode `modulate);
    Tex.bind texid;
    GlDraw.color ~alpha (1.0,1.0,1.0);
  end else if greyscale then
    GlDraw.color ~alpha (0.45,0.45,0.45)
  else
    GlDraw.color ~alpha (0.5,0.45,0.3);

  if greyscale then begin
    GlLight.material ~face:`front (`ambient_and_diffuse(0.45,0.45,0.45,alpha));
    GlLight.material ~face:`front (`specular(0.1,0.1,0.1,1.0));
  end else begin
    GlLight.material ~face:`front (`ambient_and_diffuse(0.5,0.45,0.3,alpha));
    GlLight.material ~face:`front (`specular(0.1,0.1,0.0,1.0));
  end;
  GlLight.material ~face:`front (`shininess 1.0);

  (* It's just a box with 6 faces *)
  GlDraw.begins `quads;
    GlDraw.normal ~x:(-1.0) ();
    if textures then GlTex.coord2 (0.0,0.0) else ();
    GlDraw.vertex ~x:(-0.5) ~y:(-0.25) ~z:(-0.5) ();
    if textures then GlTex.coord2 (1.0,0.0) else ();
    GlDraw.vertex ~x:(-0.5) ~y:(-0.25) ~z:0.5 ();
    if textures then GlTex.coord2 (1.0,0.5) else ();
    GlDraw.vertex ~x:(-0.5) ~y:0.25    ~z:0.5 ();
    if textures then GlTex.coord2 (0.0,0.5) else ();
    GlDraw.vertex ~x:(-0.5) ~y:0.25    ~z:(-0.5) ();
    GlDraw.normal ~x:1.0 ();
    if textures then GlTex.coord2 (0.0,0.0) else ();
    GlDraw.vertex ~x:0.5    ~y:(-0.25) ~z:(-0.5) ();
    if textures then GlTex.coord2 (0.0,0.5) else ();
    GlDraw.vertex ~x:0.5    ~y:0.25    ~z:(-0.5) ();
    if textures then GlTex.coord2 (1.0,0.5) else ();
    GlDraw.vertex ~x:0.5    ~y:0.25    ~z:0.5 ();
    if textures then GlTex.coord2 (1.0,0.0) else ();
    GlDraw.vertex ~x:0.5    ~y:(-0.25) ~z:0.5 ();
    GlDraw.normal ~x:0.0 ~y:(-1.0) ();
    if textures then GlTex.coord2 (0.0,0.0) else ();
    GlDraw.vertex ~x:(-0.5) ~y:(-0.25) ~z:(-0.5) ();
    if textures then GlTex.coord2 (1.0,0.0) else ();
    GlDraw.vertex ~x:0.5    ~y:(-0.25) ~z:(-0.5) ();
    if textures then GlTex.coord2 (1.0,1.0) else ();
    GlDraw.vertex ~x:0.5    ~y:(-0.25) ~z:0.5 ();
    if textures then GlTex.coord2 (0.0,1.0) else ();
    GlDraw.vertex ~x:(-0.5) ~y:(-0.25) ~z:0.5 ();
    GlDraw.normal ~x:0.0 ~y:1.0 ();
    if textures then GlTex.coord2 (0.0,0.0) else ();
    GlDraw.vertex ~x:(-0.5) ~y:0.25    ~z:(-0.5) ();
    if textures then GlTex.coord2 (0.0,1.0) else ();
    GlDraw.vertex ~x:(-0.5) ~y:0.25    ~z:0.5 ();
    if textures then GlTex.coord2 (1.0,1.0) else ();
    GlDraw.vertex ~x:0.5    ~y:0.25    ~z:0.5 ();
    if textures then GlTex.coord2 (1.0,0.0) else ();
    GlDraw.vertex ~x:0.5    ~y:0.25    ~z:(-0.5) ();
    GlDraw.normal ~x:0.0 ~y:0.0 ~z:(-1.0) ();
    if textures then GlTex.coord2 (0.0,0.0) else ();
    GlDraw.vertex ~x:(-0.5) ~y:(-0.25) ~z:(-0.5) ();
    if textures then GlTex.coord2 (0.0,0.5) else ();
    GlDraw.vertex ~x:(-0.5) ~y:0.25    ~z:(-0.5) ();
    if textures then GlTex.coord2 (1.0,0.5) else ();
    GlDraw.vertex ~x:0.5    ~y:0.25    ~z:(-0.5) ();
    if textures then GlTex.coord2 (1.0,0.0) else ();
    GlDraw.vertex ~x:0.5    ~y:(-0.25) ~z:(-0.5) ();
    GlDraw.normal ~x:0.0 ~y:0.0 ~z:1.0 ();
    if textures then GlTex.coord2 (0.0,0.0) else ();
    GlDraw.vertex ~x:(-0.5) ~y:(-0.25) ~z:0.5 ();
    if textures then GlTex.coord2 (1.0,0.0) else ();
    GlDraw.vertex ~x:0.5    ~y:(-0.25) ~z:0.5 ();
    if textures then GlTex.coord2 (1.0,0.5) else ();
    GlDraw.vertex ~x:0.5    ~y:0.25    ~z:0.5 ();
    if textures then GlTex.coord2 (0.0,0.5) else ();
    GlDraw.vertex ~x:(-0.5) ~y:0.25    ~z:0.5 ();
  GlDraw.ends ();

  if textures then
    Gl.disable `texture_2d
  else ()


(* initialize stuff before rendering *)
let init () =
  (* set up textures *)
  barstexid := Tex.loadPPM "Data/bars.ppm";
  if !barstexid = Nativeint.zero then begin
    prerr_endline ("Problem loading image Data/bars.ppm");
    exit 0
  end else ();
  barstexgreyid := Tex.loadPPM "Data/bars_grey.ppm";
  if !barstexgreyid = Nativeint.zero then begin
    prerr_endline ("Problem loading image Data/bars_grey.ppm");
    exit 0
  end else ();
  steptexid := Tex.loadPPM "Data/step.ppm";
  if !steptexid = Nativeint.zero then begin
    prerr_endline ("Problem loading image Data/step.ppm");
    exit 0
  end else ();
  steptexgreyid := Tex.loadPPM "Data/step_grey.ppm";
  if !steptexgreyid = Nativeint.zero then begin
    prerr_endline ("Problem loading image Data/step_grey.ppm");
    exit 0
  end else ();

  (* set up display lists for objects *)
  let plains = GlList.gen_lists ~len:2 in
  plainlists := DisplayLists plains;
  let texs = GlList.gen_lists ~len:2 in
  texlists := DisplayLists texs;
  let plaingreys = GlList.gen_lists ~len:2 in
  plaingreylists := DisplayLists plaingreys;
  let texgreys = GlList.gen_lists ~len:2 in
  texgreylists := DisplayLists texgreys;

  (* BarsXY *)
  GlList.begins (GlList.nth plains ~pos:0) ~mode:`compile;
  drawbarsxy 1.0 false false Nativeint.zero;
  GlList.ends ();
  GlList.begins (GlList.nth texs ~pos:0) ~mode:`compile;
  drawbarsxy 1.0 true false !barstexid;
  GlList.ends ();
  GlList.begins (GlList.nth plaingreys ~pos:0) ~mode:`compile;
  drawbarsxy 1.0 false true Nativeint.zero;
  GlList.ends ();
  GlList.begins (GlList.nth texgreys ~pos:0) ~mode:`compile;
  drawbarsxy 1.0 true true !barstexgreyid;
  GlList.ends ();

  (* Step *)
  GlList.begins (GlList.nth plains ~pos:1) ~mode:`compile;
  drawstep 1.0 false false Nativeint.zero;
  GlList.ends ();
  GlList.begins (GlList.nth texs ~pos:1) ~mode:`compile;
  drawstep 1.0 true false !steptexid;
  GlList.ends ();
  GlList.begins (GlList.nth plaingreys ~pos:1) ~mode:`compile;
  drawstep 1.0 false true Nativeint.zero;
  GlList.ends ();
  GlList.begins (GlList.nth texgreys ~pos:1) ~mode:`compile;
  drawstep 1.0 true true !steptexgreyid;
  GlList.ends ()


(* render the object *)
let render ob textureson greyscaleon =
  if ob.dmtimer > 0.0 then  (* if the object is dematterized *)
    let alpha = 1.0 -. ob.dmtimer in  (* draw it semi-transparent *)
    GlMat.push ();
    GlMat.translate ~x:ob.hull.x ~y:ob.hull.y ~z:ob.hull.z ();
    begin match ob.kind with
      BarsXY ->
         drawbarsxy alpha textureson greyscaleon
                    (if greyscaleon then !barstexgreyid else !barstexid)
    | BarsZY -> begin
         GlMat.rotate ~y:1.0 ~angle:90.0 ();
         drawbarsxy alpha textureson greyscaleon
                    (if greyscaleon then !barstexgreyid else !barstexid)
       end
    | BarsXZ -> begin
         GlMat.rotate ~x:1.0 ~angle:90.0 ();
         drawbarsxy alpha textureson greyscaleon
                    (if greyscaleon then !barstexgreyid else !barstexid)
       end
    | Step   ->
         drawstep alpha textureson greyscaleon
                  (if greyscaleon then !steptexgreyid else !steptexid)
    end;
    GlMat.pop ()

  else begin
    (* otherwise, it's solid and we can use the display lists *)
    GlMat.push ();
    GlMat.translate ~x:ob.hull.x ~y:ob.hull.y ~z:ob.hull.z ();

    let DisplayLists dls =
      if textureson then begin
        if greyscaleon then !texgreylists
        else !texlists
      end else begin
        if greyscaleon then !plaingreylists
        else !plainlists
      end in

    begin match ob.kind with
      BarsXY -> GlList.call (GlList.nth dls ~pos:0)
    | BarsZY -> begin
         GlMat.rotate ~y:1.0 ~angle:90.0 ();
         GlList.call (GlList.nth dls ~pos:0);
       end
    | BarsXZ -> begin
         GlMat.rotate ~x:1.0 ~angle:90.0 ();
         GlList.call (GlList.nth dls ~pos:0)
       end
    | Step   -> GlList.call (GlList.nth dls ~pos:1)
    end;
    GlMat.pop ()
  end

