(* editor.ml: the game level editor. This code needs a lot of work... *)

open Level
open Collision

(* Simple file selector ******************************************************)
let fileselect ~title ~action ~actionname ?(name="") top =
  let fs = Toplevel.create top in
  Wm.title_set fs title;
  let fname = Textvariable.create ~on:fs () in
  Textvariable.set fname name;
  let fentry = Entry.create ~textvariable:fname fs in
  let bframe = Frame.create fs in
  let ok = Button.create ~text:actionname bframe
    ~command:(fun () -> let fn = Textvariable.get fname in
                if fn = "" then ()
                else begin
                  Tk.destroy fs;
                  action fn
                end) in
  let cancel = Button.create ~text:"Cancel" bframe
                             ~command:(fun () -> Tk.destroy fs) in
  Tk.pack ~fill:`Both ~side:`Left [ok; cancel];
  Tk.pack ~fill:`Both ~padx:3 ~pady:3 ~side:`Top [Tk.coe fentry; Tk.coe bframe]


(* Ensure a level is consistent **********************************************)
(* (fill in list of free objects and badguys, add scents, etc.) *)
let finish_level lev =
  lev.freeobj <- [];
  lev.badguys <- [];
  (* loop over all cubes, putting their stuff into the global lists *)
  for i = 0 to lev.lenx*lev.leny*lev.lenz-1 do
    lev.freeobj <- List.rev_append lev.layout.(i).free lev.freeobj;
    lev.badguys <- List.rev_append lev.layout.(i).badg lev.badguys
  done;
  (* and put good scent information in *)
  for i = 0 to lev.lenx-1 do
    let dx = abs_float ((0.5 +. float_of_int i) -. lev.initspiffx) in
    for j = 0 to lev.leny-1 do
      let dy = abs_float ((0.5 +. float_of_int j) -. lev.initspiffy) in
      for k = 0 to lev.lenz-1 do
        let dz = abs_float ((0.5 +. float_of_int k) -. lev.initspiffz) in
        let cube = getcube lev i j k in
        cube.scent <- 2.0/.(dx +. dy +. dz +. 10.0)
      done
    done
  done


(* Save a given level to the given file **************************************)
let save_level lev filename =
  try
    let outchan = open_out_bin filename in
    Marshal.to_channel outchan lev ~mode:[];
    close_out outchan
  with _ -> prerr_endline "Could not save file"


(* Pop up a file selector to save file ***************************************)
let asksave lev top name () =
  fileselect ~title:"Save level"
    ~action:(fun fn -> finish_level lev;
                       save_level lev fn) ~actionname:"Save" ~name top



(* Create a viewer control for editing levels *******************************)
(* The frame the controls are in is returned, along with a function
 *  int -> int -> int -> viewobject list -> unit
 * that when applied to i j k vl, alters the display of cube (i,j,k)
 * to match the things specified in vl, and a function
 *  (unit -> unit) -> unit
 * that turns on the mouse bindings for selecting cube, changing solidity
 * (executing the callback keeps other displays of current cube up to date).
 *)
let makelevelview lev currcube container =
  (* first just set up the display *)
  let cubesize = 21 in 
  let viewpos lev i j k =
    (i*cubesize+1 + cubesize*(lev.lenx+1)*j, k*cubesize+1) in
  let vf = Frame.create container in
  let w = cubesize*(lev.lenx+1)*lev.leny-cubesize+1 in
  let h = 1+cubesize*lev.lenz in
  (* the region where we'll display slices *)
  let c = Canvas.create ~confine:true ~width:(min 800 w) ~height:h
                        ~scrollregion:(0,0,w,h) 
                        ~xscrollincrement:cubesize vf in
  let scroll = Scrollbar.create ~orient:`Horizontal
        ~command:(Canvas.xview c) vf in
  Canvas.configure c ~xscrollcommand:(Scrollbar.set scroll);
  Tk.pack ~side:`Top ~fill:`X [Tk.coe c; Tk.coe scroll];

  (* next create the basic rectangle layout showing the level *)
  let numcubes = lev.lenx * lev.leny * lev.lenz in
  let layoutview = Array.make numcubes (`Id(0),[]) in  (* dummy element! *)
  let cubeindex (i,j,k) = i+lev.lenx*(j+lev.leny*k) in
  for i = 0 to lev.lenx-1 do
    for j = 0 to lev.leny-1 do
      for k = 0 to lev.lenz-1 do
        (* find where in the array this is *)
        let n = cubeindex (i,j,k) in
        (* display a square to show the cube *)
        let (x1, y1) = viewpos lev i j k in
        layoutview.(n) <- (Canvas.create_rectangle ~x1 ~y1
                           ~x2:(x1+cubesize-2) ~y2:(y1+cubesize-2) c, []);
      done
    done
  done;

  (* now create the view update function *)
  let updateviewfun i j k =
    let n = cubeindex (i,j,k) in
    (* first eliminate the old displayed objects, apart from square *)
    let (square,rest) = layoutview.(n) in
    Canvas.delete c rest;
    (* now create new display objects *)
    let (x, y) = viewpos lev i j k in
    let cube = getcube lev i j k in

    let newrest = if cube.fixed != [] then
                    [Canvas.create_rectangle ~x1:(x+5) ~y1:(y+5)
                                             ~x2:(x+9) ~y2:(y+9) ~fill:`Red c]
                  else [] in
    let newrest = if cube.free != [] then
                    (Canvas.create_oval ~x1:(x+10) ~y1:(y+5)
                               ~x2:(x+14) ~y2:(y+9) ~fill:`Blue c)::newrest
                  else newrest in
    let newrest = if cube.badg != [] then
                    (Canvas.create_oval ~x1:(x+5) ~y1:(y+10)
                               ~x2:(x+9) ~y2:(y+14) ~fill:`Green c)::newrest
                  else newrest in
    let spiffincube =   (int_of_float lev.initspiffx)=i
                      & (int_of_float lev.initspiffy)=j
                      & (int_of_float lev.initspiffz)=k in
    let newrest = if spiffincube then
                    (Canvas.create_rectangle ~x1:(x+10) ~y1:(y+10)
                               ~x2:(x+14) ~y2:(y+14) ~fill:`White c)::newrest
                  else newrest in
    layoutview.(n) <- (square,newrest)
  in

  (* and finally create the binding function *)
  let highlight n =
    Canvas.configure_rectangle ~outline:`Red c (fst layoutview.(n)) in
  let unhighlight n =
    Canvas.configure_rectangle ~outline:`Black c (fst layoutview.(n)) in
  let bindingfun callback =
    for i = 0 to lev.lenx-1 do
      for j = 0 to lev.leny-1 do
        for k = 0 to lev.lenz-1 do
          let n = cubeindex (i,j,k) in
          let square = fst layoutview.(n) in
          (* colour square appropriately *)
          Canvas.configure_rectangle
            ~fill:(match lev.layout.(n).w with Empty    -> `White
                                             | Solid(_) -> `Black)
            c square;
          (* and update its view *)
          updateviewfun i j k;
          (* bind button 1: make this cube current *)
          Canvas.bind ~events:[`ButtonPressDetail(1)]
            ~action:(fun ev ->
               unhighlight (cubeindex !currcube);
               highlight n;
               currcube := (i,j,k);
               callback ())
            c square;
          (* if we're not on the boundary, bind button 2: toggle solidity *)
          if i>0 & i<lev.lenx-1 & j>0 & j<lev.leny-1 & k>0 & k<lev.lenz-1 then
            Canvas.bind ~events:[`ButtonPressDetail(3)]
              ~action:(fun ev ->
                 unhighlight (cubeindex !currcube);
                 highlight n;
                 currcube := (i,j,k);
                 begin match lev.layout.(n).w with
                   Empty ->
                     lev.layout.(n).w <- Solid([| Wall.Blocks; Wall.Blocks;
                                                  Wall.Tiles; Wall.Stone;
                                                  Wall.Blocks; Wall.Blocks |]);
                     Canvas.configure_rectangle ~fill:`Black c square
                   | Solid(_) ->
                     lev.layout.(n).w <- Empty;
                     Canvas.configure_rectangle ~fill:`White c square
                 end;
                 callback ())
              c square
           else ();
        done
      done
    done;
    let (i,j,k) = !currcube in
    (* highlight the current cube *)
    highlight (cubeindex (i,j,k));
    (* and make sure the display is up to date *)
    callback ()
  in

  (vf, updateviewfun, bindingfun)


(* Create a frame full of radiobuttons, given a container to put them in,
 * a nonempty list of strings (choices), and a callback that takes a string
 * (the newly selected choice) and returns unit. This function returns the
 * frame and a callback which changes the selection (given a string) but
 * does not recursively call the associated command.
 *)
let makeradioframe choicelist callback container =
  let frame = Frame.create container in
  let var = Textvariable.create ~on:frame () in
  Textvariable.set var (List.hd choicelist);
  let blist = List.map ~f:(fun choice ->
              Radiobutton.create ~indicatoron:true ~text:choice ~value:choice
                ~variable:var ~command:(fun () -> callback choice) frame)
            choicelist in
  Tk.pack ~side:`Left ~fill:`X ~expand:true blist;
  let updateradio choice =
    List.iter2 ~f:(fun c b -> if c = choice then Radiobutton.select b else ())
      choicelist blist
  in
  (frame, updateradio)


(* Create a choice menu, given a container to put it in, a label, a
 * nonempty list of strings (choices), and a callback that takes a string
 * (the choice that was just selected) and returns unit.
 * Returns the menubutton and a command to update the state of the menu
 * (given one of the choices) *)
let makepopup text choicelist callback container =
  let meb = Menubutton.create ~text:(text ^ ": " ^ (List.hd choicelist))
                              ~relief:`Raised container in
  let men = Menu.create ~tearoff:false meb in
  let updatemenu choice =
    Menubutton.configure ~text:(text ^ ": " ^ choice) meb;
  in
  List.iter ~f:(fun c -> Menu.add_command ~label:c
                                 ~command:(fun () -> updatemenu c; callback c)
                         men) choicelist;
  Menubutton.configure ~menu:men meb;
  (meb,updatemenu)


(* miscellaneous conversion commands for the game data *)

let wallchoices = ["Stone";"Blocks";"Tiles";"SlimePuddle"]

let string_of_wall w =
  match w with
    Wall.Stone       -> "Stone"
  | Wall.Blocks      -> "Blocks"
  | Wall.Tiles       -> "Tiles"
  | Wall.SlimePuddle -> "SlimePuddle"

let wall_of_string s =
  match s with
    "Blocks"      -> Wall.Blocks
  | "Tiles"       -> Wall.Tiles
  | "SlimePuddle" -> Wall.SlimePuddle
  | _ -> Wall.Stone

let fixedchoices = ["None";"BarsXY";"BarsZY";"BarsXZ";"Step"]

let string_of_fixedlist fl =
  match fl with
    []      -> "None"
  | f::rest -> (match f.Fixedobj.kind with
                  Fixedobj.BarsXY     -> "BarsXY"
                | Fixedobj.BarsZY     -> "BarsZY"
                | Fixedobj.BarsXZ     -> "BarsXZ"
                | Fixedobj.Step       -> "Step")

let makefixedlist s i j k =
  let x = 0.5 +. float_of_int i
  and y = 0.5 +. float_of_int j
  and z = 0.5 +. float_of_int k in
  match s with
    "BarsXY"     -> [ (Fixedobj.newfixed Fixedobj.BarsXY x y z) ]
  | "BarsZY"     -> [ (Fixedobj.newfixed Fixedobj.BarsZY x y z) ]
  | "BarsXZ"     -> [ (Fixedobj.newfixed Fixedobj.BarsXZ x y z) ]
  | "Step"       -> [ (Fixedobj.newfixed Fixedobj.Step x (y-.0.25) z) ]
  | _            -> []

let freechoices = ["None";"Charge";"Bribe";"Teleporter"]

let string_of_freelist fl =
  match fl with
    []      -> "None"
  | f::rest -> (match f.Freeobj.kind with
                  Freeobj.Charge -> "Charge"
                | Freeobj.Bribe  -> "Bribe"
                | Freeobj.Teleporter -> "Teleporter")

let makefreelist s i j k =
  let x = float_of_int i
  and y = float_of_int j
  and z = float_of_int k in
  match s with
    "Charge" -> [ (Freeobj.newfree Freeobj.Charge (x+.0.3+.Random.float 0.4)
                                                  (y+.0.5)
                                                  (z+.0.3+.Random.float 0.4)) ]
  | "Bribe"  -> [ (Freeobj.newfree Freeobj.Bribe (x+.0.3+.Random.float 0.4)
                                                 (y+.0.5)
                                                 (z+.0.3+.Random.float 0.4)) ]
  | "Teleporter" -> [ (Freeobj.newfree Freeobj.Teleporter
                                       (x+.0.5) (y+.0.5) (z+.0.5)) ]
  | _        -> []

let badguychoices = ["None";"Guard"]

let string_of_badguylist bl =
  match bl with
    []      -> "None"
  | b::rest -> (match b.Badguy.kind with
                  Badguy.Guard -> "Guard")

let makebadguylist s i j k =
  let x = 0.5 +. float_of_int i
  and y = 0.5 +. float_of_int j
  and z = 0.5 +. float_of_int k in
  match s with
    "Guard" -> [ (Badguy.newbadguy Badguy.Guard x y z (Random.float 360.0)) ]
  | _       -> []


(* Create and return a widget inside the container where we can alter the
 * walls of the current cube in the level. This returns a function
 * unit->unit to be called when the current cube is changed, so that the
 * display can update. *)
let makewallCtrl lev currcube container =
  let wf = Frame.create ~relief:`Groove container ~borderwidth:2 in
  Tk.pack ~fill:`Both ~side:`Top
    [Message.create ~text:"Walls" ~justify:`Center ~width:120 wf];

  let changewall n s =
    let (i,j,k) = !currcube in
    match (getcube lev i j k).w with
      Solid(walls) ->   walls.(n) <- wall_of_string s
    | Empty -> ()  (* should not occur *)
  in
  let (mxp,updatexp) = makepopup "x+" wallchoices (changewall 0) wf
  and (mxm,updatexm) = makepopup "x-" wallchoices (changewall 1) wf
  and (myp,updateyp) = makepopup "y+" wallchoices (changewall 2) wf
  and (mym,updateym) = makepopup "y-" wallchoices (changewall 3) wf
  and (mzp,updatezp) = makepopup "z+" wallchoices (changewall 4) wf
  and (mzm,updatezm) = makepopup "z-" wallchoices (changewall 5) wf in
  Tk.pack ~fill:`X ~side:`Left ~ipadx:2 ~ipady:2 ~expand:true
    [mxp; mxm; myp; mym; mzp; mzm];

  let cubechangefun () =
    let (i,j,k) = !currcube in
    match (getcube lev i j k).w with
      Empty ->
        List.iter ~f:(Menubutton.configure ~state:`Disabled)
                  [mxp; mxm; myp; mym; mzp; mzm]
    | Solid(walls) ->
        List.iter ~f:(Menubutton.configure ~state:`Normal)
                  [mxp; mxm; myp; mym; mzp; mzm];
        updatexp (string_of_wall walls.(0));
        updatexm (string_of_wall walls.(1));
        updateyp (string_of_wall walls.(2));
        updateym (string_of_wall walls.(3));
        updatezp (string_of_wall walls.(4));
        updatezm (string_of_wall walls.(5));
  in
  (wf,cubechangefun)


(* Create a widget inside the container where we can alter the list of fixed
 * objects inside the current cube in the level, given a callback to be
 * called when the cube is altered. This returns a function unit->unit to be
 * called when the current cube is changed, so that the display can update. *)
let makefixedlistCtrl lev currcube updateviewfun container =
  let fxf = Frame.create ~relief:`Groove container ~borderwidth:2 in
  Tk.pack ~fill:`Both ~side:`Top
    [Message.create ~text:"Fixed objects" ~justify:`Center ~width:120 fxf];
  let changef s =
    let (i,j,k) = !currcube in
    (getcube lev i j k).fixed <- makefixedlist s i j k;
    updateviewfun i j k
  in
  let (bframe,bupdate) = makeradioframe fixedchoices changef fxf in
  Tk.pack ~fill:`X ~side:`Top [bframe];

  let cubechangefun () =
    let (i,j,k) = !currcube in
    let cube = getcube lev i j k in
    bupdate (string_of_fixedlist cube.fixed)
  in
  (fxf, cubechangefun)


(* Create a widget inside the container where we can alter the list of free
 * objects inside the current cube in the level, given a callback to be
 * called when the cube is altered. This returns a function unit->unit to be
 * called when the current cube is changed, so that the display can update. *)
let makefreelistCtrl lev currcube updateviewfun container =
  let frf = Frame.create ~relief:`Groove container ~borderwidth:2 in
  Tk.pack ~fill:`Both ~side:`Top
    [Message.create ~text:"Free objects" ~justify:`Center ~width:120 frf];
  let changef s =
    let (i,j,k) = !currcube in
    (getcube lev i j k).free <- makefreelist s i j k;
    updateviewfun i j k
  in
  let (bframe,bupdate) = makeradioframe freechoices changef frf in
  Tk.pack ~fill:`X ~side:`Top [bframe];

  let cubechangefun () =
    let (i,j,k) = !currcube in
    let cube = getcube lev i j k in
    bupdate (string_of_freelist cube.free)
  in
  (frf, cubechangefun)


(* Create a widget inside the container where we can alter the list of badguys
 * inside the current cube in the level, given a callback to be
 * called when the cube is altered. This returns a function unit->unit to be
 * called when the current cube is changed, so that the display can update. *)
let makebadguysCtrl lev currcube updateviewfun container =
  let bgf = Frame.create ~relief:`Groove container ~borderwidth:2 in
  Tk.pack ~fill:`Both ~side:`Top
    [Message.create ~text:"Badguys" ~justify:`Center ~width:120 bgf];
  let changeb s =
    let (i,j,k) = !currcube in
    (getcube lev i j k).badg <- makebadguylist s i j k;
    updateviewfun i j k
  in
  let (bframe,bupdate) = makeradioframe badguychoices changeb bgf in
  Tk.pack ~fill:`X ~side:`Top [bframe];

  let cubechangefun () =
    let (i,j,k) = !currcube in
    let cube = getcube lev i j k in
    bupdate (string_of_badguylist cube.badg)
  in
  (bgf, cubechangefun)


(* Create a widget inside the container where we can put Spiff
 * inside the current cube in the level, given a callback to be
 * called when the cube is altered. This returns a function unit->unit to be
 * called when the current cube is changed, so that the display can update. *)
let makespiffCtrl lev currcube updateviewfun container =
  let spf = Frame.create ~relief:`Groove container ~borderwidth:2 in
  let placespiff () =
    let oldi = int_of_float lev.initspiffx
    and oldj = int_of_float lev.initspiffy
    and oldk = int_of_float lev.initspiffy in
    let (i,j,k) = !currcube in
    lev.initspiffx <- 0.5 +. float_of_int i;
    lev.initspiffy <- 0.3001 +. float_of_int j;
    lev.initspiffz <- 0.5 +. float_of_int k;
    updateviewfun oldi oldj oldk;
    updateviewfun i j k
  in
  let spiffhereb =
    Button.create ~text:"Put Spiff here" ~command:placespiff spf in
  let headmesg = Message.create ~text:"Heading:" ~justify:`Center
                                ~width:120 spf in
  let headscale =
    Scale.create ~digits:3 ~max:359.0 ~min:0.0 ~orient:`Horizontal
      ~command:(fun h -> lev.initspiffheading <- h) spf in
  Tk.pack ~fill:`None ~side:`Top [Tk.coe spiffhereb; Tk.coe headmesg;
                                  Tk.coe headscale];
  (spf, fun () -> ())


(* Open up a level editing window for the given level ************************)
let editlevel lev name top =
  let ed = Toplevel.create top in
  Wm.title_set ed name;
  let currcube = ref (1,1,1) in
  let (vc,updateviewfun,bindingfun) = makelevelview lev currcube ed in

  (* create the auxiliary controls *)
  let (sc,updatewall) = makewallCtrl lev currcube ed in
  let (xc,updatefixed) = makefixedlistCtrl lev currcube updateviewfun ed in
  let (fc,updatefree) = makefreelistCtrl lev currcube updateviewfun ed in
  let (bc,updatebadguys) = makebadguysCtrl lev currcube updateviewfun ed in
  let (pc,updatespiff) = makespiffCtrl lev currcube updateviewfun ed in
  let sframe = Frame.create ~relief:`Groove ed in
  let saveb = Button.create ~command:(asksave lev top name) ~text:"Save..."
                            sframe in
  let closeb = Button.create ~command:(fun () -> Tk.destroy ed) ~text:"Close"
                              sframe in
  Tk.pack ~side:`Left ~fill:`X ~expand:true ~padx:3 ~pady:3 [saveb; closeb];

  (* start the viewer up *)
  bindingfun (fun () -> updatewall ();
                        updatefixed ();
                        updatefree ();
                        updatebadguys ();
                        updatespiff ());

  (* and finally pack the controls in *)
  Tk.pack ~side:`Top ~fill:`X ~expand:true ~padx:3 ~pady:3
          [vc; sc; xc; fc; bc; pc; sframe]


(* attempt to open the given level, and use it to set default x,y,z lengths *)
let tryopen top filename =
  try
    let lev = load_level filename in
    editlevel lev filename top
  with _ -> prerr_endline "Could not open file"


(* pop up a file selector for opening level files *)
let askopen top () =
  fileselect ~title:"Open level"
    ~action:(fun fn -> tryopen top fn) ~actionname:"Open" top


(* returns a new empty cube *)
let newempty () =
  { w = Empty;
    fixed = [];
    free = [];
    badg = [];
    dmtimer = 0.0;
    scent = 0.0}


(* returns a new solid cube with default walls *)
let newsolid () =
  { w = Solid([| Wall.Blocks; Wall.Blocks; Wall.Tiles;
                 Wall.Stone; Wall.Blocks; Wall.Blocks |]);
    fixed = [];
    free = [];
    badg = [];
    dmtimer = 0.0;
    scent = 0.0}


(* create a new level and open a window for editing it *)
let newlev lenx leny lenz top () =
  let lev = {
    currtime = 0.0;
    lenx = !lenx;
    leny = !leny;
    lenz = !lenz;
    layout = Array.make (!lenx * !leny * !lenz) (newempty ());
    dmcubes = [];
    dmfixed = [];
    dmfree = [];
    freeobj = [];
    badguys = [];
    initspiffx = 1.5;
    initspiffy = 1.300001;
    initspiffz = 1.5;
    initspiffheading = 0.0} in
  (* now we have to make sure each cube is a distinct object, and that *)
  (* the outer walls are solid (so Spiff can't see "outside") *)
  for i = 0 to !lenx-1 do
    for j = 0 to !leny-1 do
      for k = 0 to !lenz-1 do
        setcube lev i j k
          (if i=0 || i= !lenx-1 || j=0 || j= !leny-1 || k=0 || k= !lenz-1 then
            newsolid ()
          else
            newempty ())
      done
    done
  done;
  editlevel lev "untitled" top


(* open up the main menu window *)
let openmainwin () =
  let top = Tk.openTk () in
  let f = Frame.create ~borderwidth:2 ~relief:`Groove top in
  let lenx = ref 3 and leny = ref 3 and lenz = ref 3 in
  let lb = Button.create ~command:(askopen top) ~text:"Open..." f in
  let nb = Button.create ~command:(newlev lenx leny lenz top)
                         ~text:"New..." f in
  let changelen len s = len := int_of_float s in
  let xf = Frame.create f in
  let xm = Message.create ~text:"x:" xf in
  let xs = Scale.create ~command:(changelen lenx) ~digits:2 ~min:3.0 ~max:25.0
                        ~showvalue:true ~orient:`Horizontal xf in
  let yf = Frame.create f in
  let ym = Message.create ~text:"y:" yf in
  let ys = Scale.create ~command:(changelen leny) ~digits:2 ~min:3.0 ~max:25.0
                        ~showvalue:true ~orient:`Horizontal yf in
  let zf = Frame.create f in
  let zm = Message.create ~text:"z:" zf in
  let zs = Scale.create ~command:(changelen lenz) ~digits:2 ~min:3.0 ~max:25.0
                        ~showvalue:true ~orient:`Horizontal zf in
  let qb = Button.create ~command:Tk.closeTk ~text:"Quit" top in
  Tk.pack ~fill:`X ~side:`Top ~padx:5 ~pady:5 [lb; nb];
  Tk.pack ~fill:`Both ~side:`Left [Tk.coe xm; Tk.coe xs];
  Tk.pack ~fill:`Both ~side:`Left [Tk.coe ym; Tk.coe ys];
  Tk.pack ~fill:`Both ~side:`Left [Tk.coe zm; Tk.coe zs];
  Tk.pack ~fill:`X ~side:`Top [xf; yf; zf];
  Tk.pack ~fill:`X ~side:`Top ~padx:5 ~pady:5 ~ipadx:5 ~ipady:5 [f];
  Tk.pack ~fill:`X ~padx:20 ~pady:5 ~side:`Top [qb];
  Wm.title_set top "Spiff level editor";
  Wm.resizable_set top ~width:false ~height:false;
  top


let main () = 
  let top = openmainwin () in
  Printexc.print Tk.mainLoop ()

let _ = main ()

