(* collision.ml: implementation of axis-aligned bounding box collision
 * detection and response *)

type hull = {
  mutable x: float;   (* position of center of axis-aligned box *)
  mutable y: float;
  mutable z: float;
  lx: float;          (* half side lengths *)
  ly: float;
  lz: float
}


(* Actions that may result from a collision *)
type action = DoNothing | Teleport | LoseChance | TeleCharge | GetBribe
            | LoseGame


(* Given two objects, return whether ob1 is resting on top of ob2 *)
let topcollision ob1 ob2 tol =
  if ob1 == ob2 then 1.0 (* arb pos indicating an object is not on itself *)
  else if ob1.x +. ob1.lx > ob2.x -. ob2.lx
    & ob2.x +. ob2.lx > ob1.x -. ob1.lx
    & ob1.z +. ob1.lz > ob2.z -. ob2.lz
    & ob2.z +. ob2.lz > ob1.z -. ob1.lz
    & ob1.y -. ob1.ly +. tol > ob2.y +. ob2.ly
  then
    ob1.y -. ob1.ly -. (ob2.y +. ob2.ly +. tol)
  else
    1.0  (* arbitrary positive value *)


(* Given two objects with a proposed movement for the first, return whether
 * or not the objects will collide. *)
let incollision ob1 ob2 =
  if ob1 == ob2 then false  (* an object can't collide with itself *)
  else
    ob1.x +. ob1.lx > ob2.x -. ob2.lx
    & ob2.x +. ob2.lx > ob1.x -. ob1.lx
    & ob1.y +. ob1.ly > ob2.y -. ob2.ly
    & ob2.y +. ob2.ly > ob1.y -. ob1.ly
    & ob1.z +. ob1.lz > ob2.z -. ob2.lz
    & ob2.z +. ob2.lz > ob1.z -. ob1.lz


(* Now the hard part: given two hulls and a proposed motion, deflect it a
 * little to make sure there are no collisions *)


(* first figure out gaps between the hulls in any axis direction *)

type gaprelation = GapGuaranteed | NoGap | GapShrinking of float

let findrelation ob1 dx dy dz ob2 =
  if ob1 == ob2 then   (* an object can't collide with itself *)
    (GapGuaranteed, GapGuaranteed, GapGuaranteed)
  else begin
    let gapx =
      if ob1.x +. ob1.lx > ob2.x -. ob2.lx +. 0.001
      && ob2.x +. ob2.lx > ob1.x -. ob1.lx +. 0.001 then
        NoGap
      else if ob1.x > ob2.x then
        let gap = (ob1.x-.ob1.lx)-.(ob2.x+.ob2.lx) in
        if dx >= -.gap then
          GapGuaranteed
        else
          GapShrinking gap
      else
        let gap = (ob2.x-.ob2.lx)-.(ob1.x+.ob1.lx) in
        if dx <= gap then
          GapGuaranteed
        else
          GapShrinking gap

    and gapy =
      if ob1.y +. ob1.ly > ob2.y -. ob2.ly +. 0.001
      && ob2.y +. ob2.ly > ob1.y -. ob1.ly +. 0.001 then
        NoGap
      else if ob1.y > ob2.y then
        let gap = (ob1.y-.ob1.ly)-.(ob2.y+.ob2.ly) in
        if dy >= -.gap then
          GapGuaranteed
        else
          GapShrinking gap
      else
        let gap = (ob2.y-.ob2.ly)-.(ob1.y+.ob1.ly) in
        if dy <= gap then
          GapGuaranteed
        else
          GapShrinking gap

    and gapz =
      if ob1.z +. ob1.lz > ob2.z -. ob2.lz +. 0.001
      && ob2.z +. ob2.lz > ob1.z -. ob1.lz +. 0.001 then
        NoGap
      else if ob1.z > ob2.z then
        let gap = (ob1.z-.ob1.lz)-.(ob2.z+.ob2.lz) in
        if dz >= -.gap then
          GapGuaranteed
        else
          GapShrinking gap
      else
        let gap = (ob2.z-.ob2.lz)-.(ob1.z+.ob1.lz) in
        if dz <= gap then
          GapGuaranteed
        else
          GapShrinking gap
    in

    (gapx, gapy, gapz)
  end


(* Given a hull, a proposed movement, and a list of possibly colliding hulls
 * (that aren't in collision before the movement), return a deflection of that
 * movement that avoids collision.
 *)
let deflect ob dx dy dz oblist =
  let ax = abs_float dx
  and ay = abs_float dy
  and az = abs_float dz in

  (* figure out for each object how we can avert collision *)
  let gaplist = List.map ~f:(findrelation ob dx dy dz) oblist in

  (* now sweep through the list, bridging gaps only where essential, and
   * eliminating uncolliding objects *)
  let (gx,gy,gz,gaplist) = List.fold_left
    ~f:(fun (gx,gy,gz,newlist) gaps ->
          match gaps with
            GapGuaranteed, _, _ -> (gx, gy, gz, newlist)
          | _, GapGuaranteed, _ -> (gx, gy, gz, newlist)
          | _, _, GapGuaranteed -> (gx, gy, gz, newlist)
          | GapShrinking(g), NoGap, NoGap -> (min (0.9*.g) gx, gy,gz,newlist)
          | NoGap, GapShrinking(g), NoGap -> (gx, min (0.9*.g) gy,gz,newlist)
          | NoGap, NoGap, GapShrinking(g) -> (gx,gy, min (0.9*.g) gz,newlist)
          | GapShrinking(g1), GapShrinking(g2), GapShrinking(g3) ->
              if g1>gx || g2>gy || g3>gz then  (gx,gy,gz,newlist)
              else                             (gx,gy,gz,gaps::newlist)
          | GapShrinking(g1), GapShrinking(g2), NoGap ->
              if g1>gx || g2>gy then  (gx,gy,gz,newlist)
              else                    (gx,gy,gz,gaps::newlist)
          | GapShrinking(g1), NoGap, GapShrinking(g3) ->
              if g1>gx || g3>gz then  (gx,gy,gz,newlist)
              else                    (gx,gy,gz,gaps::newlist)
          | NoGap, GapShrinking(g2), GapShrinking(g3) ->
              if g2>gy || g3>gz then  (gx,gy,gz,newlist)
              else                    (gx,gy,gz,gaps::newlist)
          | NoGap, NoGap, NoGap -> (* this should never happen... *)
              (gx, gy, gz, newlist) )
    ~init:(ax, ay, az, []) gaplist in

  (* now all that's left in gaplist is objects with more than one gap,
   * where we might have to make a choice. (Some may already be ok though) *)
  let (gx,gy,gz) = List.fold_left
    ~f:(fun (gx,gy,gz) gaps ->
          match gaps with
            GapShrinking(g1), GapShrinking(g2), GapShrinking(g3) ->
              if g1 > gx || g2 > gy || g3 > gz then
                (gx, gy, gz)   (* the gap has already been guaranteed *)
              else if g1 >= g2 && g1 >= g3 then
                (min (0.9*.g1) gx, gy, gz)
              else if g2 >= g3 then
                (gx, min (0.9*.g2) gy, gz)
              else 
                (gx, gy, min (0.9*.g3) gz)

          | GapShrinking(g1), GapShrinking(g2), NoGap ->
              if g1 > gx || g2 > gy then
                (gx, gy, gz)   (* the gap has already been guaranteed *)
              else if g1 >= g2 then
                (min (0.9*.g1) gx, gy, gz)
              else
                (gx, min (0.9*.g2) gy, gz)

          | GapShrinking(g1), NoGap, GapShrinking(g3) ->
              if g1 > gx || g3 > gz then
                (gx, gy, gz)   (* the gap has already been guaranteed *)
              else if g1 >= g3 then
                (min (0.9*.g1) gx, gy, gz)
              else
                (gx, gy, min (0.9*.g3) gz)

          | NoGap, GapShrinking(g2), GapShrinking(g3) ->
              if g2 > gy || g3 > gz then
                (gx, gy, gz)   (* the gap has already been guaranteed *)
              else if g2 >= g3 then
                (gx, min (0.9*.g2) gy, gz)
              else
                (gx, gy, min (0.9*.g3) gz)

          | _ -> (prerr_endline "Weird unsatisfied case"; (gx, gy, gz))
        )
    ~init:(gx,gy,gz) gaplist in

  (* try to avoid squeezing gaps to 0 where roundoff is a major concern *)
  let gx = if gx < 0.001 then 0.0 else gx
  and gy = if gy < 0.001 then 0.0 else gy
  and gz = if gz < 0.001 then 0.0 else gz in

  let dx = if dx > 0.0 then gx else -.gx
  and dy = if dy > 0.0 then gy else -.gy
  and dz = if dz > 0.0 then gz else -.gz in
  (dx, dy, dz)

