/* kiddyhop.c

CS248 project by Sean Bailey 11/2000

NOTE: IF the compilation doesn't work, make sure you are using
	OpenGL v1.1

*/

///////////////////////////////////////////////////////////////////////////////
//INCLUDES
///////////////////////////////////////////////////////////////////////////////

#include <time.h>
#include <math.h>
#include <windows.h>
#include <Gl/glut.h>
#include <stdlib.h>
#include <stdio.h>
#include <mmsystem.h>
#include <conio.h>

///////////////////////////////////////////////////////////////////////////////
//CONSTANTS
///////////////////////////////////////////////////////////////////////////////

//General
#define TRUE 1
#define FALSE 0
#define HOPPER 0
#define SPHERE 1
#define PI 3.14159265359

//Window
#define WINDOW_WIDTH 800.0
#define WINDOW_HEIGHT 600.0
#define SPIN_SLICE_WIDTH 100.0

#define VIEW_ANGLE_CHANGE 3.0*PI/2.0
#define MAX_BLUR_HOPPER 40
#define BLUR_HOPPER_SCALE 0.8
#define HOPPERSCALE 1.0/16.0
#define STANDALONE_BEHAVIOR_START_TIME 25.0  //seconds
#define HOLD_TIME 1.1  //seconds
#define EPSILON 0.0001

//Colors
#define BACKGROUND_COLOR 0.65, 0.55, 1.0, 1.0
//Balls (joints)
#define BALL_DIFFUSE_COLOR { 1.0, 0.82, 0.35, 1.0 }
//#define BALL_SPECULAR_COLOR { 1.0, 1.0, 1.0, 1.0 }  //plastic
#define BALL_SPECULAR_COLOR BALL_DIFFUSE_COLOR  //metal
#define BALL_SHININESS { 75 }
//Limbs
#define LIMB_DIFFUSE_COLOR { 0.75, 0.0, 0.0, 1.0 }
//#define LIMB_SPECULAR_COLOR { 1.0, 1.0, 1.0, 1.0 }  //plastic
#define LIMB_SPECULAR_COLOR LIMB_DIFFUSE_COLOR  //metal
#define LIMB_SHININESS { 50 }

///////////////////////////////////////////////////////////////////////////////
//STRUCTURES
///////////////////////////////////////////////////////////////////////////////

typedef struct  {
	double X;
	double Y;
	double Z;
} PositionStruct;

typedef struct  {
	//Positions
   PositionStruct Hip;
   PositionStruct Knee;
   PositionStruct Ankle;
   PositionStruct Foot;
   //Velocity (only need hip)
   PositionStruct HipVelocity;
   //Simulation Parameters
   double Stiffness;
   double EquilibriumLength;
   double Damping;
   double ActiveForce;
   double Mass;
   double Gravity;
   //Wall Parameters
	double HorizontalForce;
   double HorizontalDamping;
	//Hopper Sizes
   double AnkleRadius;
   double TibiaRadius;
   double KneeRadius;
   double FemurRadius;
   double HipRadius;
   double TibiaLength;
   double FemurLength;
   double TrunkLength;
} HopperStruct;


///////////////////////////////////////////////////////////////////////////////
//ENUMERATIONS
///////////////////////////////////////////////////////////////////////////////
enum TouchEnum { Left, Right, Top, Bottom };


///////////////////////////////////////////////////////////////////////////////
//GLOBALS
///////////////////////////////////////////////////////////////////////////////

//Viewing
static double ViewAngle = 0, InitialViewAngle = 0;
//Quadrant 1 = [0,90), Quadrant2 = [90,180), etc
static int ViewQuadrant = 1;  //needs to be initialized when viewing position is defined
static int Panning = FALSE;
static int HorizontalSlice = 1;
static int VerticalSlice = 1;

//Options
static int DisplayInfoOn = TRUE;
static int TextureOn = TRUE;
static int Model = HOPPER;
static int FramesPerSecond = 60;
static int LevelOfDetail = 40;
static double GravityPercent = 0.7;

//Hopper
static HopperStruct Hopper = {
	//Positions
   {0.0, 0.0, 0.5}, //Hip (m)
   {0.0, 0.5, 0.495}, //Knee (m) (positioned later)
   {0.0, 0.0, 0.49}, //Ankle (m) (positioned later)
   {0.0, 0.0, 0.485}, //Foot (m) (positioned later)
	//Velocity (only need hip)
   {0.0, 0.0, 0.0}, //HipVelocity
   //Simulation Parameters
   250.0, //Stiffness (N/m)
   4.45 *HOPPERSCALE, //EquilibriumLength (m)
   10.0, //Damping (N s/m)
   40.0, //ActiveForce (N)
   1.0, //Mass (kg)
   9.81, //Gravity (m/s^2)
   //Horizontal Parameters
	4.0, //HorizontalForce (N/m)
   6.0, //HorizontalDamping (N s/m)
	//Hopper Sizes
   0.5 *HOPPERSCALE, //AnkleRadius (m)
   0.2 *HOPPERSCALE, //TibiaRadius (m)
   0.5 *HOPPERSCALE, //KneeRadius (m)
   0.2 *HOPPERSCALE, //FemurRadius (m)
   1.0 *HOPPERSCALE, //HipRadius (m)
   2.0 *HOPPERSCALE, //TibiaLength (m)
   2.0 *HOPPERSCALE, //FemurLength (m)
   2.0 *HOPPERSCALE //TrunkLength (m)
};
static int FootContact = FALSE;
static int ForceOn = FALSE;
enum TouchEnum VerticalTouch, HorizontalTouch;
static double HorizontalForceX = 0;
static double HorizontalForceY = 0;

//For texture
int TextureWidth;
int TextureHeight;
static GLubyte * TextureImage = NULL;
#ifdef GL_VERSION_1_1
static GLuint texName;
#endif

//For motion blur
HopperStruct BlurHopper[MAX_BLUR_HOPPER];
static int BlurHopperInitialized = FALSE;
static int BlurTimeFrames = 5;
static int PanBlurTimeFrames = 15;
static double BlurTransparency = 0.025;  //crappy monitor
//static double BlurTransparency = 0.009;  //good monitor
static double BlurTransparencyDecay = 0.957;
static double TempBlurTransparency, TempBlurTransparencyDecay;
static double PanBlurTransparency = 0.1;
static double PanBlurTransparencyDecay = 0.95;

//For timing
static DWORD InitialTime, CurrentTime;
static double TimeSinceLastInteraction = 0.0;
static double TimeSinceDisplayRefresh = 0.0;
static int ActualFramesPerSecond = 0;
static int HoldTimeCounter = 0;
static double PanTime = 3.0; //seconds
static double PanningTime;
static DWORD InitialPanningTime, CurrentPanningTime, PreviousPanningTime;

//Color/Material properties
//Common Values
//Ankle
GLfloat AnkleAmbient[] = { 0.0, 0.0, 0.0, 1.0 };
GLfloat AnkleDiffuse[] = BALL_DIFFUSE_COLOR;
GLfloat AnkleSpecular[] = BALL_SPECULAR_COLOR;
GLfloat AnkleEmission[] = { 0.0, 0.0, 0.0, 1.0 };
GLfloat AnkleShininess[] = BALL_SHININESS;
//Tibia
GLfloat TibiaAmbient[] = { 0.0, 0.0, 0.0, 1.0 };
GLfloat TibiaDiffuse[] = LIMB_DIFFUSE_COLOR;
GLfloat TibiaSpecular[] = LIMB_SPECULAR_COLOR;
GLfloat TibiaEmission[] = { 0.0, 0.0, 0.0, 1.0 };
GLfloat TibiaShininess[] = LIMB_SHININESS;
//Knee
GLfloat KneeAmbient[] = { 0.0, 0.0, 0.0, 1.0 };
GLfloat KneeDiffuse[] = BALL_DIFFUSE_COLOR;
GLfloat KneeSpecular[] = BALL_SPECULAR_COLOR;
GLfloat KneeEmission[] = { 0.0, 0.0, 0.0, 1.0 };
GLfloat KneeShininess[] = BALL_SHININESS;
//Femur
GLfloat FemurAmbient[] = { 0.0, 0.0, 0.0, 1.0 };
GLfloat FemurDiffuse[] = LIMB_DIFFUSE_COLOR;
GLfloat FemurSpecular[] = LIMB_SPECULAR_COLOR;
GLfloat FemurEmission[] = { 0.0, 0.0, 0.0, 1.0 };
GLfloat FemurShininess[] = LIMB_SHININESS;
//Hip
GLfloat HipAmbient[] = { 0.0, 0.0, 0.0, 1.0 };
GLfloat HipDiffuse[] = BALL_DIFFUSE_COLOR;
GLfloat HipSpecular[] = BALL_SPECULAR_COLOR;
GLfloat HipEmission[] = { 0.0, 0.0, 0.0, 1.0 };
GLfloat HipShininess[] = BALL_SHININESS;
//Trunk
GLfloat TrunkAmbient[] = { 0.0, 0.0, 0.0, 1.0 };
GLfloat TrunkDiffuse[] = BALL_DIFFUSE_COLOR;
GLfloat TrunkSpecular[] = BALL_SPECULAR_COLOR;
GLfloat TrunkEmission[] = { 0.0, 0.0, 0.0, 1.0 };
GLfloat TrunkShininess[] = BALL_SHININESS;


///////////////////////////////////////////////////////////////////////////////
//FUNCTION PROTOTYPES
///////////////////////////////////////////////////////////////////////////////
double GetVerticalAcceleration( void );
void GetXYHorizontalForce( void );
double GetXAcceleration( void );
double GetYAcceleration( void );
void MoveHopper( void );
void PanRight( void );
void PanLeft( void );
void Mouse( int button, int state, int MouseX, int MouseY );
void MouseMotion( int MouseX, int MouseY );
void DrawHopper( void );
void Display(void);
void DisplayInfo( double DeltaTime );
void Keyboard( unsigned char Key, int X, int Y );
void TextureInit( void );
void LightingInit( void );
void DepthInit( void );
void Reshape (int Width, int Height);
GLubyte* glmReadPPM(char* filename, int* width, int* height);
void CALLBACK errorCallback(GLenum errorCode);


///////////////////////////////////////////////////////////////////////////////
//FUNCTIONS
///////////////////////////////////////////////////////////////////////////////


double GetVerticalAcceleration( void )  {
	double ZDotDot;

	ZDotDot = -Hopper.Gravity;  //ZDotDot = -g
	if( FootContact )  {
   	ZDotDot +=
      	//-k(L-z)
      	-Hopper.Stiffness/Hopper.Mass*(Hopper.Hip.Z-Hopper.EquilibriumLength)
      	//-b ZDot
      	-Hopper.Damping*Hopper.HipVelocity.Z
         //+g (so that it is more stable, less dependant on k and b values)
			+GravityPercent*Hopper.Gravity;
		if( ForceOn && (Hopper.HipVelocity.Z > 0) )  {
	   	ZDotDot +=
   	   	// ActiveForce
      		Hopper.ActiveForce;
      }
   }
   return( ZDotDot );
}


void GetXYHorizontalForce( void )  {

   //Get temp locations of moved spring ends
   //dependant on where the scene is viewed from
	switch( ViewQuadrant )  {
   	case 1: //x is pointing at the viewer, y to the right
			//force vector points to side opposite of touch
			if( HorizontalTouch == Left )
         	HorizontalForceY = Hopper.HorizontalForce;
         else
         	HorizontalForceY = -Hopper.HorizontalForce;
			if( VerticalTouch == Bottom )
         	HorizontalForceX = -Hopper.HorizontalForce;
         else
         	HorizontalForceX = Hopper.HorizontalForce;
			break;
   	case 2: //y is pointing at the viewer, x to the left
			//force vector points to side opposite of touch
			if( HorizontalTouch == Left )
         	HorizontalForceX = -Hopper.HorizontalForce;
         else
         	HorizontalForceX = Hopper.HorizontalForce;
			if( VerticalTouch == Bottom )
         	HorizontalForceY = -Hopper.HorizontalForce;
         else
         	HorizontalForceY = Hopper.HorizontalForce;
			break;
   	case 3: //x is pointing away from the viewer, y to the left
			//force vector points to side opposite of touch
			if( HorizontalTouch == Left )
         	HorizontalForceY = -Hopper.HorizontalForce;
         else
         	HorizontalForceY = Hopper.HorizontalForce;
			if( VerticalTouch == Bottom )
         	HorizontalForceX = Hopper.HorizontalForce;
         else
         	HorizontalForceX = -Hopper.HorizontalForce;
			break;
   	case 4: //y is pointing away from the viewer, x to the right
			//force vector points to side opposite of touch
			if( HorizontalTouch == Left )
         	HorizontalForceX = Hopper.HorizontalForce;
         else
         	HorizontalForceX = -Hopper.HorizontalForce;
			if( VerticalTouch == Bottom )
         	HorizontalForceY = Hopper.HorizontalForce;
         else
         	HorizontalForceY = -Hopper.HorizontalForce;
			break;
   }
}


double GetXAcceleration( void )  {
	double XDotDot;

  	XDotDot = 1/Hopper.Mass * (
		//HorizontalForce due to touch
		HorizontalForceX
     	//-b XDot
     	- Hopper.HorizontalDamping*Hopper.HipVelocity.X
		);

   return( XDotDot );

}

double GetYAcceleration( void )  {
	double YDotDot;

  	YDotDot = 1/Hopper.Mass * (
		//HorizontalForce due to touch
		HorizontalForceY
     	//-b YDot
     	- Hopper.HorizontalDamping*Hopper.HipVelocity.Y
		);

   return( YDotDot );

}


void MoveHopper( void )  {
	//Dynamics are not exact, but good enough for graphics
   static double XDotDot, YDotDot, ZDotDot;
	static double DeltaTime;
   static DWORD PreviousTime;
   static double OldHopperHipX, OldHopperHipVelocityX;
   static double AX, BX, CX, DX, AXDot, BXDot, CXDot, DXDot;
   static double OldHopperHipY, OldHopperHipVelocityY;
   static double AY, BY, CY, DY, AYDot, BYDot, CYDot, DYDot;
   static double OldHopperHipZ, OldHopperHipVelocityZ;
   static double AZ, BZ, CZ, DZ, AZDot, BZDot, CZDot, DZDot;
   static double RAnkleHipZ, RAnkleKneeZ, RAnkleKneeY;
   static double RandomNumber1, RandomNumber2;

	//Get timing info
	PreviousTime = CurrentTime;  //Since it has yet to be updated
   CurrentTime = timeGetTime();
   DeltaTime = ( (double)(CurrentTime - PreviousTime) )/1000.0;  //convert to ms
   TimeSinceDisplayRefresh += DeltaTime;
	TimeSinceLastInteraction += DeltaTime;

	///////////////////////////////
	//Standalone Behavior Generator
	///////////////////////////////

   //Do standalone behavior if the user hasn't done anything for a while
   if( (TimeSinceLastInteraction > STANDALONE_BEHAVIOR_START_TIME) &&
   	( (int)(TimeSinceLastInteraction/HOLD_TIME) > HoldTimeCounter) )  {
      //HoldTimeCounter keeps the last randomly generated command current for a while (more pleasing)
      HoldTimeCounter = (int)(TimeSinceLastInteraction/HOLD_TIME);
		RandomNumber1 = rand()/(float)RAND_MAX;
		RandomNumber2 = rand()/(float)RAND_MAX;
		//ForceOn
      if( RandomNumber1 > 0.3 )
      	ForceOn = TRUE;
      else
      	ForceOn = FALSE;
      //HorizontalTouch
      if( RandomNumber1 > .5 )
        	HorizontalTouch = Left;
		else
        	HorizontalTouch = Right;
      //VerticalTouch
      if( RandomNumber2 > .5 )
        	VerticalTouch = Top;
		else
        	VerticalTouch = Bottom;
      //Rotate scene or move hopper - seldom rotates scene
		if( RandomNumber1 < 0.9 )
      	glutIdleFunc( MoveHopper );
      else if( RandomNumber1 < .95 )  {
         Panning = TRUE;
			//Initialize then change transparency
			TempBlurTransparency = BlurTransparency;
			TempBlurTransparencyDecay = BlurTransparencyDecay;
			BlurTransparency = PanBlurTransparency;
			BlurTransparencyDecay *= PanBlurTransparencyDecay;
		   //Initialize panning time
			InitialPanningTime = timeGetTime();
		   CurrentPanningTime = InitialPanningTime;
			PanningTime = 0.0;
		   //Record initial view angle
			InitialViewAngle = ViewAngle;
      	glutIdleFunc( PanLeft );
      }  else  {
         Panning = TRUE;
			//Initialize then change transparency
			TempBlurTransparency = BlurTransparency;
			TempBlurTransparencyDecay = BlurTransparencyDecay;
			BlurTransparency = PanBlurTransparency;
			BlurTransparencyDecay *= PanBlurTransparencyDecay;
		   //Initialize panning time
			InitialPanningTime = timeGetTime();
		   CurrentPanningTime = InitialPanningTime;
			PanningTime = 0.0;
		   //Record initial view angle
			InitialViewAngle = ViewAngle;
      	glutIdleFunc( PanRight );
		}
		//Make sure that the number doesn't overflow
      if( TimeSinceLastInteraction > 10000.0 )  {  //~3 hours
      	TimeSinceLastInteraction = STANDALONE_BEHAVIOR_START_TIME + EPSILON;
			HoldTimeCounter = 0;
      }
	}

	//////////////////////////
	//Deal with Vertical First
	//////////////////////////

	//Runge Kutta 4th order integration

   //Store old values
   OldHopperHipZ = Hopper.Hip.Z;
   OldHopperHipVelocityZ = Hopper.HipVelocity.Z;

   //A = DeltaTime*y_dot(t0,yi)
   //Values already initialized
   //Get Z acceleration ( y_dot(t0,yi) )
	ZDotDot = GetVerticalAcceleration();
   //Get A
   //DeltaTime*y_dot(t0,yi)
   AZ = Hopper.HipVelocity.Z*DeltaTime;
   AZDot = ZDotDot*DeltaTime;


   //B = DeltaTime*y_dot(t0+.5DeltaTime,yi+.5A)
   //Initialize Values
   Hopper.HipVelocity.Z = OldHopperHipVelocityZ + 0.5*AZDot;
   Hopper.Hip.Z = OldHopperHipZ + 0.5*AZ;
   //Get Z acceleration ( y_dot(t0,yi) )
	ZDotDot = GetVerticalAcceleration();
   //Get B
   //DeltaTime*y_dot(t0+.5DeltaTime,yi+.5A)
   BZ = Hopper.HipVelocity.Z*DeltaTime;
   BZDot = ZDotDot*DeltaTime;

   //C = DeltaTime*y_dot(t0+.5DeltaTime,yi+.5B)
   //Initialize Values
   Hopper.HipVelocity.Z = OldHopperHipVelocityZ + 0.5*BZDot;
   Hopper.Hip.Z = OldHopperHipZ + 0.5*BZ;
   //Get Z acceleration ( y_dot(t0,yi) )
	ZDotDot = GetVerticalAcceleration();
   //Get C
   //DeltaTime*y_dot(t0+.5DeltaTime,yi+.5B)
   CZ = Hopper.HipVelocity.Z*DeltaTime;
   CZDot = ZDotDot*DeltaTime;

   //D = DeltaTime*y_dot(t0+DeltaTime,yi+C)
   //Initialize Values
   Hopper.HipVelocity.Z = OldHopperHipVelocityZ + CZDot;
   Hopper.Hip.Z = OldHopperHipZ + CZ;
   //Get Z acceleration ( y_dot(t0,yi) )
	ZDotDot = GetVerticalAcceleration();
   //Get D
   //DeltaTime*y_dot(t0+DeltaTime,yi+C)
   DZ = Hopper.HipVelocity.Z*DeltaTime;
   DZDot = ZDotDot*DeltaTime;

   //Finally get the new Values
   //y = 1/6( A + 2B + 2C + D )
   Hopper.HipVelocity.Z = OldHopperHipVelocityZ +
   	1.0/6.0*( AZDot + 2.0*BZDot + 2.0*CZDot + DZDot );
   Hopper.Hip.Z = OldHopperHipZ +
   	1.0/6.0*( AZ + 2.0*BZ + 2.0*CZ + DZ );

   //Clamp Hip Values if need be
	if( Hopper.Hip.Z > (10 - Hopper.TrunkLength) )  { //way too high
		Hopper.Hip.Z = (10 - Hopper.TrunkLength - EPSILON );
		Hopper.HipVelocity.Z = 0;
   }
	if( Hopper.Hip.Z < (Hopper.HipRadius+Hopper.AnkleRadius) )  { //foot and hip interpenetrate
		Hopper.Hip.Z = (Hopper.HipRadius+Hopper.AnkleRadius+EPSILON);
		Hopper.HipVelocity.Z = 0;
	}

   //Find Foot Z Location, FootContact, and clamp if in contact
   Hopper.Foot.Z = Hopper.Hip.Z - Hopper.EquilibriumLength;
   if( Hopper.Foot.Z <= 0 )  {
   	Hopper.Foot.Z = 0;
      FootContact = TRUE;
   }  else  {
   	FootContact = FALSE;
   }

	//////////////////////////
	//Deal with Horizontal Now
	//////////////////////////

   if( ForceOn && !FootContact )  {
   	//In the air and the user is touching the screen in the middle
      GetXYHorizontalForce();

		//Integrate to get new x y positions and velocities (of the hip)
		//Runge Kutta 4th order integration

   	//Store old values
	   OldHopperHipX = Hopper.Hip.X;
	   OldHopperHipVelocityX = Hopper.HipVelocity.X;
	   OldHopperHipY = Hopper.Hip.Y;
	   OldHopperHipVelocityY = Hopper.HipVelocity.Y;

  	   //A = DeltaTime*y_dot(t0,yi)
	   //Values already initialized
   	//Get acceleration ( y_dot(t0,yi) )
		XDotDot = GetXAcceleration();
		YDotDot = GetYAcceleration();
	   //Get A
   	//DeltaTime*y_dot(t0,yi)
	   AX = Hopper.HipVelocity.X*DeltaTime;
   	AXDot = XDotDot*DeltaTime;
	   AY = Hopper.HipVelocity.Y*DeltaTime;
   	AYDot = YDotDot*DeltaTime;

	   //B = DeltaTime*y_dot(t0+.5DeltaTime,yi+.5A)
   	//Initialize Values
	   Hopper.HipVelocity.X = OldHopperHipVelocityX + 0.5*AXDot;
   	Hopper.Hip.X = OldHopperHipX + 0.5*AX;
	   Hopper.HipVelocity.Y = OldHopperHipVelocityY + 0.5*AYDot;
   	Hopper.Hip.Y = OldHopperHipY + 0.5*AY;
	   //Get acceleration ( y_dot(t0,yi) )
		XDotDot = GetXAcceleration();
		YDotDot = GetYAcceleration();
   	//Get B
	   //DeltaTime*y_dot(t0+.5DeltaTime,yi+.5A)
   	BX = Hopper.HipVelocity.X*DeltaTime;
	   BXDot = XDotDot*DeltaTime;
   	BY = Hopper.HipVelocity.Y*DeltaTime;
	   BYDot = YDotDot*DeltaTime;

	   //C = DeltaTime*y_dot(t0+.5DeltaTime,yi+.5B)
   	//Initialize Values
	   Hopper.HipVelocity.X = OldHopperHipVelocityX + 0.5*BXDot;
   	Hopper.Hip.X = OldHopperHipX + 0.5*BX;
	   Hopper.HipVelocity.Y = OldHopperHipVelocityY + 0.5*BYDot;
   	Hopper.Hip.Y = OldHopperHipY + 0.5*BY;
	   //Get acceleration ( y_dot(t0,yi) )
		XDotDot = GetXAcceleration();
		YDotDot = GetYAcceleration();
   	//Get C
	   //DeltaTime*y_dot(t0+.5DeltaTime,yi+.5B)
   	CX = Hopper.HipVelocity.X*DeltaTime;
	   CXDot = XDotDot*DeltaTime;
   	CY = Hopper.HipVelocity.Y*DeltaTime;
	   CYDot = YDotDot*DeltaTime;

	   //D = DeltaTime*y_dot(t0+DeltaTime,yi+C)
   	//Initialize Values
	   Hopper.HipVelocity.X = OldHopperHipVelocityX + CXDot;
   	Hopper.Hip.X = OldHopperHipX + CX;
	   Hopper.HipVelocity.Y = OldHopperHipVelocityY + CYDot;
   	Hopper.Hip.Y = OldHopperHipY + CY;
	   //Get acceleration ( y_dot(t0,yi) )
		XDotDot = GetXAcceleration();
	 	YDotDot = GetYAcceleration();
	   //Get D
   	//DeltaTime*y_dot(t0+DeltaTime,yi+C)
	   DX = Hopper.HipVelocity.X*DeltaTime;
   	DXDot = XDotDot*DeltaTime;
	   DY = Hopper.HipVelocity.Y*DeltaTime;
   	DYDot = YDotDot*DeltaTime;

	   //Finally get the new Values
   	//y = 1/6( A + 2B + 2C + D )
	   Hopper.HipVelocity.X = OldHopperHipVelocityX +
   		1.0/6.0*( AXDot + 2.0*BXDot + 2.0*CXDot + DXDot );
	   Hopper.Hip.X = OldHopperHipX +
   		1.0/6.0*( AX + 2.0*BX + 2.0*CX + DX );
	   Hopper.HipVelocity.Y = OldHopperHipVelocityY +
   		1.0/6.0*( AYDot + 2.0*BYDot + 2.0*CYDot + DYDot );
	   Hopper.Hip.Y = OldHopperHipY +
   		1.0/6.0*( AY + 2.0*BY + 2.0*CY + DY );

   }  else if( !ForceOn && !FootContact )  {
   	//in the air, but the user is not longer touching the screen
      //Velocity stays the same, but position changes
      //No need for Runge Kutta method since velocity doesn't change over dt
	   //Simple Newton Integration
	   Hopper.Hip.X += Hopper.HipVelocity.X*DeltaTime;
	   Hopper.Hip.Y += Hopper.HipVelocity.Y*DeltaTime;

   }  else  {  //FootContact
   	//not in the air, user input doesn't matter
		//Positions stay the same
      //Clamp velocities to zero
	   //Hopper.Hip.X = Hopper.Hip.X;
      Hopper.HipVelocity.X = 0;
	   //Hopper.Hip.Y = Hopper.Hip.Y;
      Hopper.HipVelocity.Y = 0;
	}

   //Clamp positions and reverse velocities (perfect elastic collisions) if need be
   if( Hopper.Hip.X > 0.5 )  {
		//Too high x
   	Hopper.Hip.X = 0.5 - EPSILON;
		if( ForceOn )
      	Hopper.HipVelocity.X = 0;  //don't want it to jitter, so just clamp it to zero
      else
      	Hopper.HipVelocity.X = -Hopper.HipVelocity.X;  //perfect elastic collision
   }  else if( Hopper.Hip.X < -0.5 )  {
		//Too low x
   	Hopper.Hip.X = -0.5 + EPSILON;
		if( ForceOn )
      	Hopper.HipVelocity.X = 0;  //don't want it to jitter, so just clamp it to zero
      else
      	Hopper.HipVelocity.X = -Hopper.HipVelocity.X;  //perfect elastic collision
   }
   if( Hopper.Hip.Y > 0.5 )  {
		//Too high y
   	Hopper.Hip.Y = 0.5 - EPSILON;
		if( ForceOn )
      	Hopper.HipVelocity.Y = 0;  //don't want it to jitter, so just clamp it to zero
      else
      	Hopper.HipVelocity.Y = -Hopper.HipVelocity.X;  //perfect elastic collision
   }  else if( Hopper.Hip.Y < -0.5 )  {
		//Too low y
   	Hopper.Hip.Y = -0.5 + EPSILON;
		if( ForceOn )
      	Hopper.HipVelocity.Y = 0;  //don't want it to jitter, so just clamp it to zero
      else
      	Hopper.HipVelocity.Y = -Hopper.HipVelocity.X;  //perfect elastic collision
	}

	/////////////////////////////////
	//Find Foot, Ankle, etc positions
	/////////////////////////////////

   //Find Foot Position
   Hopper.Foot.X = Hopper.Hip.X;
   Hopper.Foot.Y = Hopper.Hip.Y;
   //Hopper.Foot.Z found above

   //Find Ankle Position
   Hopper.Ankle.X = Hopper.Foot.X;
   Hopper.Ankle.Y = Hopper.Foot.Y;
   Hopper.Ankle.Z = Hopper.Foot.Z + Hopper.AnkleRadius;

	//Find Knee Location (via vectors RFromTo)
   Hopper.Knee.X = Hopper.Ankle.X; //easy (in plane), y and z harder
   //doesn't assume equal Femur and Tibia Lengths
   RAnkleHipZ = Hopper.Hip.Z - Hopper.Ankle.Z;
   RAnkleKneeZ =
   	( pow( Hopper.FemurLength, 2.0 ) - pow( Hopper.TibiaLength, 2 )
      - pow( RAnkleHipZ, 2 ) ) /
      ( -2*RAnkleHipZ );
   //RKneeHipZ = RAnkleHipZ - RAnkleKnee;  //But we don't use this
   RAnkleKneeY = sqrt( pow( Hopper.TibiaLength, 2 ) - pow( RAnkleKneeZ, 2 ) );
   Hopper.Knee.Y = Hopper.Ankle.Y + RAnkleKneeY;
   Hopper.Knee.Z = Hopper.Ankle.Z + RAnkleKneeZ;

	/////////////////////////////////
	//Display refreshing
	/////////////////////////////////

   //Redisplay if past FramesPerSecond period
   if( TimeSinceDisplayRefresh > 1.0/FramesPerSecond )  {
      ActualFramesPerSecond = (int) ( 1.0/TimeSinceDisplayRefresh );
   	TimeSinceDisplayRefresh = 0;
      glutPostRedisplay();
   }

	//Redisplay text info if option selected
	if( DisplayInfoOn )
		DisplayInfo( DeltaTime );
}


void PanRight( void )  {
	static double DeltaPanningTime;

 	//Compute PanningDeltaTime
	PreviousPanningTime = CurrentPanningTime;
	CurrentPanningTime = timeGetTime();
   DeltaPanningTime = ( (double)(CurrentPanningTime - PreviousPanningTime) )
   	/1000.0;  //convert to ms
	PanningTime += DeltaPanningTime;

   //Compute new view angle
	ViewAngle = InitialViewAngle + VIEW_ANGLE_CHANGE*  	//sinusoidal function - should be smoother
   	( 0.5 - 0.5*cos( PI*PanningTime/PanTime ) );

   //if done Clamp and reset and set quadrants
   if( PanningTime >= PanTime )  {
      Panning = FALSE;
	   //Change idle function
   	glutIdleFunc( MoveHopper );
   	//Reset transparency
		BlurTransparency = TempBlurTransparency;
		BlurTransparencyDecay = TempBlurTransparencyDecay;
      //change quadrant and clamp view angle value
      ViewQuadrant += (int) (VIEW_ANGLE_CHANGE/(PI/2) + 0.4);
      ViewAngle = InitialViewAngle + VIEW_ANGLE_CHANGE;
      //Reset angle if needed
	   while( ViewAngle >= 2*PI )  {
   		ViewAngle -= 2*PI;
      	ViewQuadrant -= 4;
	   }
   }

   //Add time to other timers so the motion is natural when reinitiated
   CurrentTime += DeltaPanningTime*1000.0;

   //Redisplay
	glutPostRedisplay();
}


void PanLeft( void )  {
	static double DeltaPanningTime;

 	//Compute PanningDeltaTime
	PreviousPanningTime = CurrentPanningTime;
	CurrentPanningTime = timeGetTime();
   DeltaPanningTime = ( (double)(CurrentPanningTime - PreviousPanningTime) )
   	/1000.0;  //convert to ms
	PanningTime += DeltaPanningTime;

   //Compute new view angle
	ViewAngle = InitialViewAngle - VIEW_ANGLE_CHANGE*  	//sinusoidal function - should be smoother
   	( 0.5 - 0.5*cos( PI*PanningTime/PanTime ) );

   //if done Clamp and reset and set quadrants
   if( PanningTime > PanTime )  {
      Panning = FALSE;
	   //Change idle function
   	glutIdleFunc( MoveHopper );
   	//Reset transparency
		BlurTransparency = TempBlurTransparency;
		BlurTransparencyDecay = TempBlurTransparencyDecay;
      //change quadrant and clamp view angle value
      ViewQuadrant -= (int) (VIEW_ANGLE_CHANGE/(PI/2)+0.4);
      ViewAngle = InitialViewAngle - VIEW_ANGLE_CHANGE;
      //Reset angle if needed
	   while( ViewAngle < 0 )  {
   		ViewAngle += 2*PI;
      	ViewQuadrant += 4;
	   }
   }

   //Add time to other timers so the motion is natural when reinitiated
   CurrentTime += DeltaPanningTime*1000.0;

   //Redisplay
	glutPostRedisplay();
}


void Mouse( int button, int state, int MouseX, int MouseY )  {
	//Make y = 0 at the bottom
	MouseY = WINDOW_HEIGHT - MouseY;
	//Reset time (and assoc. counter) that user has been idle (for standalone behavior)
	TimeSinceLastInteraction = 0.0;
	HoldTimeCounter = 0.0;
   if( !Panning )  {  //Ignore input if currently panning
		//Deal with mouse
		switch( button )  {
   		case GLUT_LEFT_BUTTON:
      		if( state == GLUT_DOWN )  {
	            if( MouseX < SPIN_SLICE_WIDTH )  {
						//Inside the left spin edge
	   	         ForceOn = FALSE;
                  Panning = TRUE;
						//Initialize then change transparency
						TempBlurTransparency = BlurTransparency;
						TempBlurTransparencyDecay = BlurTransparencyDecay;
						BlurTransparency = PanBlurTransparency;
						BlurTransparencyDecay *= PanBlurTransparencyDecay;
					   //Initialize panning time
						InitialPanningTime = timeGetTime();
					   CurrentPanningTime = InitialPanningTime;
						PanningTime = 0.0;
					   //Record initial view angle
						InitialViewAngle = ViewAngle;
         	   	glutIdleFunc( PanLeft );
	            }  else if( MouseX > WINDOW_WIDTH - SPIN_SLICE_WIDTH )  {
						//Inside the right spin edge
	   	         ForceOn = FALSE;
                  Panning = TRUE;
						//Initialize then change transparency
						TempBlurTransparency = BlurTransparency;
						TempBlurTransparencyDecay = BlurTransparencyDecay;
						BlurTransparency = PanBlurTransparency;
						BlurTransparencyDecay *= PanBlurTransparencyDecay;
					   //Initialize panning time
						InitialPanningTime = timeGetTime();
					   CurrentPanningTime = InitialPanningTime;
						PanningTime = 0.0;
					   //Record initial view angle
						InitialViewAngle = ViewAngle;
         	   	glutIdleFunc( PanRight );
            	}  else  {
						//Turn forces on
		            ForceOn = TRUE;
						//Figure out whether on top or bottom
         	      if( MouseY > 0.4*WINDOW_HEIGHT )
            	   	VerticalTouch = Top;
						else
               		VerticalTouch = Bottom;
						//Figure out whether on left or right
   	            if( MouseX > 0.5*WINDOW_WIDTH )
      	         	HorizontalTouch = Right;
						else
            	   	HorizontalTouch = Left;
					}
   	      }  else  {  //  state == GLUT_UP (button released)
         	   ForceOn = FALSE;
					glutIdleFunc( MoveHopper );
	         }
   	   default:
      		break;
	   }
	}
}


void MouseMotion( int MouseX, int MouseY )  {
	//Make y = 0 at the bottom
	MouseY = WINDOW_HEIGHT - MouseY;
	//Reset time (and assoc. counter) that user has been idle (for standalone behavior)
	TimeSinceLastInteraction = 0.0;
	HoldTimeCounter = 0.0;
   if( !Panning )  {  //Ignore input if currently panning
		//Deal with mouse
      if( MouseX < 1.25*SPIN_SLICE_WIDTH )  {
			//Inside the left spin edge
         ForceOn = FALSE;
         Panning = TRUE;
			//Initialize then change transparency
			TempBlurTransparency = BlurTransparency;
			TempBlurTransparencyDecay = BlurTransparencyDecay;
			BlurTransparency = PanBlurTransparency;
			BlurTransparencyDecay *= PanBlurTransparencyDecay;
		   //Initialize panning time
			InitialPanningTime = timeGetTime();
		   CurrentPanningTime = InitialPanningTime;
			PanningTime = 0.0;
		   //Record initial view angle
			InitialViewAngle = ViewAngle;
  	   	glutIdleFunc( PanLeft );
      }  else if( MouseX > WINDOW_WIDTH - 1.25*SPIN_SLICE_WIDTH )  {
			//Inside the right spin edge
         ForceOn = FALSE;
         Panning = TRUE;
			//Initialize then change transparency
			TempBlurTransparency = BlurTransparency;
			TempBlurTransparencyDecay = BlurTransparencyDecay;
			BlurTransparency = PanBlurTransparency;
			BlurTransparencyDecay *= PanBlurTransparencyDecay;
		   //Initialize panning time
			InitialPanningTime = timeGetTime();
		   CurrentPanningTime = InitialPanningTime;
			PanningTime = 0.0;
		   //Record initial view angle
			InitialViewAngle = ViewAngle;
  	   	glutIdleFunc( PanRight );
		}  else if( MouseX < 0.3*WINDOW_WIDTH )  {
	         ForceOn = TRUE;
         	HorizontalTouch = Left;
      }  else if( MouseX > 0.7*WINDOW_WIDTH )  {
	         ForceOn = TRUE;
         	HorizontalTouch = Right;
      }  else  {
	         ForceOn = TRUE;
		}
	}
}


void DrawHopper( void )  {
	GLUquadricObj *Cylinder;
   //Set up Hopper vector variables
   double RFootAnkleX = 0.0;
   double RFootAnkleY = 0.0;
   double RFootAnkleZ = Hopper.AnkleRadius;
   double RAnkleKneeX = Hopper.Knee.X - Hopper.Ankle.X;
   double RAnkleKneeY = Hopper.Knee.Y - Hopper.Ankle.Y;
   double RAnkleKneeZ = Hopper.Knee.Z - Hopper.Ankle.Z;
   double ThetaTibiaScene = -180/PI*atan2( RAnkleKneeY, RAnkleKneeZ );
   double RKneeHipX = Hopper.Hip.X - Hopper.Knee.X;
   double RKneeHipY = Hopper.Hip.Y - Hopper.Knee.Y;
   double RKneeHipZ = Hopper.Hip.Z - Hopper.Knee.Z;
   double ThetaFemurScene = -180/PI*atan2( RKneeHipY, RKneeHipZ );

		//Draw Hopper
      //Initialize Quadric Objects (cylinders)
      Cylinder = gluNewQuadric();
		gluQuadricCallback( Cylinder, GLU_ERROR, errorCallback );
		gluQuadricDrawStyle( Cylinder, GLU_FILL );
		gluQuadricNormals( Cylinder, GLU_SMOOTH );
		//Start Drawing
      glPushMatrix();  //From Origin of scene
         //Go to foot
			glTranslatef( Hopper.Foot.X, Hopper.Foot.Y, Hopper.Foot.Z );
         glPushMatrix();  //From Foot
         	//Go to Ankle
            glTranslatef( RFootAnkleX, RFootAnkleY, RFootAnkleZ );
				//Draw Tibia
            glPushMatrix();  //From Knee
              	glRotatef( ThetaTibiaScene, 1.0, 0.0, 0.0 );
					//Materials
				   glMaterialfv( GL_FRONT, GL_AMBIENT, TibiaAmbient );
				   glMaterialfv( GL_FRONT, GL_DIFFUSE, TibiaDiffuse );
				   glMaterialfv( GL_FRONT, GL_SPECULAR, TibiaSpecular );
				   glMaterialfv( GL_FRONT, GL_EMISSION, TibiaEmission );
			   	glMaterialfv( GL_FRONT, GL_SHININESS, TibiaShininess );
               gluCylinder( Cylinder, Hopper.TibiaRadius,
                	Hopper.TibiaRadius, Hopper.TibiaLength,
                 	(int) ( 1+LevelOfDetail*sqrt( Hopper.TibiaRadius ) ),
                  (int) ( 1+LevelOfDetail*sqrt( Hopper.TibiaRadius ) ) );
            glPopMatrix();  //Back to knee
				glPushMatrix();  //From Ankle
					//Go to Knee
               glTranslatef( RAnkleKneeX, RAnkleKneeY, RAnkleKneeZ );
					//Draw Femur
               glPushMatrix();  //From Knee
               	glRotatef( ThetaFemurScene, 1.0, 0.0, 0.0 );
						//Materials
					   glMaterialfv( GL_FRONT, GL_AMBIENT, FemurAmbient );
					   glMaterialfv( GL_FRONT, GL_DIFFUSE, FemurDiffuse );
					   glMaterialfv( GL_FRONT, GL_SPECULAR, FemurSpecular );
					   glMaterialfv( GL_FRONT, GL_EMISSION, FemurEmission );
			   		glMaterialfv( GL_FRONT, GL_SHININESS, FemurShininess );
                  gluCylinder( Cylinder, Hopper.FemurRadius,
                  	Hopper.FemurRadius, Hopper.FemurLength,
                  	(int) ( 1+LevelOfDetail*sqrt( Hopper.FemurRadius ) ),
                     (int) ( 1+LevelOfDetail*sqrt( Hopper.FemurRadius ) ) );
               glPopMatrix();  //Back to knee
               glPushMatrix();  //From Knee
               	//Go to Hip
               	glTranslatef( RKneeHipX, RKneeHipY, RKneeHipZ );
 						//Draw Body Cylinder
						//Materials
					   glMaterialfv( GL_FRONT, GL_AMBIENT, TrunkAmbient );
					   glMaterialfv( GL_FRONT, GL_DIFFUSE, TrunkDiffuse );
					   glMaterialfv( GL_FRONT, GL_SPECULAR, TrunkSpecular );
					   glMaterialfv( GL_FRONT, GL_EMISSION, TrunkEmission );
			   		glMaterialfv( GL_FRONT, GL_SHININESS, TrunkShininess );
                  gluCylinder( Cylinder, Hopper.HipRadius,
                  	Hopper.HipRadius, Hopper.TrunkLength,
                  	(int) ( 1+LevelOfDetail*sqrt( Hopper.HipRadius ) ),
                     (int) ( 1+LevelOfDetail*sqrt( Hopper.HipRadius ) ) );
                  glPushMatrix();  //From Hip
                  	//Draw any body detail
                  glPopMatrix();  //Back to Hip
                  //Draw hip sphere
						//Materials
					   glMaterialfv( GL_FRONT, GL_AMBIENT, HipAmbient );
					   glMaterialfv( GL_FRONT, GL_DIFFUSE, HipDiffuse );
					   glMaterialfv( GL_FRONT, GL_SPECULAR, HipSpecular );
					   glMaterialfv( GL_FRONT, GL_EMISSION, HipEmission );
			   		glMaterialfv( GL_FRONT, GL_SHININESS, HipShininess );
			         glutSolidSphere( Hopper.HipRadius,
                  	(int) ( 1+LevelOfDetail*sqrt( Hopper.HipRadius ) ),
                     (int) ( 1+LevelOfDetail*sqrt( Hopper.HipRadius ) ) );
               glPopMatrix();  //Back to Knee
					//Draw knee sphere
					//Materials
				   glMaterialfv( GL_FRONT, GL_AMBIENT, KneeAmbient );
				   glMaterialfv( GL_FRONT, GL_DIFFUSE, KneeDiffuse );
				   glMaterialfv( GL_FRONT, GL_SPECULAR, KneeSpecular );
				   glMaterialfv( GL_FRONT, GL_EMISSION, KneeEmission );
		   		glMaterialfv( GL_FRONT, GL_SHININESS, KneeShininess );
		         glutSolidSphere( Hopper.KneeRadius,
                 	(int) ( 1+LevelOfDetail*sqrt( Hopper.KneeRadius ) ),
                  (int) ( 1+LevelOfDetail*sqrt( Hopper.KneeRadius ) ) );
            glPopMatrix();  //Back to Ankle
            //Draw ankle sphere
				//Materials
			   glMaterialfv( GL_FRONT, GL_AMBIENT, AnkleAmbient );
			   glMaterialfv( GL_FRONT, GL_DIFFUSE, AnkleDiffuse );
			   glMaterialfv( GL_FRONT, GL_SPECULAR, AnkleSpecular );
			   glMaterialfv( GL_FRONT, GL_EMISSION, AnkleEmission );
	   		glMaterialfv( GL_FRONT, GL_SHININESS, AnkleShininess );
	         glutSolidSphere( Hopper.AnkleRadius,
              	(int) ( 1+LevelOfDetail*sqrt( Hopper.AnkleRadius ) ),
               (int) ( 1+LevelOfDetail*sqrt( Hopper.AnkleRadius ) ) );
         glPopMatrix();  //Back to Foot
      glPopMatrix();  //Back to Origin of Scene
}


void DrawBlurHopperN( int BlurHopperIndex )  {
	//Cylinder object
	GLUquadricObj *Cylinder;
   //Set up Hopper vector variables
   double RFootAnkleX = 0.0;
   double RFootAnkleY = 0.0;
   double RFootAnkleZ = BlurHopper[BlurHopperIndex].AnkleRadius;
   double RAnkleKneeX = BlurHopper[BlurHopperIndex].Knee.X -
    	BlurHopper[BlurHopperIndex].Ankle.X;
   double RAnkleKneeY = BlurHopper[BlurHopperIndex].Knee.Y -
     	BlurHopper[BlurHopperIndex].Ankle.Y;
   double RAnkleKneeZ = BlurHopper[BlurHopperIndex].Knee.Z -
     	BlurHopper[BlurHopperIndex].Ankle.Z;
   double ThetaTibiaScene = -180/PI*atan2( RAnkleKneeY, RAnkleKneeZ );
   double RKneeHipX = BlurHopper[BlurHopperIndex].Hip.X -
    	BlurHopper[BlurHopperIndex].Knee.X;
   double RKneeHipY = BlurHopper[BlurHopperIndex].Hip.Y -
    	BlurHopper[BlurHopperIndex].Knee.Y;
   double RKneeHipZ = BlurHopper[BlurHopperIndex].Hip.Z -
     	BlurHopper[BlurHopperIndex].Knee.Z;
   double ThetaFemurScene = -180/PI*atan2( RKneeHipY, RKneeHipZ );
   //Blur
   static int ColorIndex;
   static double BlurAlpha;
   //Color/Material properties for blur
	//Ankle
   GLfloat BlurAnkleAmbient[4], BlurAnkleDiffuse[4], BlurAnkleSpecular[4];
	GLfloat BlurAnkleEmission[4], BlurAnkleShininess[1];
	//Tibia
	GLfloat BlurTibiaAmbient[4], BlurTibiaDiffuse[4], BlurTibiaSpecular[4];
	GLfloat BlurTibiaEmission[4], BlurTibiaShininess[1];
	//Knee
	GLfloat BlurKneeAmbient[4], BlurKneeDiffuse[4], BlurKneeSpecular[4];
	GLfloat BlurKneeEmission[4], BlurKneeShininess[1];
	//Femur
	GLfloat BlurFemurAmbient[4], BlurFemurDiffuse[4];
	GLfloat BlurFemurSpecular[4], BlurFemurEmission[4], BlurFemurShininess[1];
	//Hip
	GLfloat BlurHipAmbient[4], BlurHipDiffuse[4], BlurHipSpecular[4];
	GLfloat BlurHipEmission[4], BlurHipShininess[1];
	//Trunk
	GLfloat BlurTrunkAmbient[4], BlurTrunkDiffuse[4], BlurTrunkSpecular[4];
	GLfloat BlurTrunkEmission[4], BlurTrunkShininess[1];

   //Copy over the material values
	//Shininess
	BlurAnkleShininess[1] = AnkleShininess[1];
	BlurTibiaShininess[1] = TibiaShininess[1];
	BlurKneeShininess[1] = KneeShininess[1];
	BlurFemurShininess[1] = FemurShininess[1];
	BlurHipShininess[1] = HipShininess[1];
	BlurTrunkShininess[1] = TrunkShininess[1];
	//Everything else
   for( ColorIndex = 0; ColorIndex < 3; ColorIndex++ )  {
		//Ankle
		BlurAnkleAmbient[ColorIndex] = AnkleAmbient[ColorIndex];
		BlurAnkleDiffuse[ColorIndex] = AnkleDiffuse[ColorIndex];
		BlurAnkleSpecular[ColorIndex] = AnkleSpecular[ColorIndex];
		BlurAnkleEmission[ColorIndex] = AnkleEmission[ColorIndex];
		//Tibia
		BlurTibiaAmbient[ColorIndex] = TibiaAmbient[ColorIndex];
		BlurTibiaDiffuse[ColorIndex] = TibiaDiffuse[ColorIndex];
		BlurTibiaSpecular[ColorIndex] = TibiaSpecular[ColorIndex];
		BlurTibiaEmission[ColorIndex] = TibiaEmission[ColorIndex];
		//Knee
		BlurKneeAmbient[ColorIndex] = KneeAmbient[ColorIndex];
		BlurKneeDiffuse[ColorIndex] = KneeDiffuse[ColorIndex];
		BlurKneeSpecular[ColorIndex] = KneeSpecular[ColorIndex];
		BlurKneeEmission[ColorIndex] = KneeEmission[ColorIndex];
		//Femur
		BlurFemurAmbient[ColorIndex] = FemurAmbient[ColorIndex];
		BlurFemurDiffuse[ColorIndex] = FemurDiffuse[ColorIndex];
		BlurFemurSpecular[ColorIndex] = FemurSpecular[ColorIndex];
		BlurFemurEmission[ColorIndex] = FemurEmission[ColorIndex];
		//Hip
		BlurHipAmbient[ColorIndex] = HipAmbient[ColorIndex];
		BlurHipDiffuse[ColorIndex] = HipDiffuse[ColorIndex];
		BlurHipSpecular[ColorIndex] = HipSpecular[ColorIndex];
		BlurHipEmission[ColorIndex] = HipEmission[ColorIndex];
		//Trunk
		BlurTrunkAmbient[ColorIndex] = TrunkAmbient[ColorIndex];
		BlurTrunkDiffuse[ColorIndex] = TrunkDiffuse[ColorIndex];
		BlurTrunkSpecular[ColorIndex] = TrunkSpecular[ColorIndex];
		BlurTrunkEmission[ColorIndex] = TrunkEmission[ColorIndex];
   }

   //Set the Alpha value for decaying
   BlurAlpha = BlurTransparency * pow( BlurTransparencyDecay, BlurHopperIndex );
   ColorIndex = 3;
	//Ankle
	BlurAnkleAmbient[ColorIndex] = BlurAlpha;
	BlurAnkleDiffuse[ColorIndex] = BlurAlpha;
	BlurAnkleSpecular[ColorIndex] = BlurAlpha;
	BlurAnkleEmission[ColorIndex] = BlurAlpha;
	//Tibia
	BlurTibiaAmbient[ColorIndex] = BlurAlpha;
	BlurTibiaDiffuse[ColorIndex] = BlurAlpha;
	BlurTibiaSpecular[ColorIndex] = BlurAlpha;
	BlurTibiaEmission[ColorIndex] = BlurAlpha;
	//Knee
	BlurKneeAmbient[ColorIndex] = BlurAlpha;
	BlurKneeDiffuse[ColorIndex] = BlurAlpha;
	BlurKneeSpecular[ColorIndex] = BlurAlpha;
	BlurKneeEmission[ColorIndex] = BlurAlpha;
	//Femur
	BlurFemurAmbient[ColorIndex] = BlurAlpha;
	BlurFemurDiffuse[ColorIndex] = BlurAlpha;
	BlurFemurSpecular[ColorIndex] = BlurAlpha;
	BlurFemurEmission[ColorIndex] = BlurAlpha;
	//Hip
	BlurHipAmbient[ColorIndex] = BlurAlpha;
	BlurHipDiffuse[ColorIndex] = BlurAlpha;
	BlurHipSpecular[ColorIndex] = BlurAlpha;
	BlurHipEmission[ColorIndex] = BlurAlpha;
	//Trunk
	BlurTrunkAmbient[ColorIndex] = BlurAlpha;
	BlurTrunkDiffuse[ColorIndex] = BlurAlpha;
	BlurTrunkSpecular[ColorIndex] = BlurAlpha;
	BlurTrunkEmission[ColorIndex] = BlurAlpha;

		//Draw Hopper
      //Initialize Quadric Objects (cylinders)
      Cylinder = gluNewQuadric();
		gluQuadricCallback( Cylinder, GLU_ERROR, errorCallback );
		gluQuadricDrawStyle( Cylinder, GLU_FILL );
		gluQuadricNormals( Cylinder, GLU_SMOOTH );
		//Start Drawing
      glPushMatrix();  //From Origin of scene
         //Go to foot
			glTranslatef( BlurHopper[BlurHopperIndex].Foot.X,
         	BlurHopper[BlurHopperIndex].Foot.Y,
            BlurHopper[BlurHopperIndex].Foot.Z );
         glPushMatrix();  //From Foot
         	//Go to Ankle
            glTranslatef( RFootAnkleX, RFootAnkleY, RFootAnkleZ );
				//Draw Tibia
            glPushMatrix();  //From Knee
              	glRotatef( ThetaTibiaScene, 1.0, 0.0, 0.0 );
					//Materials
				   glMaterialfv( GL_FRONT, GL_AMBIENT, BlurTibiaAmbient );
				   glMaterialfv( GL_FRONT, GL_DIFFUSE, BlurTibiaDiffuse );
				   glMaterialfv( GL_FRONT, GL_SPECULAR, BlurTibiaSpecular );
				   glMaterialfv( GL_FRONT, GL_EMISSION, BlurTibiaEmission );
			   	glMaterialfv( GL_FRONT, GL_SHININESS, BlurTibiaShininess );
               gluCylinder( Cylinder, BlurHopper[BlurHopperIndex].TibiaRadius,
                	BLUR_HOPPER_SCALE*BlurHopper[BlurHopperIndex].TibiaRadius,
                  BlurHopper[BlurHopperIndex].TibiaLength,
                 	(int) ( 1+LevelOfDetail*sqrt( BlurHopper[BlurHopperIndex].TibiaRadius ) ),
                  (int) ( 1+LevelOfDetail*sqrt( BlurHopper[BlurHopperIndex].TibiaRadius ) ) );
            glPopMatrix();  //Back to knee
				glPushMatrix();  //From Ankle
					//Go to Knee
               glTranslatef( RAnkleKneeX, RAnkleKneeY, RAnkleKneeZ );
					//Draw Femur
               glPushMatrix();  //From Knee
               	glRotatef( ThetaFemurScene, 1.0, 0.0, 0.0 );
						//Materials
					   glMaterialfv( GL_FRONT, GL_AMBIENT, BlurFemurAmbient );
					   glMaterialfv( GL_FRONT, GL_DIFFUSE, BlurFemurDiffuse );
					   glMaterialfv( GL_FRONT, GL_SPECULAR, BlurFemurSpecular );
					   glMaterialfv( GL_FRONT, GL_EMISSION, BlurFemurEmission );
			   		glMaterialfv( GL_FRONT, GL_SHININESS, BlurFemurShininess );
                  gluCylinder( Cylinder, BlurHopper[BlurHopperIndex].FemurRadius,
                  	BLUR_HOPPER_SCALE*BlurHopper[BlurHopperIndex].FemurRadius,
                     BlurHopper[BlurHopperIndex].FemurLength,
                  	(int) ( 1+LevelOfDetail*sqrt( BlurHopper[BlurHopperIndex].FemurRadius ) ),
                     (int) ( 1+LevelOfDetail*sqrt( BlurHopper[BlurHopperIndex].FemurRadius ) ) );
               glPopMatrix();  //Back to knee
               glPushMatrix();  //From Knee
               	//Go to Hip
               	glTranslatef( RKneeHipX, RKneeHipY, RKneeHipZ );
 						//Draw Body Cylinder
						//Materials
					   glMaterialfv( GL_FRONT, GL_AMBIENT, BlurTrunkAmbient );
					   glMaterialfv( GL_FRONT, GL_DIFFUSE, BlurTrunkDiffuse );
					   glMaterialfv( GL_FRONT, GL_SPECULAR, BlurTrunkSpecular );
					   glMaterialfv( GL_FRONT, GL_EMISSION, BlurTrunkEmission );
			   		glMaterialfv( GL_FRONT, GL_SHININESS, BlurTrunkShininess );
                  gluCylinder( Cylinder,
                  	BLUR_HOPPER_SCALE*BlurHopper[BlurHopperIndex].HipRadius,
                  	BLUR_HOPPER_SCALE*BlurHopper[BlurHopperIndex].HipRadius,
                     BlurHopper[BlurHopperIndex].TrunkLength,
                  	(int) ( 1+LevelOfDetail*sqrt( BlurHopper[BlurHopperIndex].HipRadius ) ),
                     (int) ( 1+LevelOfDetail*sqrt( BlurHopper[BlurHopperIndex].HipRadius ) ) );
                  glPushMatrix();  //From Hip
                  	//Draw any body detail
                  glPopMatrix();  //Back to Hip
                  //Draw hip sphere
						//Materials
					   glMaterialfv( GL_FRONT, GL_AMBIENT, BlurHipAmbient );
					   glMaterialfv( GL_FRONT, GL_DIFFUSE, BlurHipDiffuse );
					   glMaterialfv( GL_FRONT, GL_SPECULAR, BlurHipSpecular );
					   glMaterialfv( GL_FRONT, GL_EMISSION, BlurHipEmission );
			   		glMaterialfv( GL_FRONT, GL_SHININESS, BlurHipShininess );
			         glutSolidSphere(
                  	BLUR_HOPPER_SCALE*BlurHopper[BlurHopperIndex].HipRadius,
                  	(int) ( 1+LevelOfDetail*sqrt( BlurHopper[BlurHopperIndex].HipRadius ) ),
                     (int) ( 1+LevelOfDetail*sqrt( BlurHopper[BlurHopperIndex].HipRadius ) ) );
               glPopMatrix();  //Back to Knee
					//Draw knee sphere
					//Materials
				   glMaterialfv( GL_FRONT, GL_AMBIENT, BlurKneeAmbient );
				   glMaterialfv( GL_FRONT, GL_DIFFUSE, BlurKneeDiffuse );
				   glMaterialfv( GL_FRONT, GL_SPECULAR, BlurKneeSpecular );
				   glMaterialfv( GL_FRONT, GL_EMISSION, BlurKneeEmission );
		   		glMaterialfv( GL_FRONT, GL_SHININESS, BlurKneeShininess );
		         glutSolidSphere(
               	BLUR_HOPPER_SCALE*BlurHopper[BlurHopperIndex].KneeRadius,
                 	(int) ( 1+LevelOfDetail*sqrt( BlurHopper[BlurHopperIndex].KneeRadius ) ),
                  (int) ( 1+LevelOfDetail*sqrt( BlurHopper[BlurHopperIndex].KneeRadius ) ) );
            glPopMatrix();  //Back to Ankle
            //Draw ankle sphere
				//Materials
			   glMaterialfv( GL_FRONT, GL_AMBIENT, BlurAnkleAmbient );
			   glMaterialfv( GL_FRONT, GL_DIFFUSE, BlurAnkleDiffuse );
			   glMaterialfv( GL_FRONT, GL_SPECULAR, BlurAnkleSpecular );
			   glMaterialfv( GL_FRONT, GL_EMISSION, BlurAnkleEmission );
	   		glMaterialfv( GL_FRONT, GL_SHININESS, BlurAnkleShininess );
	         glutSolidSphere(
            	BLUR_HOPPER_SCALE*Hopper.AnkleRadius,
              	(int) ( 1+LevelOfDetail*sqrt( BlurHopper[BlurHopperIndex].AnkleRadius ) ),
               (int) ( 1+LevelOfDetail*sqrt( BlurHopper[BlurHopperIndex].AnkleRadius ) ) );
         glPopMatrix();  //Back to Foot
      glPopMatrix();  //Back to Origin of Scene
}


void Display(void)  {
	//Light positions
   GLfloat Light00Position[] = { 0.0, 0.0, 0.0, 1.0 };
   GLfloat Light01Position[] = { 0.0, 0.0, 0.0, 1.0 };
   GLfloat Light02Position[] = { 0.0, 0.0, 0.0, 1.0 };
   //Motion blur
   int BlurHopperIndex;

	//Clear all pixels and depth buffer
   glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

   //Set viewing position and draw everything
   //Everything is drawn relative to the origin
   //The origin is the center of the floor
   //z is up, ignoring normal convention
   glPushMatrix();  //From real origin

   	//Set viewing position
      if( Panning )  {
   		gluLookAt ( 2.0*cos( ViewAngle ), 2.0*sin( ViewAngle ), 0.5, //eye x y z
   			(1.0-pow( 2.0*(PanningTime-PanTime/2.0)/PanTime, 2 ))*Hopper.Hip.X,
            	(1.0-pow( 2.0*(PanningTime-PanTime/2.0)/PanTime, 2 ))*Hopper.Hip.Y,
					0.5*pow( 2.0*(PanningTime-PanTime/2.0)/PanTime, 2 )
               + (1.0-pow( 2.0*(PanningTime-PanTime/2.0)/PanTime, 2 ))*Hopper.Hip.Z
               ,  //look at x y z
	     		0.0, 0.0, 1.0);  //up x y z
   	}  else  {
   		gluLookAt ( 2.0*cos( ViewAngle ), 2.0*sin( ViewAngle ), 0.5, //eye x y z
   			0.0, 0.0, 0.5,  //look at x y z
	     		0.0, 0.0, 1.0);  //up x y z
		}

		//Position Light00 and Light00 Box
      glPushMatrix();
   		glTranslatef( 1.5*1.414, 0.0, 0.3 );
		   glLightfv (GL_LIGHT0, GL_POSITION, Light00Position);
   		glDisable( GL_LIGHTING );
		   glColor4f( 1.0, 0.0, 0.0, 1.0 );
   		//glutWireCube (0.01);
   		glEnable (GL_LIGHTING);
   	glPopMatrix ();

		//Position Light01 and Light01 Box
      glPushMatrix();
   		glTranslatef( 1.5*(-0.5), 1.5*0.5, 0.3 );
		   glLightfv (GL_LIGHT1, GL_POSITION, Light01Position);
   		glDisable( GL_LIGHTING );
		   glColor4f( 1.0, 0.0, 0.0, 1.0 );
   		//glutWireCube (0.01);
   		glEnable (GL_LIGHTING);
   	glPopMatrix ();

		//Position Light02 and Light02 Box
      glPushMatrix();
   		glTranslatef( 1.5*(-0.5), 1.5*(-0.5), 0.3 );
		   glLightfv (GL_LIGHT2, GL_POSITION, Light02Position);
   		glDisable( GL_LIGHTING );
		   glColor4f( 1.0, 0.0, 0.0, 1.0 );
   		//glutWireCube (0.01);
   		glEnable (GL_LIGHTING);
   	glPopMatrix ();

      //Draw floor
      if( TextureOn )  {
	      //Enable Texturing
   	   glEnable( GL_TEXTURE_2D );
			//Set the texturing function (see p. 402 of red book) to overwrite
		   glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);
			#ifdef GL_VERSION_1_1
      		//Make this texture current(?)
	   		glBindTexture(GL_TEXTURE_2D, texName);
			#endif
      	//Start drawing quadruple of vertices (which make a 4 sided polygon)
	   	glBegin(GL_QUADS);
   			glTexCoord2f(0.0, 0.0); glVertex3f(-0.5, -0.5, 0.0);
   			glTexCoord2f(2.0, 0.0); glVertex3f(0.5, -0.5, 0.0);
   			glTexCoord2f(2.0, 2.0); glVertex3f(0.5, 0.5, 0.0);
	   		glTexCoord2f(0.0, 2.0); glVertex3f(-0.5, 0.5, 0.0);
   		glEnd();
      	//Disable the texturing
		   glDisable(GL_TEXTURE_2D);
      }  else  {  // !TextureOn
      	//Start drawing quadruple of vertices (which make a 4 sided polygon)
	   	glBegin(GL_QUADS);
				glVertex3f(-0.5, -0.5, 0.0);
   			glVertex3f(-0.5, 0.5, 0.0);
   			glVertex3f(0.5, 0.5, 0.0);
	   		glVertex3f(0.5, -0.5, 0.0);
   		glEnd();
		}

      //Draw the opaque hopper
      DrawHopper();

		//Transparent objects - DRAW AFTER ALL OPAQUE OBJECTS!!!
		//Drawing the hopper and its dimmed cousins for blurring
      //Make sure that the blurred hopper array has amiable values for drawing
		if( !BlurHopperInitialized )  {
      	BlurHopperInitialized = TRUE;
      	for( BlurHopperIndex = 0; BlurHopperIndex < MAX_BLUR_HOPPER;
         	BlurHopperIndex++ )  {
         	BlurHopper[BlurHopperIndex] = Hopper;
         }
      }
      //Enable blending
      glEnable( GL_BLEND );
      //Make the depth buffer read only
		glDepthMask( GL_FALSE );
      //Set Blending function
      glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
      //Loop thru and draw blurred hoppers
		if( !Panning )  {  //if not panning, shift the array and store new
			for( BlurHopperIndex = BlurTimeFrames-1; BlurHopperIndex >= 0;
   	   	BlurHopperIndex-- )  {
      		DrawBlurHopperN( BlurHopperIndex );
			}
      }  else  {  //Draw more if panning
			for( BlurHopperIndex = PanBlurTimeFrames-1; BlurHopperIndex >= 0;
   	   	BlurHopperIndex-- )  {
      		DrawBlurHopperN( BlurHopperIndex );
			}
		}
      //Store the hoppers in an array
		if( !Panning )  {  //if not panning, shift the array and store new
			for( BlurHopperIndex = MAX_BLUR_HOPPER-1; BlurHopperIndex >= 0;
   	   	BlurHopperIndex-- )  {
         	if( BlurHopperIndex != 0 )  //shift up in the array
         		BlurHopper[BlurHopperIndex] = BlurHopper[BlurHopperIndex-1];
	         else  //Put the latest at index 0
   	      	BlurHopper[BlurHopperIndex] = Hopper;
      	}
      }
		//Set the depth buffer back to writeable
      glDepthMask( GL_TRUE );
      //Disable blending
      glDisable( GL_BLEND );

   glPopMatrix();  //Back to Real Origin

   //Swap the buffers to update the display
 	glutSwapBuffers();

}


void DisplayInfo( double DeltaTime )  {
	static int Row = 0;

	//Clear screen
	clrscr();

	//Start display
   Row++; //gotoxy(1,Row);
   	textcolor(LIGHTRED); cprintf( "KiddyHop" );
      printf( "\n" );

   Row++; //gotoxy(1,Row);
   	textcolor(CYAN); cprintf( "Toggle this display (i): " );
   	textcolor(WHITE); cprintf( "%d\n", DisplayInfoOn );
      printf( "\n" );

   Row++; //gotoxy(1,Row);
   	textcolor(YELLOW); cprintf( "Timing" );
      printf( "\n" );

   Row++; //gotoxy(1,Row);
   	textcolor(CYAN); cprintf( "   DeltaTime: " );
   	textcolor(WHITE); cprintf( "%f", DeltaTime );
   	printf( "\n" );

   Row++; //gotoxy(1,Row);
   	textcolor(CYAN); cprintf( "   Frames Per Second (R/r): " );
   	textcolor(WHITE); cprintf( "%d", FramesPerSecond );
   	printf( "\n" );

   Row++; //gotoxy(1,Row);
   	textcolor(CYAN); cprintf( "   Actual Frames Per Second: " );
   	textcolor(WHITE); cprintf( "%d", ActualFramesPerSecond );
   	printf( "\n" );

   Row++; //gotoxy(1,Row);
   	textcolor(YELLOW); cprintf( "Detail parameters" );
      printf( "\n" );

   Row++; //gotoxy(1,Row);
   	textcolor(CYAN); cprintf( "   Textures (t): " );
   	textcolor(WHITE); cprintf( "%d", TextureOn );
      printf( "\n" );

   Row++; //gotoxy(1,Row);
   	textcolor(CYAN); cprintf( "   LevelOfDetail (+/-): " );
   	textcolor(WHITE); cprintf( "%d", LevelOfDetail );
      printf( "\n" );

   Row++; //gotoxy(1,Row);
   	textcolor(YELLOW); cprintf( "Simulation parameters" );
      printf( "\n" );

   Row++; //gotoxy(1,Row);
   	textcolor(CYAN); cprintf( "   Stiffness (K/k): " );
   	textcolor(WHITE); cprintf( "%f", Hopper.Stiffness );
      printf( "\n" );

   Row++; //gotoxy(1,Row);
   	textcolor(CYAN); cprintf( "   Damping (B/b): " );
   	textcolor(WHITE); cprintf( "%f", Hopper.Damping );
      printf( "\n" );

   Row++; //gotoxy(1,Row);
   	textcolor(CYAN); cprintf( "   Active Force (F/f): " );
   	textcolor(WHITE); cprintf( "%f", Hopper.ActiveForce );
      printf( "\n" );

   Row++; //gotoxy(1,Row);
   	textcolor(CYAN); cprintf( "   Mass (M/m): " );
   	textcolor(WHITE); cprintf( "%f", Hopper.Mass );
      printf( "\n" );

   Row++; //gotoxy(1,Row);
   	textcolor(CYAN); cprintf( "   Gravity (G/g): " );
   	textcolor(WHITE); cprintf( "%f", Hopper.Gravity );
      printf( "\n" );

   Row++; //gotoxy(1,Row);
   	textcolor(CYAN); cprintf( "   Gravity Offset (O/o): " );
   	textcolor(WHITE); cprintf( "%f", GravityPercent );
      printf( "\n" );

   Row++; //gotoxy(1,Row);
   	textcolor(CYAN); cprintf( "   HorizontalForce (H/h): " );
   	textcolor(WHITE); cprintf( "%f", Hopper.HorizontalForce );
      printf( "\n" );

   Row++; //gotoxy(1,Row);
   	textcolor(CYAN); cprintf( "   HorizontalDamping (S/s): " );
   	textcolor(WHITE); cprintf( "%f", Hopper.HorizontalDamping );
      printf( "\n" );

   Row++; //gotoxy(1,Row);
   	textcolor(CYAN); cprintf( "   BlurTimeFrames (U/u): " );
   	textcolor(WHITE); cprintf( "%d", BlurTimeFrames );
      printf( "\n" );

   Row++; //gotoxy(1,Row);
   	textcolor(CYAN); cprintf( "   BlurTransparency (Y/y): " );
   	textcolor(WHITE); cprintf( "%f", BlurTransparency );
      printf( "\n" );

   Row++; //gotoxy(1,Row);
   	textcolor(CYAN); cprintf( "   BlurTransparencyDecay (P/p): " );
   	textcolor(WHITE); cprintf( "%f", BlurTransparencyDecay );
      printf( "\n" );

}


void Keyboard( unsigned char Key, int X, int Y )  {

	switch( Key )  {

		//Display Info
   	case 'i':
      	DisplayInfoOn = DisplayInfoOn ^ TRUE;
			if( !DisplayInfoOn )  {
				//Clear screen
				clrscr();

            //Start display
   			//Row++; //gotoxy(1,Row);
   				textcolor(LIGHTRED); cprintf( "KiddyHop" );
      			printf( "\n" );
            //Row++; //gotoxy(1,Row);
   				printf( "\n" );  //Space
			   //Row++; //gotoxy(1,Row);
   				textcolor(CYAN); cprintf( "Toggle this display (i): " );
   				textcolor(WHITE); cprintf( "%d\n", DisplayInfoOn );
      			printf( "\n" );
         }
         break;

      //Frame rate
   	case 'R':
         FramesPerSecond++;
         break;
   	case 'r':
         FramesPerSecond--;
         break;

      //Texture
   	case 't':
      	TextureOn = TextureOn ^ TRUE;
			break;

      //Level of Detail
   	case '+':
      	LevelOfDetail += 10;
			break;
   	case '-':
      	LevelOfDetail -= 10;
         while( LevelOfDetail < 10 )
	      	LevelOfDetail += 10;
			break;

		//Stiffness
      case 'K':
      	Hopper.Stiffness += 1.0;
         break;
      case 'k':
      	Hopper.Stiffness -= 1.0;
         break;

      //Damping
      case 'B':
      	Hopper.Damping += 0.1;
         break;
      case 'b':
      	Hopper.Damping -= 0.1;
         break;

      //ActiveForce
      case 'F':
      	Hopper.ActiveForce += 0.5;
         break;
      case 'f':
      	Hopper.ActiveForce -= 0.5;
         break;

      //Mass
      case 'M':
      	Hopper.Mass += 0.1;
         break;
      case 'm':
      	Hopper.Mass -= 0.1;
         while( Hopper.Mass <= 0 )  //need to clamp mass since it is a denominator
              Hopper.Mass += 0.1;
         break;

      //Gravity
      case 'G':
      	Hopper.Gravity += 0.1;
         break;
      case 'g':
      	Hopper.Gravity -= 0.1;
         break;

      //Gravity Percent
      case 'O':
      	GravityPercent += 0.01;
         break;
      case 'o':
      	GravityPercent -= 0.01;
         break;

      //HorizontalForce
      case 'H':
      	Hopper.HorizontalForce += 1.0;
         break;
      case 'h':
      	Hopper.HorizontalForce -= 1.0;
         break;

      //HorizontalDamping
      case 'S':
      	Hopper.HorizontalDamping += 0.1;
         break;
      case 's':
      	Hopper.HorizontalDamping -= 0.1;
         break;

      //BlurTimeFrames
      case 'U':
      	BlurTimeFrames++;
         while( BlurTimeFrames >= MAX_BLUR_HOPPER )
         	BlurTimeFrames--;
         break;
      case 'u':
      	BlurTimeFrames--;
         break;

      //BlurTransparency
      case 'Y':
      	BlurTransparency += 0.001;
         break;
      case 'y':
      	BlurTransparency -= 0.001;
         break;

      //BlurTransparencyDecay
      case 'P':
      	BlurTransparencyDecay += 0.001;
         break;
      case 'p':
      	BlurTransparencyDecay -= 0.001;
         break;

		//Quit
      case 'q':
      	exit( 0 );
         break;

   }
}


void TextureInit( void )  {
	//Much code borrowed from OpenGL Programming Guide (aka "red book")
   //example program checker.c

   //Load texture and do something with how OpenGL stores it
	//IMPORTANT! - Texture must be square of size 2^n (p. 363 of redbook)
   TextureImage = glmReadPPM("dragoncc.ppm", &TextureWidth, &TextureHeight);
   glPixelStorei(GL_UNPACK_ALIGNMENT, 1);

   //Name and create texture object
   glGenTextures(1, &texName);
   glBindTexture(GL_TEXTURE_2D, texName);

   //Set texture parameters (not really sure if this needs to be done)
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);

   //Specify the texture
   glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, TextureWidth, TextureHeight,
                0, GL_RGB, GL_UNSIGNED_BYTE, TextureImage);

   //Fix the perspective problem
	glHint( GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST );
}


void LightingInit( void )  {
	//Light 00
	GLfloat Light00Ambient[] = { 1.0, 1.0, 1.0, 1.0 };
	GLfloat Light00Diffuse[] = { 1.0, 1.0, 1.0, 1.0 };
	GLfloat Light00Specular[] = { 1.0, 1.0, 1.0, 1.0 };
	//Light 01
	GLfloat Light01Ambient[] = { 1.0, 1.0, 1.0, 1.0 };
	GLfloat Light01Diffuse[] = { 1.0, 1.0, 1.0, 1.0 };
	GLfloat Light01Specular[] = { 1.0, 1.0, 1.0, 1.0 };
	//Light 01
	GLfloat Light02Ambient[] = { 1.0, 1.0, 1.0, 1.0 };
	GLfloat Light02Diffuse[] = { 1.0, 1.0, 1.0, 1.0 };
	GLfloat Light02Specular[] = { 1.0, 1.0, 1.0, 1.0 };

	//Light 00
	glLightfv( GL_LIGHT0, GL_AMBIENT, Light00Ambient );
	glLightfv( GL_LIGHT0, GL_DIFFUSE, Light00Diffuse );
	glLightfv( GL_LIGHT0, GL_SPECULAR, Light00Specular );
	glShadeModel( GL_SMOOTH );
   glEnable( GL_LIGHTING );
   glEnable( GL_LIGHT0 );

	//Light 01
	glLightfv( GL_LIGHT1, GL_AMBIENT, Light01Ambient );
	glLightfv( GL_LIGHT1, GL_DIFFUSE, Light01Diffuse );
	glLightfv( GL_LIGHT1, GL_SPECULAR, Light01Specular );
	glShadeModel( GL_SMOOTH );
   glEnable( GL_LIGHTING );
   glEnable( GL_LIGHT1 );

	//Light 02
	glLightfv( GL_LIGHT2, GL_AMBIENT, Light02Ambient );
	glLightfv( GL_LIGHT2, GL_DIFFUSE, Light02Diffuse );
	glLightfv( GL_LIGHT2, GL_SPECULAR, Light02Specular );
	glShadeModel( GL_SMOOTH );
   glEnable( GL_LIGHTING );
   glEnable( GL_LIGHT2 );

}


void DepthInit( void )  {
	glEnable( GL_DEPTH_TEST );
}


void Reshape (int Width, int Height)  {
   glViewport (0, 0, (GLsizei) Width, (GLsizei) Height);
   glMatrixMode (GL_PROJECTION);
   glLoadIdentity();
   gluPerspective(40.0, (GLfloat) Width/(GLfloat) Height, 0.0001, 20.0);
   glMatrixMode(GL_MODELVIEW);
   glLoadIdentity();
}


GLubyte* glmReadPPM(char* filename, int* width, int* height)  {
  	//Code copied and slightly modified from:
	/*
      glm.c
      Nate Robins, 1997, 2000
      nate@pobox.com, http://www.pobox.com/~nate

      Wavefront OBJ model file format reader/writer/manipulator.

      Includes routines for generating smooth normals with
      preservation of edges, welding redundant vertices & texture
      coordinate generation (spheremap and planar projections) + more.

	*/
	/* glmReadPPM: read a PPM raw (type P6) file.  The PPM file has a header
	 * that should look something like:
	 *
	 *    P6
	 *    # comment
	 *    width height max_value
  	 *    rgbrgbrgb...
	 *
	 * where "P6" is the magic cookie which identifies the file type and
  	 * should be the only characters on the first line followed by a
	 * carriage return.  Any line starting with a # mark will be treated
	 * as a comment and discarded.   After the magic cookie, three integer
	 * values are expected: width, height of the image and the maximum
	 * value for a pixel (max_value must be < 256 for PPM raw files).  The
	 * data section consists of width*height rgb triplets (one byte each)
	 * in binary format (i.e., such as that written with fwrite() or
	 * equivalent).
	 *
	 * The rgb data is returned as an array of unsigned chars (packed
	 * rgb).  The malloc()'d memory should be free()'d by the caller.  If
	 * an error occurs, an error message is sent to stderr and NULL is
	 * returned.
	 *
	 * filename   - name of the .ppm file.
	 * width      - will contain the width of the image on return.
	 * height     - will contain the height of the image on return.
	 *
	 */

    FILE* fp;
    int i, w, h, d;
    unsigned char* image;
    char head[70];          /* max line <= 70 in PPM (per spec). */

    fp = fopen(filename, "rb");
    if (!fp) {
        perror(filename);
        return NULL;
    }

    /* grab first two chars of the file and make sure that it has the
       correct magic cookie for a raw PPM file. */
    fgets(head, 70, fp);
    if (strncmp(head, "P6", 2)) {
        fprintf(stderr, "%s: Not a raw PPM file\n", filename);
        return NULL;
    }

    /* grab the three elements in the header (width, height, maxval). */
    i = 0;
    while(i < 3) {
        fgets(head, 70, fp);
        if (head[0] == '#')     /* skip comments. */
            continue;
        if (i == 0)
            i += sscanf(head, "%d %d %d", &w, &h, &d);
        else if (i == 1)
            i += sscanf(head, "%d %d", &h, &d);
        else if (i == 2)
            i += sscanf(head, "%d", &d);
    }

    /* grab all the image data in one fell swoop. */
    image = (unsigned char*)malloc(sizeof(unsigned char)*w*h*3);
    fread(image, sizeof(unsigned char), w*h*3, fp);
    fclose(fp);

    *width = w;
    *height = h;
    return image;
}


void CALLBACK errorCallback(GLenum errorCode)  {
	//code copied from redbook...
   const GLubyte *estring;

   estring = gluErrorString(errorCode);
   fprintf(stderr, "Quadric Error: %s\n", estring);
   exit(0);
}


///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
//MAIN
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////

void main( int argc, char** argv )  {
	static int BlurHopperIndex;

	//Deal with lesser versions of OpenGL
	#ifdef GL_VERSION_1_1
		printf( "OpenGL Version 1.1\n" );
	#else
		printf( "Not OpenGL Version 1.1 (go get it www.opengl.org)\n" );
		exit( -1 );
	#endif

   //Initialize BlurHopper array
   for( BlurHopperIndex = 0; BlurHopper < MAX_BLUR_HOPPER; BlurHopperIndex++ )
   	BlurHopper[BlurHopperIndex] = Hopper;

   //Initialize OpenGL
	glutInit( &argc, argv );
	glutInitDisplayMode( GLUT_DOUBLE | GLUT_RGB | GLUT_DEPTH );
	glutInitWindowSize( WINDOW_WIDTH, WINDOW_HEIGHT );
	glutInitWindowPosition( 0, 0 );
	glutCreateWindow( "KiddyHop" );

   //Initialize Background Color, Texture, Depth, and Lighting info
   glClearColor( BACKGROUND_COLOR );
	LightingInit();
   DepthInit();
   TextureInit();

   //Initialize time
   InitialTime = timeGetTime();
   CurrentTime = InitialTime;
	TimeSinceLastInteraction = 0.0;
	HoldTimeCounter = 0.0;

   //Seed random number generator (for standalone behavior)
   srand( time( NULL ) );

   //Turn mouse off
   glutSetCursor( GLUT_CURSOR_NONE );

   //Tell OpenGL where to look when things happen
	glutDisplayFunc( Display );
	glutReshapeFunc( Reshape );
	glutMouseFunc( Mouse );
	glutMotionFunc( MouseMotion );
   glutKeyboardFunc( Keyboard );
   glutIdleFunc( MoveHopper );

	//let opengl take over
	glutMainLoop();

   //exit (even tho it never gets here)
   exit( 0 );
}




