// Compile on cygwin:
// /usr/bin/g++ Shader.cpp `sdl-config --cflags --libs` -I/usr/include -lglew32 -lopengl32 -lglu32

#include <stdlib.h>
#include <stdio.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <GL/glew.h>
#include <OpenGL/gl.h>
#include <OpenGL/glu.h>
#include <SDL.h>



class World {
private:
    float theta;
    GLuint textureId;


public:
    World() {
	theta = 0;

        // generate a texture ID for the dog
	glGenTextures(1, &textureId);

        // load the image using SDL
	SDL_Surface *texture = SDL_LoadBMP("texture.bmp");

        // bind the dog's texture ID
	glBindTexture(GL_TEXTURE_2D, textureId);

        // read the image data into openGL, building mipmaps
	gluBuild2DMipmaps(GL_TEXTURE_2D, GL_RGB, 
			  texture->w, texture->h,
			  GL_BGR, GL_UNSIGNED_BYTE, 
			  texture->pixels);

        // specify that the magnification should be bilinear
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, 
			GL_LINEAR);

        // specify that the minification filter should use mipmaps (trilinear)
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, 
			GL_LINEAR_MIPMAP_LINEAR);

	SDL_FreeSurface(texture);
    }

    void draw() {
        
        
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glLoadIdentity();
        glRotatef(theta, 0, 1, 1);

        glEnable(GL_TEXTURE_2D);
	glBindTexture(GL_TEXTURE_2D, textureId);

	glBegin(GL_QUADS);
	glColor3f(1, 1, 1);
	glNormal3f( 0,  0,  1);

	glTexCoord2f(0, 1);
	glVertex3f(-1, -1,  1);

        glTexCoord2f(1, 1);
	glVertex3f( 1, -1,  1);

	glTexCoord2f(1, 0);
	glVertex3f( 1,  1,  1);

	glTexCoord2f(0, 0);
	glVertex3f(-1,  1,  1);
        glEnd();

        glDisable(GL_TEXTURE_2D);
        glBegin(GL_QUADS);
	glColor3f(0.4, 0.4, 1);
	glNormal3f( 0,  0, -1);
	glVertex3f(-1, -1, -1);
	glVertex3f(-1,  1, -1);
	glVertex3f( 1,  1, -1);
	glVertex3f( 1, -1, -1);

	glColor3f(0.4, 1.0, 0.4);
	glNormal3f( 1,  0,  0);
	glVertex3f( 1, -1, -1);
	glVertex3f( 1,  1, -1);
	glVertex3f( 1,  1,  1);
	glVertex3f( 1, -1,  1);

	glNormal3f(-1,  0,  0);
	glVertex3f(-1, -1, -1);
	glVertex3f(-1,  1, -1);
	glVertex3f(-1,  1,  1);
	glVertex3f(-1, -1,  1);

	glColor3f(1.0, 0.4, 0.4);
	glNormal3f( 0,  1,  0);
	glVertex3f(-1,  1, -1);
	glVertex3f(-1,  1,  1);
	glVertex3f( 1,  1,  1);
	glVertex3f( 1,  1, -1);

	glNormal3f( 0, -1,  0);
	glVertex3f(-1, -1, -1);
	glVertex3f(-1, -1,  1);
	glVertex3f( 1, -1,  1);
	glVertex3f( 1, -1, -1);
	glEnd();
	
    }

    void update() {
	theta ++;
    }

    void handleEvents() {
	SDL_Event event;
	while (SDL_PollEvent(&event)) {
	    switch (event.type) {

	    case SDL_QUIT:
		SDL_Quit();
		exit(0);
		break;

	    case SDL_KEYDOWN:
		switch (event.key.keysym.sym) {
		case SDLK_ESCAPE:
		    SDL_Quit();
		    exit(0);
		    break;
		default:
		    theta += 10;
		    break;
		}
		
		break;

	    default:
		break;
	    }
	}

    }
};


char *readFile(const char *filename) {
    struct stat s;
    stat(filename, &s);
    char *buffer = new char[s.st_size+1];
    buffer[s.st_size] = '\0';
    int fd = open(filename, O_RDONLY, 0);
    read(fd, buffer, s.st_size);    
    close(fd);
    return buffer;
}

void checkCompilationLog(GLuint shaderId) {
    char log[4096];
    int len;
    glGetInfoLogARB(shaderId, 4096, &len, log);
    printf("Shader compilation log:\n");
    printf(log);
    printf("\n");
    fflush(stdout);

    if (strlen(log)) exit(1);
}


int main(int argc, char **argv) {
    SDL_Init(SDL_INIT_VIDEO);
    
    int flags = SDL_OPENGL | SDL_GL_DOUBLEBUFFER; // | SDL_FULLSCREEN;

    SDL_SetVideoMode(800, 600, 32, flags);

    glViewport(0, 0, 800, 600);

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective(45, 4.0/3, 0.1, 100);
    gluLookAt(0, 0, 5,
	      0, 0, 0,
	      0, 1, 0);

    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();

    glEnable(GL_DEPTH_TEST);
    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);
    glEnable(GL_COLOR_MATERIAL);

    GLfloat position[4] = {-1.5, 1.5, 2, 1};
    GLfloat color[4] = {1, 1, 1, 1};
    glLightfv(GL_LIGHT0, GL_POSITION, position);
    glLightfv(GL_LIGHT0, GL_DIFFUSE, color);

    // get the extension functions
    glewInit();
    glewGetExtension("GL_ARB_fragment_shader");
    glewGetExtension("GL_ARB_vertex_shader");
    glewGetExtension("GL_ARB_shader_objects");
    glewGetExtension("GL_ARB_shading_language_100");

    // make the vertex shader
    char *vertsource = readFile(argv[1]);
    GLuint vertshader = glCreateShaderObjectARB(GL_VERTEX_SHADER_ARB);
    glShaderSourceARB(vertshader, 1, 
		      (const GLcharARB **)(&vertsource), NULL);
    free(vertsource);
    glCompileShaderARB(vertshader);
    checkCompilationLog(vertshader);

    // make the fragment shader
    char *fragsource = readFile(argv[2]);
    GLuint fragshader = glCreateShaderObjectARB(GL_FRAGMENT_SHADER_ARB);
    glShaderSourceARB(fragshader, 1, 
		      (const GLcharARB **)&fragsource, NULL);
    free(fragsource);
    glCompileShaderARB(fragshader);
    checkCompilationLog(fragshader);

    // link them into a program
    GLuint program = glCreateProgramObjectARB();
    glAttachObjectARB(program, vertshader);   
    glAttachObjectARB(program, fragshader);
    glLinkProgramARB(program);
    checkCompilationLog(program);

    // activate the program
    glUseProgramObjectARB(program);


    World world;

    while (1) {
	world.handleEvents();
	world.update();
	world.draw();

	SDL_GL_SwapBuffers();	
	SDL_Delay(30);
    }

}
